extends Control
class_name RichTextLabel

## A control for displaying text that can contain different font styles, images, and basic formatting.
##
## A control for displaying text that can contain custom fonts, images, and basic formatting. [RichTextLabel] manages these as an internal tag stack. It also adapts itself to given width/heights.
## [b]Note:[/b] [method newline], [method push_paragraph], [code]"\n"[/code], [code]"\r\n"[/code], [code]p[/code] tag, and alignment tags start a new paragraph. Each paragraph is processed independently, in its own BiDi context. If you want to force line wrapping within paragraph, any other line breaking character can be used, for example, Form Feed (U+000C), Next Line (U+0085), Line Separator (U+2028).
## [b]Note:[/b] Assignments to [member text] clear the tag stack and reconstruct it from the property's contents. Any edits made to [member text] will erase previous edits made from other manual sources such as [method append_text] and the [code]push_*[/code] / [method pop] methods.
## [b]Note:[/b] RichTextLabel doesn't support entangled BBCode tags. For example, instead of using [code skip-lint][b]bold[i]bold italic[/b]italic[/i][/code], use [code skip-lint][b]bold[i]bold italic[/i][/b][i]italic[/i][/code].
## [b]Note:[/b] [code]push_*/pop_*[/code] functions won't affect BBCode.
## [b]Note:[/b] Unlike [Label], [RichTextLabel] doesn't have a [i]property[/i] to horizontally align text to the center. Instead, enable [member bbcode_enabled] and surround the text in a [code skip-lint][center][/code] tag as follows: [code skip-lint][center]Example[/center][/code]. There is currently no built-in way to vertically align text either, but this can be emulated by relying on anchors/containers and the [member fit_content] property.
##
## @tutorial(BBCode in RichTextLabel): https://docs.godotengine.org/en/stable/tutorials/ui/bbcode_in_richtextlabel.html
## @tutorial(Rich Text Label with BBCode Demo): https://godotengine.org/asset-library/asset/2774
## @tutorial(Operating System Testing Demo): https://godotengine.org/asset-library/asset/2789


## Triggered when the document is fully loaded.
## [b]Note:[/b] This can happen before the text is processed for drawing. Scrolling values may not be valid until the document is drawn for the first time after this signal.
signal finished
## Triggered when the user clicks on content between meta (URL) tags. If the meta is defined in BBCode, e.g. [code skip-lint][url={"key": "value"}]Text[/url][/code], then the parameter for this signal will always be a [String] type. If a particular type or an object is desired, the [method push_meta] method must be used to manually insert the data into the tag stack. Alternatively, you can convert the [String] input to the desired type based on its contents (such as calling [method JSON.parse] on it).
## For example, the following method can be connected to [signal meta_clicked] to open clicked URLs using the user's default web browser:
## [codeblocks]
## [gdscript]
## # This assumes RichTextLabel's `meta_clicked` signal was connected to
## # the function below using the signal connection dialog.
## func _richtextlabel_on_meta_clicked(meta):
## # `meta` is of Variant type, so convert it to a String to avoid script errors at run-time.
## OS.shell_open(str(meta))
## [/gdscript]
## [/codeblocks]
signal meta_clicked(meta: Variant)
## Triggers when the mouse exits a meta tag.
signal meta_hover_ended(meta: Variant)
## Triggers when the mouse enters a meta tag.
signal meta_hover_started(meta: Variant)
## Each list item has a number marker.
## Each list item has a letter marker.
## Each list item has a roman number marker.
## Each list item has a filled circle marker.
## Copies the selected text.
## Selects the whole [RichTextLabel] text.
## Represents the size of the [enum MenuItems] enum.
## Meta tag does not display an underline, even if [member meta_underlined] is [code]true[/code].
## If [member meta_underlined] is [code]true[/code], meta tag always display an underline.
## If [member meta_underlined] is [code]true[/code], meta tag display an underline when the mouse cursor is over it.
## If this bit is set, [method update_image] changes image texture.
## If this bit is set, [method update_image] changes image size.
## If this bit is set, [method update_image] changes image color.
## If this bit is set, [method update_image] changes image inline alignment.
## If this bit is set, [method update_image] changes image texture region.
## If this bit is set, [method update_image] changes image padding.
## If this bit is set, [method update_image] changes image tooltip.
## If this bit is set, [method update_image] changes image width from/to percents.

#enum ListType
enum {
    LIST_NUMBERS = 0,
    LIST_LETTERS = 1,
    LIST_ROMAN = 2,
    LIST_DOTS = 3,
}
#enum MenuItems
enum {
    MENU_COPY = 0,
    MENU_SELECT_ALL = 1,
    MENU_MAX = 2,
}
#enum MetaUnderline
enum {
    META_UNDERLINE_NEVER = 0,
    META_UNDERLINE_ALWAYS = 1,
    META_UNDERLINE_ON_HOVER = 2,
}
#enum ImageUpdateMask
enum {
    UPDATE_TEXTURE = 1,
    UPDATE_SIZE = 2,
    UPDATE_COLOR = 4,
    UPDATE_ALIGNMENT = 8,
    UPDATE_REGION = 16,
    UPDATE_PAD = 32,
    UPDATE_TOOLTIP = 64,
    UPDATE_WIDTH_IN_PERCENT = 128,
}
## If set to something other than [constant TextServer.AUTOWRAP_OFF], the text gets wrapped inside the node's bounding rectangle. To see how each mode behaves, see [enum TextServer.AutowrapMode].
var autowrap_mode: int:
	get = get_autowrap_mode, set = set_autowrap_mode

## If [code]true[/code], the label uses BBCode formatting.
## [b]Note:[/b] This only affects the contents of [member text], not the tag stack.
var bbcode_enabled: bool:
	get = is_using_bbcode, set = set_use_bbcode

var clip_contents: bool:
	get = is_clipping_contents, set = set_clip_contents

## If [code]true[/code], a right-click displays the context menu.
var context_menu_enabled: bool:
	get = is_context_menu_enabled, set = set_context_menu_enabled

## The currently installed custom effects. This is an array of [RichTextEffect]s.
## To add a custom effect, it's more convenient to use [method install_effect].
var custom_effects: Array:
	get = get_effects, set = set_effects

## If [code]true[/code], the selected text will be deselected when focus is lost.
var deselect_on_focus_loss_enabled: bool:
	get = is_deselect_on_focus_loss_enabled, set = set_deselect_on_focus_loss_enabled

## If [code]true[/code], allow drag and drop of selected text.
var drag_and_drop_selection_enabled: bool:
	get = is_drag_and_drop_selection_enabled, set = set_drag_and_drop_selection_enabled

## If [code]true[/code], the label's minimum size will be automatically updated to fit its content, matching the behavior of [Label].
var fit_content: bool:
	get = is_fit_content_enabled, set = set_fit_content

## If [code]true[/code], the label underlines hint tags such as [code skip-lint][hint=description]{text}[/hint][/code].
var hint_underlined: bool:
	get = is_hint_underlined, set = set_hint_underline

## Controls the text's horizontal alignment. Supports left, center, right, and fill, or justify. Set it to one of the [enum HorizontalAlignment] constants.
var horizontal_alignment: int:
	get = get_horizontal_alignment, set = set_horizontal_alignment

## Line fill alignment rules. See [enum TextServer.JustificationFlag] for more information.
var justification_flags: int:
	get = get_justification_flags, set = set_justification_flags

## Language code used for line-breaking and text shaping algorithms, if left empty current locale is used instead.
var language: String:
	get = get_language, set = set_language

## If [code]true[/code], the label underlines meta tags such as [code skip-lint][url]{text}[/url][/code]. These tags can call a function when clicked if [signal meta_clicked] is connected to a function.
var meta_underlined: bool:
	get = is_meta_underlined, set = set_meta_underline

## The delay after which the loading progress bar is displayed, in milliseconds. Set to [code]-1[/code] to disable progress bar entirely.
## [b]Note:[/b] Progress bar is displayed only if [member threaded] is enabled.
var progress_bar_delay: int:
	get = get_progress_bar_delay, set = set_progress_bar_delay

## If [code]true[/code], the scrollbar is visible. Setting this to [code]false[/code] does not block scrolling completely. See [method scroll_to_line].
var scroll_active: bool:
	get = is_scroll_active, set = set_scroll_active

## If [code]true[/code], the window scrolls down to display new content automatically.
var scroll_following: bool:
	get = is_scroll_following, set = set_scroll_follow

## If [code]true[/code], the label allows text selection.
var selection_enabled: bool:
	get = is_selection_enabled, set = set_selection_enabled

## If [code]true[/code], shortcut keys for context menu items are enabled, even if the context menu is disabled.
var shortcut_keys_enabled: bool:
	get = is_shortcut_keys_enabled, set = set_shortcut_keys_enabled

## Set BiDi algorithm override for the structured text.
var structured_text_bidi_override: int:
	get = get_structured_text_bidi_override, set = set_structured_text_bidi_override

## Set additional options for BiDi override.
var structured_text_bidi_override_options: Array:
	get = get_structured_text_bidi_override_options, set = set_structured_text_bidi_override_options

## The number of spaces associated with a single tab length. Does not affect [code]\t[/code] in text tags, only indent tags.
var tab_size: int:
	get = get_tab_size, set = set_tab_size

## Aligns text to the given tab-stops.
var tab_stops: PackedFloat32Array:
	get = get_tab_stops, set = set_tab_stops

## The label's text in BBCode format. Is not representative of manual modifications to the internal tag stack. Erases changes made by other methods when edited.
## [b]Note:[/b] If [member bbcode_enabled] is [code]true[/code], it is unadvised to use the [code]+=[/code] operator with [member text] (e.g. [code]text += "some string"[/code]) as it replaces the whole text and can cause slowdowns. It will also erase all BBCode that was added to stack using [code]push_*[/code] methods. Use [method append_text] for adding text instead, unless you absolutely need to close a tag that was opened in an earlier method call.
var text: String:
	get = get_text, set = set_text

## Base text writing direction.
var text_direction: int:
	get = get_text_direction, set = set_text_direction

## If [code]true[/code], text processing is done in a background thread.
var threaded: bool:
	get = is_threaded, set = set_threaded

## Controls the text's vertical alignment. Supports top, center, bottom, and fill. Set it to one of the [enum VerticalAlignment] constants.
var vertical_alignment: int:
	get = get_vertical_alignment, set = set_vertical_alignment

## The number of characters to display. If set to [code]-1[/code], all characters are displayed. This can be useful when animating the text appearing in a dialog box.
## [b]Note:[/b] Setting this property updates [member visible_ratio] accordingly.
var visible_characters: int:
	get = get_visible_characters, set = set_visible_characters

## Sets the clipping behavior when [member visible_characters] or [member visible_ratio] is set. See [enum TextServer.VisibleCharactersBehavior] for more info.
var visible_characters_behavior: int:
	get = get_visible_characters_behavior, set = set_visible_characters_behavior

## The fraction of characters to display, relative to the total number of characters (see [method get_total_character_count]). If set to [code]1.0[/code], all characters are displayed. If set to [code]0.5[/code], only half of the characters will be displayed. This can be useful when animating the text appearing in a dialog box.
## [b]Note:[/b] Setting this property updates [member visible_characters] accordingly.
var visible_ratio: float:
	get = get_visible_ratio, set = set_visible_ratio



## Adds an image's opening and closing tags to the tag stack, optionally providing a [param width] and [param height] to resize the image, a [param color] to tint the image and a [param region] to only use parts of the image.
## If [param width] or [param height] is set to 0, the image size will be adjusted in order to keep the original aspect ratio.
## If [param width] and [param height] are not set, but [param region] is, the region's rect will be used.
## [param key] is an optional identifier, that can be used to modify the image via [method update_image].
## If [param pad] is set, and the image is smaller than the size specified by [param width] and [param height], the image padding is added to match the size instead of upscaling.
## If [param size_in_percent] is set, [param width] and [param height] values are percentages of the control width instead of pixels.
func add_image(image: Texture2D, width: int = 0, height: int = 0, color: Color = Color(1, 1, 1, 1), inline_align: int = 5, region: Rect2 = Rect2(0, 0, 0, 0), key: Variant = null, pad: bool = false, tooltip: String = "", size_in_percent: bool = false) -> void:
	pass;

## Adds raw non-BBCode-parsed text to the tag stack.
func add_text(text: String) -> void:
	pass;

## Parses [param bbcode] and adds tags to the tag stack as needed.
## [b]Note:[/b] Using this method, you can't close a tag that was opened in a previous [method append_text] call. This is done to improve performance, especially when updating large RichTextLabels since rebuilding the whole BBCode every time would be slower. If you absolutely need to close a tag in a future method call, append the [member text] instead of using [method append_text].
func append_text(bbcode: String) -> void:
	pass;

## Clears the tag stack, causing the label to display nothing.
## [b]Note:[/b] This method does not affect [member text], and its contents will show again if the label is redrawn. However, setting [member text] to an empty [String] also clears the stack.
func clear() -> void:
	pass;

## Clears the current selection.
func deselect() -> void:
	pass;

## Returns the line number of the character position provided. Line and character numbers are both zero-indexed.
## [b]Note:[/b] If [member threaded] is enabled, this method returns a value for the loaded part of the document. Use [method is_finished] or [signal finished] to determine whether document is fully loaded.
func get_character_line(character: int) -> int:
	pass;

## Returns the paragraph number of the character position provided. Paragraph and character numbers are both zero-indexed.
## [b]Note:[/b] If [member threaded] is enabled, this method returns a value for the loaded part of the document. Use [method is_finished] or [signal finished] to determine whether document is fully loaded.
func get_character_paragraph(character: int) -> int:
	pass;

## Returns the height of the content.
## [b]Note:[/b] If [member threaded] is enabled, this method returns a value for the loaded part of the document. Use [method is_finished] or [signal finished] to determine whether document is fully loaded.
func get_content_height() -> int:
	pass;

## Returns the width of the content.
## [b]Note:[/b] If [member threaded] is enabled, this method returns a value for the loaded part of the document. Use [method is_finished] or [signal finished] to determine whether document is fully loaded.
func get_content_width() -> int:
	pass;

## Returns the total number of lines in the text. Wrapped text is counted as multiple lines.
## [b]Note:[/b] If [member visible_characters_behavior] is set to [constant TextServer.VC_CHARS_BEFORE_SHAPING] only visible wrapped lines are counted.
## [b]Note:[/b] If [member threaded] is enabled, this method returns a value for the loaded part of the document. Use [method is_finished] or [signal finished] to determine whether document is fully loaded.
func get_line_count() -> int:
	pass;

## Returns the vertical offset of the line found at the provided index.
## [b]Note:[/b] If [member threaded] is enabled, this method returns a value for the loaded part of the document. Use [method is_finished] or [signal finished] to determine whether document is fully loaded.
func get_line_offset(line: int) -> float:
	pass;

## Returns the indexes of the first and last visible characters for the given [param line], as a [Vector2i].
## [b]Note:[/b] If [member visible_characters_behavior] is set to [constant TextServer.VC_CHARS_BEFORE_SHAPING] only visible wrapped lines are counted.
## [b]Note:[/b] If [member threaded] is enabled, this method returns a value for the loaded part of the document. Use [method is_finished] or [signal finished] to determine whether document is fully loaded.
func get_line_range(line: int) -> Vector2i:
	pass;

## Returns the [PopupMenu] of this [RichTextLabel]. By default, this menu is displayed when right-clicking on the [RichTextLabel].
## You can add custom menu items or remove standard ones. Make sure your IDs don't conflict with the standard ones (see [enum MenuItems]). For example:
## [codeblocks]
## [gdscript]
## func _ready():
## var menu = get_menu()
## # Remove "Select All" item.
## menu.remove_item(MENU_SELECT_ALL)
## # Add custom items.
## menu.add_separator()
## menu.add_item("Duplicate Text", MENU_MAX + 1)
## # Connect callback.
## menu.id_pressed.connect(_on_item_pressed)
## func _on_item_pressed(id):
## if id == MENU_MAX + 1:
## add_text("\n" + get_parsed_text())
## [/gdscript]
## [csharp]
## public override void _Ready()
## {
## var menu = GetMenu();
## // Remove "Select All" item.
## menu.RemoveItem(RichTextLabel.MenuItems.SelectAll);
## // Add custom items.
## menu.AddSeparator();
## menu.AddItem("Duplicate Text", RichTextLabel.MenuItems.Max + 1);
## // Add event handler.
## menu.IdPressed += OnItemPressed;
## }
## public void OnItemPressed(int id)
## {
## if (id == TextEdit.MenuItems.Max + 1)
## {
## AddText("\n" + GetParsedText());
## }
## }
## [/csharp]
## [/codeblocks]
## [b]Warning:[/b] This is a required internal node, removing and freeing it may cause a crash. If you wish to hide it or any of its children, use their [member Window.visible] property.
func get_menu() -> PopupMenu:
	pass;

## Returns the total number of paragraphs (newlines or [code]p[/code] tags in the tag stack's text tags). Considers wrapped text as one paragraph.
func get_paragraph_count() -> int:
	pass;

## Returns the vertical offset of the paragraph found at the provided index.
## [b]Note:[/b] If [member threaded] is enabled, this method returns a value for the loaded part of the document. Use [method is_finished] or [signal finished] to determine whether document is fully loaded.
func get_paragraph_offset(paragraph: int) -> float:
	pass;

## Returns the text without BBCode mark-up.
func get_parsed_text() -> String:
	pass;

## Returns the current selection text. Does not include BBCodes.
func get_selected_text() -> String:
	pass;

## Returns the current selection first character index if a selection is active, [code]-1[/code] otherwise. Does not include BBCodes.
func get_selection_from() -> int:
	pass;

## Returns the current selection vertical line offset if a selection is active, [code]-1.0[/code] otherwise.
func get_selection_line_offset() -> float:
	pass;

## Returns the current selection last character index if a selection is active, [code]-1[/code] otherwise. Does not include BBCodes.
func get_selection_to() -> int:
	pass;

## Returns the total number of characters from text tags. Does not include BBCodes.
func get_total_character_count() -> int:
	pass;

## Returns the vertical scrollbar.
## [b]Warning:[/b] This is a required internal node, removing and freeing it may cause a crash. If you wish to hide it or any of its children, use their [member CanvasItem.visible] property.
func get_v_scroll_bar() -> VScrollBar:
	pass;

## Returns the number of visible lines.
## [b]Note:[/b] If [member threaded] is enabled, this method returns a value for the loaded part of the document. Use [method is_finished] or [signal finished] to determine whether document is fully loaded.
func get_visible_line_count() -> int:
	pass;

## Returns the number of visible paragraphs. A paragraph is considered visible if at least one of its lines is visible.
## [b]Note:[/b] If [member threaded] is enabled, this method returns a value for the loaded part of the document. Use [method is_finished] or [signal finished] to determine whether document is fully loaded.
func get_visible_paragraph_count() -> int:
	pass;

## Installs a custom effect. This can also be done in the Inspector through the [member custom_effects] property. [param effect] should be a valid [RichTextEffect].
## [b]Example:[/b] With the following script extending from [RichTextEffect]:
## [codeblock]
## # effect.gd
## class_name MyCustomEffect
## extends RichTextEffect
## var bbcode = "my_custom_effect"
## # ...
## [/codeblock]
## The above effect can be installed in [RichTextLabel] from a script:
## [codeblock]
## # rich_text_label.gd
## extends RichTextLabel
## func _ready():
## install_effect(MyCustomEffect.new())
## # Alternatively, if not using `class_name` in the script that extends RichTextEffect:
## install_effect(preload("res://effect.gd").new())
## [/codeblock]
func install_effect(effect: Variant) -> void:
	pass;

## Invalidates [param paragraph] and all subsequent paragraphs cache.
func invalidate_paragraph(paragraph: int) -> bool:
	pass;

## If [member threaded] is enabled, returns [code]true[/code] if the background thread has finished text processing, otherwise always return [code]true[/code].
func is_finished() -> bool:
	pass;

## Returns whether the menu is visible. Use this instead of [code]get_menu().visible[/code] to improve performance (so the creation of the menu is avoided).
func is_menu_visible() -> bool:
	pass;

## If [member threaded] is enabled, returns [code]true[/code] if the background thread has finished text processing, otherwise always return [code]true[/code].
func is_ready() -> bool:
	pass;

## Executes a given action as defined in the [enum MenuItems] enum.
func menu_option(option: int) -> void:
	pass;

## Adds a newline tag to the tag stack.
func newline() -> void:
	pass;

## The assignment version of [method append_text]. Clears the tag stack and inserts the new content.
func parse_bbcode(bbcode: String) -> void:
	pass;

## Parses BBCode parameter [param expressions] into a dictionary.
func parse_expressions_for_values(expressions: PackedStringArray) -> Dictionary:
	pass;

## Terminates the current tag. Use after [code]push_*[/code] methods to close BBCodes manually. Does not need to follow [code]add_*[/code] methods.
func pop() -> void:
	pass;

## Terminates all tags opened by [code]push_*[/code] methods.
func pop_all() -> void:
	pass;

## Terminates tags opened after the last [method push_context] call (including context marker), or all tags if there's no context marker on the stack.
func pop_context() -> void:
	pass;

## Adds a [code skip-lint][bgcolor][/code] tag to the tag stack.
func push_bgcolor(bgcolor: Color) -> void:
	pass;

## Adds a [code skip-lint][font][/code] tag with a bold font to the tag stack. This is the same as adding a [code skip-lint][b][/code] tag if not currently in a [code skip-lint][i][/code] tag.
func push_bold() -> void:
	pass;

## Adds a [code skip-lint][font][/code] tag with a bold italics font to the tag stack.
func push_bold_italics() -> void:
	pass;

## Adds a [code skip-lint][cell][/code] tag to the tag stack. Must be inside a [code skip-lint][table][/code] tag. See [method push_table] for details. Use [method set_table_column_expand] to set column expansion ratio, [method set_cell_border_color] to set cell border, [method set_cell_row_background_color] to set cell background, [method set_cell_size_override] to override cell size, and [method set_cell_padding] to set padding.
func push_cell() -> void:
	pass;

## Adds a [code skip-lint][color][/code] tag to the tag stack.
func push_color(color: Color) -> void:
	pass;

## Adds a context marker to the tag stack. See [method pop_context].
func push_context() -> void:
	pass;

## Adds a custom effect tag to the tag stack. The effect does not need to be in [member custom_effects]. The environment is directly passed to the effect.
func push_customfx(effect: RichTextEffect, env: Dictionary) -> void:
	pass;

## Adds a [code skip-lint][dropcap][/code] tag to the tag stack. Drop cap (dropped capital) is a decorative element at the beginning of a paragraph that is larger than the rest of the text.
func push_dropcap(string: String, font: Font, size: int, dropcap_margins: Rect2 = Rect2(0, 0, 0, 0), color: Color = Color(1, 1, 1, 1), outline_size: int = 0, outline_color: Color = Color(0, 0, 0, 0)) -> void:
	pass;

## Adds a [code skip-lint][fgcolor][/code] tag to the tag stack.
func push_fgcolor(fgcolor: Color) -> void:
	pass;

## Adds a [code skip-lint][font][/code] tag to the tag stack. Overrides default fonts for its duration.
## Passing [code]0[/code] to [param font_size] will use the existing default font size.
func push_font(font: Font, font_size: int = 0) -> void:
	pass;

## Adds a [code skip-lint][font_size][/code] tag to the tag stack. Overrides default font size for its duration.
func push_font_size(font_size: int) -> void:
	pass;

## Adds a [code skip-lint][hint][/code] tag to the tag stack. Same as BBCode [code skip-lint][hint=something]{text}[/hint][/code].
func push_hint(description: String) -> void:
	pass;

## Adds an [code skip-lint][indent][/code] tag to the tag stack. Multiplies [param level] by current [member tab_size] to determine new margin length.
func push_indent(level: int) -> void:
	pass;

## Adds a [code skip-lint][font][/code] tag with an italics font to the tag stack. This is the same as adding an [code skip-lint][i][/code] tag if not currently in a [code skip-lint][b][/code] tag.
func push_italics() -> void:
	pass;

## Adds language code used for text shaping algorithm and Open-Type font features.
func push_language(language: String) -> void:
	pass;

## Adds [code skip-lint][ol][/code] or [code skip-lint][ul][/code] tag to the tag stack. Multiplies [param level] by current [member tab_size] to determine new margin length.
func push_list(level: int, type: int, capitalize: bool, bullet: String = "•") -> void:
	pass;

## Adds a meta tag to the tag stack. Similar to the BBCode [code skip-lint][url=something]{text}[/url][/code], but supports non-[String] metadata types.
## If [member meta_underlined] is [code]true[/code], meta tags display an underline. This behavior can be customized with [param underline_mode].
## [b]Note:[/b] Meta tags do nothing by default when clicked. To assign behavior when clicked, connect [signal meta_clicked] to a function that is called when the meta tag is clicked.
func push_meta(data: Variant, underline_mode: int = 1, tooltip: String = "") -> void:
	pass;

## Adds a [code skip-lint][font][/code] tag with a monospace font to the tag stack.
func push_mono() -> void:
	pass;

## Adds a [code skip-lint][font][/code] tag with a normal font to the tag stack.
func push_normal() -> void:
	pass;

## Adds a [code skip-lint][outline_color][/code] tag to the tag stack. Adds text outline for its duration.
func push_outline_color(color: Color) -> void:
	pass;

## Adds a [code skip-lint][outline_size][/code] tag to the tag stack. Overrides default text outline size for its duration.
func push_outline_size(outline_size: int) -> void:
	pass;

## Adds a [code skip-lint][p][/code] tag to the tag stack.
func push_paragraph(alignment: int, base_direction: int = 0, language: String = "", st_parser: int = 0, justification_flags: int = 163, tab_stops: PackedFloat32Array = PackedFloat32Array()) -> void:
	pass;

## Adds a [code skip-lint][s][/code] tag to the tag stack.
func push_strikethrough() -> void:
	pass;

## Adds a [code skip-lint][table=columns,inline_align][/code] tag to the tag stack. Use [method set_table_column_expand] to set column expansion ratio. Use [method push_cell] to add cells.
func push_table(columns: int, inline_align: int = 0, align_to_row: int = -1) -> void:
	pass;

## Adds a [code skip-lint][u][/code] tag to the tag stack.
func push_underline() -> void:
	pass;

## Removes a paragraph of content from the label. Returns [code]true[/code] if the paragraph exists.
## The [param paragraph] argument is the index of the paragraph to remove, it can take values in the interval [code][0, get_paragraph_count() - 1][/code].
## If [param no_invalidate] is set to [code]true[/code], cache for the subsequent paragraphs is not invalidated. Use it for faster updates if deleted paragraph is fully self-contained (have no unclosed tags), or this call is part of the complex edit operation and [method invalidate_paragraph] will be called at the end of operation.
func remove_paragraph(paragraph: int, no_invalidate: bool = false) -> bool:
	pass;

## Scrolls the window's top line to match [param line].
func scroll_to_line(line: int) -> void:
	pass;

## Scrolls the window's top line to match first line of the [param paragraph].
func scroll_to_paragraph(paragraph: int) -> void:
	pass;

## Scrolls to the beginning of the current selection.
func scroll_to_selection() -> void:
	pass;

## Select all the text.
## If [member selection_enabled] is [code]false[/code], no selection will occur.
func select_all() -> void:
	pass;

## Sets color of a table cell border.
func set_cell_border_color(color: Color) -> void:
	pass;

## Sets inner padding of a table cell.
func set_cell_padding(padding: Rect2) -> void:
	pass;

## Sets color of a table cell. Separate colors for alternating rows can be specified.
func set_cell_row_background_color(odd_row_bg: Color, even_row_bg: Color) -> void:
	pass;

## Sets minimum and maximum size overrides for a table cell.
func set_cell_size_override(min_size: Vector2, max_size: Vector2) -> void:
	pass;

## Edits the selected column's expansion options. If [param expand] is [code]true[/code], the column expands in proportion to its expansion ratio versus the other columns' ratios.
## For example, 2 columns with ratios 3 and 4 plus 70 pixels in available width would expand 30 and 40 pixels, respectively.
## If [param expand] is [code]false[/code], the column will not contribute to the total ratio.
func set_table_column_expand(column: int, expand: bool, ratio: int = 1, shrink: bool = true) -> void:
	pass;

## Updates the existing images with the key [param key]. Only properties specified by [param mask] bits are updated. See [method add_image].
func update_image(key: Variant, mask: int, image: Texture2D, width: int = 0, height: int = 0, color: Color = Color(1, 1, 1, 1), inline_align: int = 5, region: Rect2 = Rect2(0, 0, 0, 0), pad: bool = false, tooltip: String = "", size_in_percent: bool = false) -> void:
	pass;


func get_autowrap_mode() -> int:
	return autowrap_mode

func set_autowrap_mode(value: int) -> void:
	autowrap_mode = value

func is_using_bbcode() -> bool:
	return bbcode_enabled

func set_use_bbcode(value: bool) -> void:
	bbcode_enabled = value

func is_clipping_contents() -> bool:
	return clip_contents

func set_clip_contents(value: bool) -> void:
	clip_contents = value

func is_context_menu_enabled() -> bool:
	return context_menu_enabled

func set_context_menu_enabled(value: bool) -> void:
	context_menu_enabled = value

func get_effects() -> Array:
	return custom_effects

func set_effects(value: Array) -> void:
	custom_effects = value

func is_deselect_on_focus_loss_enabled() -> bool:
	return deselect_on_focus_loss_enabled

func set_deselect_on_focus_loss_enabled(value: bool) -> void:
	deselect_on_focus_loss_enabled = value

func is_drag_and_drop_selection_enabled() -> bool:
	return drag_and_drop_selection_enabled

func set_drag_and_drop_selection_enabled(value: bool) -> void:
	drag_and_drop_selection_enabled = value

func is_fit_content_enabled() -> bool:
	return fit_content

func set_fit_content(value: bool) -> void:
	fit_content = value

func is_hint_underlined() -> bool:
	return hint_underlined

func set_hint_underline(value: bool) -> void:
	hint_underlined = value

func get_horizontal_alignment() -> int:
	return horizontal_alignment

func set_horizontal_alignment(value: int) -> void:
	horizontal_alignment = value

func get_justification_flags() -> int:
	return justification_flags

func set_justification_flags(value: int) -> void:
	justification_flags = value

func get_language() -> String:
	return language

func set_language(value: String) -> void:
	language = value

func is_meta_underlined() -> bool:
	return meta_underlined

func set_meta_underline(value: bool) -> void:
	meta_underlined = value

func get_progress_bar_delay() -> int:
	return progress_bar_delay

func set_progress_bar_delay(value: int) -> void:
	progress_bar_delay = value

func is_scroll_active() -> bool:
	return scroll_active

func set_scroll_active(value: bool) -> void:
	scroll_active = value

func is_scroll_following() -> bool:
	return scroll_following

func set_scroll_follow(value: bool) -> void:
	scroll_following = value

func is_selection_enabled() -> bool:
	return selection_enabled

func set_selection_enabled(value: bool) -> void:
	selection_enabled = value

func is_shortcut_keys_enabled() -> bool:
	return shortcut_keys_enabled

func set_shortcut_keys_enabled(value: bool) -> void:
	shortcut_keys_enabled = value

func get_structured_text_bidi_override() -> int:
	return structured_text_bidi_override

func set_structured_text_bidi_override(value: int) -> void:
	structured_text_bidi_override = value

func get_structured_text_bidi_override_options() -> Array:
	return structured_text_bidi_override_options

func set_structured_text_bidi_override_options(value: Array) -> void:
	structured_text_bidi_override_options = value

func get_tab_size() -> int:
	return tab_size

func set_tab_size(value: int) -> void:
	tab_size = value

func get_tab_stops() -> PackedFloat32Array:
	return tab_stops

func set_tab_stops(value: PackedFloat32Array) -> void:
	tab_stops = value

func get_text() -> String:
	return text

func set_text(value: String) -> void:
	text = value

func get_text_direction() -> int:
	return text_direction

func set_text_direction(value: int) -> void:
	text_direction = value

func is_threaded() -> bool:
	return threaded

func set_threaded(value: bool) -> void:
	threaded = value

func get_vertical_alignment() -> int:
	return vertical_alignment

func set_vertical_alignment(value: int) -> void:
	vertical_alignment = value

func get_visible_characters() -> int:
	return visible_characters

func set_visible_characters(value: int) -> void:
	visible_characters = value

func get_visible_characters_behavior() -> int:
	return visible_characters_behavior

func set_visible_characters_behavior(value: int) -> void:
	visible_characters_behavior = value

func get_visible_ratio() -> float:
	return visible_ratio

func set_visible_ratio(value: float) -> void:
	visible_ratio = value

