extends Light3D
class_name OmniLight3D

## Omnidirectional light, such as a light bulb or a candle.
##
## An Omnidirectional light is a type of [Light3D] that emits light in all directions. The light is attenuated by distance and this attenuation can be configured by changing its energy, radius, and attenuation parameters.
## [b]Note:[/b] When using the Mobile rendering method, only 8 omni lights can be displayed on each mesh resource. Attempting to display more than 8 omni lights on a single mesh resource will result in omni lights flickering in and out as the camera moves. When using the Compatibility rendering method, only 8 omni lights can be displayed on each mesh resource by default, but this can be increased by adjusting [member ProjectSettings.rendering/limits/opengl/max_lights_per_object].
## [b]Note:[/b] When using the Mobile or Compatibility rendering methods, omni lights will only correctly affect meshes whose visibility AABB intersects with the light's AABB. If using a shader to deform the mesh in a way that makes it go outside its AABB, [member GeometryInstance3D.extra_cull_margin] must be increased on the mesh. Otherwise, the light may not be visible on the mesh.
##
## @tutorial(3D lights and shadows): https://docs.godotengine.org/en/stable/tutorials/3d/lights_and_shadows.html
## @tutorial(Faking global illumination): https://docs.godotengine.org/en/stable/tutorials/3d/global_illumination/faking_global_illumination.html


## Shadows are rendered to a dual-paraboloid texture. Faster than [constant SHADOW_CUBE], but lower-quality.
## Shadows are rendered to a cubemap. Slower than [constant SHADOW_DUAL_PARABOLOID], but higher-quality.

#enum ShadowMode
enum {
    SHADOW_DUAL_PARABOLOID = 0,
    SHADOW_CUBE = 1,
}
var light_specular: float:
	get = get_param, set = set_param

## Controls the distance attenuation function for omnilights.
## A value of [code]0.0[/code] will maintain a constant brightness through most of the range, but smoothly attenuate the light at the edge of the range. Use a value of [code]2.0[/code] for physically accurate lights as it results in the proper inverse square attenutation.
## [b]Note:[/b] Setting attenuation to [code]2.0[/code] or higher may result in distant objects receiving minimal light, even within range. For example, with a range of [code]4096[/code], an object at [code]100[/code] units is attenuated by a factor of [code]0.0001[/code]. With a default brightness of [code]1[/code], the light would not be visible at that distance.
## [b]Note:[/b] Using negative or values higher than [code]10.0[/code] may lead to unexpected results.
var omni_attenuation: float:
	get = get_param, set = set_param

## The light's radius. Note that the effectively lit area may appear to be smaller depending on the [member omni_attenuation] in use. No matter the [member omni_attenuation] in use, the light will never reach anything outside this radius.
## [b]Note:[/b] [member omni_range] is not affected by [member Node3D.scale] (the light's scale or its parent's scale).
var omni_range: float:
	get = get_param, set = set_param

## See [enum ShadowMode].
var omni_shadow_mode: int:
	get = get_shadow_mode, set = set_shadow_mode

var shadow_normal_bias: float:
	get = get_param, set = set_param




func get_param() -> float:
	return light_specular

func set_param(value: float) -> void:
	light_specular = value

func get_param() -> float:
	return omni_attenuation

func set_param(value: float) -> void:
	omni_attenuation = value

func get_param() -> float:
	return omni_range

func set_param(value: float) -> void:
	omni_range = value

func get_shadow_mode() -> int:
	return omni_shadow_mode

func set_shadow_mode(value: int) -> void:
	omni_shadow_mode = value

func get_param() -> float:
	return shadow_normal_bias

func set_param(value: float) -> void:
	shadow_normal_bias = value

