extends Texture2D
class_name NoiseTexture2D

## A 2D texture filled with noise generated by a [Noise] object.
##
## Uses the [FastNoiseLite] library or other noise generators to fill the texture data of your desired size. [NoiseTexture2D] can also generate normal map textures.
## The class uses [Thread]s to generate the texture data internally, so [method Texture2D.get_image] may return [code]null[/code] if the generation process has not completed yet. In that case, you need to wait for the texture to be generated before accessing the image and the generated byte data:
## [codeblock]
## var texture = NoiseTexture2D.new()
## texture.noise = FastNoiseLite.new()
## await texture.changed
## var image = texture.get_image()
## var data = image.get_data()
## [/codeblock]



## If [code]true[/code], the resulting texture contains a normal map created from the original noise interpreted as a bump map.
var as_normal_map: bool:
	get = is_normal_map, set = set_as_normal_map

## Strength of the bump maps used in this texture. A higher value will make the bump maps appear larger while a lower value will make them appear softer.
var bump_strength: float:
	get = get_bump_strength, set = set_bump_strength

## A [Gradient] which is used to map the luminance of each pixel to a color value.
var color_ramp: Gradient:
	get = get_color_ramp, set = set_color_ramp

## Determines whether mipmaps are generated for this texture. Enabling this results in less texture aliasing in the distance, at the cost of increasing memory usage by roughly 33% and making the noise texture generation take longer.
## [b]Note:[/b] [member generate_mipmaps] requires mipmap filtering to be enabled on the material using the [NoiseTexture2D] to have an effect.
var generate_mipmaps: bool:
	get = is_generating_mipmaps, set = set_generate_mipmaps

## Height of the generated texture (in pixels).
var height: int:
	get = get_height, set = set_height

## Determines whether the noise image is calculated in 3D space. May result in reduced contrast.
var in_3d_space: bool:
	get = is_in_3d_space, set = set_in_3d_space

## If [code]true[/code], inverts the noise texture. White becomes black, black becomes white.
var invert: bool:
	get = get_invert, set = set_invert

## The instance of the [Noise] object.
var noise: Noise:
	get = get_noise, set = set_noise

## If [code]true[/code], the noise image coming from the noise generator is normalized to the range [code]0.0[/code] to [code]1.0[/code].
## Turning normalization off can affect the contrast and allows you to generate non repeating tileable noise textures.
var normalize: bool:
	get = is_normalized, set = set_normalize

var resource_local_to_scene: bool:
	get = is_local_to_scene, set = set_local_to_scene

## If [code]true[/code], a seamless texture is requested from the [Noise] resource.
## [b]Note:[/b] Seamless noise textures may take longer to generate and/or can have a lower contrast compared to non-seamless noise depending on the used [Noise] resource. This is because some implementations use higher dimensions for generating seamless noise.
## [b]Note:[/b] The default [FastNoiseLite] implementation uses the fallback path for seamless generation. If using a [member width] or [member height] lower than the default, you may need to increase [member seamless_blend_skirt] to make seamless blending more effective.
var seamless: bool:
	get = get_seamless, set = set_seamless

## Used for the default/fallback implementation of the seamless texture generation. It determines the distance over which the seams are blended. High values may result in less details and contrast. See [Noise] for further details.
## [b]Note:[/b] If using a [member width] or [member height] lower than the default, you may need to increase [member seamless_blend_skirt] to make seamless blending more effective.
var seamless_blend_skirt: float:
	get = get_seamless_blend_skirt, set = set_seamless_blend_skirt

## Width of the generated texture (in pixels).
var width: int:
	get = get_width, set = set_width




func is_normal_map() -> bool:
	return as_normal_map

func set_as_normal_map(value: bool) -> void:
	as_normal_map = value

func get_bump_strength() -> float:
	return bump_strength

func set_bump_strength(value: float) -> void:
	bump_strength = value

func get_color_ramp() -> Gradient:
	return color_ramp

func set_color_ramp(value: Gradient) -> void:
	color_ramp = value

func is_generating_mipmaps() -> bool:
	return generate_mipmaps

func set_generate_mipmaps(value: bool) -> void:
	generate_mipmaps = value

func get_height() -> int:
	return height

func set_height(value: int) -> void:
	height = value

func is_in_3d_space() -> bool:
	return in_3d_space

func set_in_3d_space(value: bool) -> void:
	in_3d_space = value

func get_invert() -> bool:
	return invert

func set_invert(value: bool) -> void:
	invert = value

func get_noise() -> Noise:
	return noise

func set_noise(value: Noise) -> void:
	noise = value

func is_normalized() -> bool:
	return normalize

func set_normalize(value: bool) -> void:
	normalize = value

func is_local_to_scene() -> bool:
	return resource_local_to_scene

func set_local_to_scene(value: bool) -> void:
	resource_local_to_scene = value

func get_seamless() -> bool:
	return seamless

func set_seamless(value: bool) -> void:
	seamless = value

func get_seamless_blend_skirt() -> float:
	return seamless_blend_skirt

func set_seamless_blend_skirt(value: float) -> void:
	seamless_blend_skirt = value

func get_width() -> int:
	return width

func set_width(value: int) -> void:
	width = value

