extends Resource
class_name MultiMesh

## Provides high-performance drawing of a mesh multiple times using GPU instancing.
##
## MultiMesh provides low-level mesh instancing. Drawing thousands of [MeshInstance3D] nodes can be slow, since each object is submitted to the GPU then drawn individually.
## MultiMesh is much faster as it can draw thousands of instances with a single draw call, resulting in less API overhead.
## As a drawback, if the instances are too far away from each other, performance may be reduced as every single instance will always render (they are spatially indexed as one, for the whole object).
## Since instances may have any behavior, the AABB used for visibility must be provided by the user.
## [b]Note:[/b] A MultiMesh is a single object, therefore the same maximum lights per object restriction applies. This means, that once the maximum lights are consumed by one or more instances, the rest of the MultiMesh instances will [b]not[/b] receive any lighting.
## [b]Note:[/b] Blend Shapes will be ignored if used in a MultiMesh.
##
## @tutorial(Using MultiMeshInstance): https://docs.godotengine.org/en/stable/tutorials/3d/using_multi_mesh_instance.html
## @tutorial(Optimization using MultiMeshes): https://docs.godotengine.org/en/stable/tutorials/performance/using_multimesh.html
## @tutorial(Animating thousands of fish with MultiMeshInstance): https://docs.godotengine.org/en/stable/tutorials/performance/vertex_animation/animating_thousands_of_fish.html


## Use this when using 2D transforms.
## Use this when using 3D transforms.
## Always interpolate using Basis lerping, which can produce warping artifacts in some situations.
## Attempt to interpolate using Basis slerping (spherical linear interpolation) where possible, otherwise fall back to lerping.

#enum TransformFormat
enum {
    TRANSFORM_2D = 0,
    TRANSFORM_3D = 1,
}
#enum PhysicsInterpolationQuality
enum {
    INTERP_QUALITY_FAST = 0,
    INTERP_QUALITY_HIGH = 1,
}
var buffer: PackedFloat32Array:
	get = get_buffer, set = set_buffer

## Array containing each [Color] used by all instances of this mesh.
var color_array: PackedColorArray:
	get = _get_color_array, set = _set_color_array

## Custom AABB for this MultiMesh resource. Setting this manually prevents costly runtime AABB recalculations.
var custom_aabb: AABB:
	get = get_custom_aabb, set = set_custom_aabb

## Array containing each custom data value used by all instances of this mesh, as a [PackedColorArray].
var custom_data_array: PackedColorArray:
	get = _get_custom_data_array, set = _set_custom_data_array

## Number of instances that will get drawn. This clears and (re)sizes the buffers. Setting data format or flags afterwards will have no effect.
## By default, all instances are drawn but you can limit this with [member visible_instance_count].
var instance_count: int:
	get = get_instance_count, set = set_instance_count

## [Mesh] resource to be instanced.
## The looks of the individual instances can be modified using [method set_instance_color] and [method set_instance_custom_data].
var mesh: Mesh:
	get = get_mesh, set = set_mesh

## Choose whether to use an interpolation method that favors speed or quality.
## When using low physics tick rates (typically below 20) or high rates of object rotation, you may get better results from the high quality setting.
## [b]Note:[/b] Fast quality does not equate to low quality. Except in the special cases mentioned above, the quality should be comparable to high quality.
var physics_interpolation_quality: int:
	get = get_physics_interpolation_quality, set = set_physics_interpolation_quality

## Array containing each [Transform2D] value used by all instances of this mesh, as a [PackedVector2Array]. Each transform is divided into 3 [Vector2] values corresponding to the transforms' [code]x[/code], [code]y[/code], and [code]origin[/code].
var transform_2d_array: PackedVector2Array:
	get = _get_transform_2d_array, set = _set_transform_2d_array

## Array containing each [Transform3D] value used by all instances of this mesh, as a [PackedVector3Array]. Each transform is divided into 4 [Vector3] values corresponding to the transforms' [code]x[/code], [code]y[/code], [code]z[/code], and [code]origin[/code].
var transform_array: PackedVector3Array:
	get = _get_transform_array, set = _set_transform_array

## Format of transform used to transform mesh, either 2D or 3D.
var transform_format: int:
	get = get_transform_format, set = set_transform_format

## If [code]true[/code], the [MultiMesh] will use color data (see [method set_instance_color]). Can only be set when [member instance_count] is [code]0[/code] or less. This means that you need to call this method before setting the instance count, or temporarily reset it to [code]0[/code].
var use_colors: bool:
	get = is_using_colors, set = set_use_colors

## If [code]true[/code], the [MultiMesh] will use custom data (see [method set_instance_custom_data]). Can only be set when [member instance_count] is [code]0[/code] or less. This means that you need to call this method before setting the instance count, or temporarily reset it to [code]0[/code].
var use_custom_data: bool:
	get = is_using_custom_data, set = set_use_custom_data

## Limits the number of instances drawn, -1 draws all instances. Changing this does not change the sizes of the buffers.
var visible_instance_count: int:
	get = get_visible_instance_count, set = set_visible_instance_count



## Returns the visibility axis-aligned bounding box in local space.
func get_aabb() -> AABB:
	pass;

## Gets a specific instance's color multiplier.
func get_instance_color(instance: int) -> Color:
	pass;

## Returns the custom data that has been set for a specific instance.
func get_instance_custom_data(instance: int) -> Color:
	pass;

## Returns the [Transform3D] of a specific instance.
func get_instance_transform(instance: int) -> Transform3D:
	pass;

## Returns the [Transform2D] of a specific instance.
func get_instance_transform_2d(instance: int) -> Transform2D:
	pass;

## When using [i]physics interpolation[/i], this function allows you to prevent interpolation on an instance in the current physics tick.
## This allows you to move instances instantaneously, and should usually be used when initially placing an instance such as a bullet to prevent graphical glitches.
func reset_instance_physics_interpolation(instance: int) -> void:
	pass;

## An alternative to setting the [member buffer] property, which can be used with [i]physics interpolation[/i]. This method takes two arrays, and can set the data for the current and previous tick in one go. The renderer will automatically interpolate the data at each frame.
## This is useful for situations where the order of instances may change from physics tick to tick, such as particle systems.
## When the order of instances is coherent, the simpler alternative of setting [member buffer] can still be used with interpolation.
func set_buffer_interpolated(buffer_curr: PackedFloat32Array, buffer_prev: PackedFloat32Array) -> void:
	pass;

## Sets the color of a specific instance by [i]multiplying[/i] the mesh's existing vertex colors. This allows for different color tinting per instance.
## [b]Note:[/b] Each component is stored in 32 bits in the Forward+ and Mobile rendering methods, but is packed into 16 bits in the Compatibility rendering method.
## For the color to take effect, ensure that [member use_colors] is [code]true[/code] on the [MultiMesh] and [member BaseMaterial3D.vertex_color_use_as_albedo] is [code]true[/code] on the material. If you intend to set an absolute color instead of tinting, make sure the material's albedo color is set to pure white ([code]Color(1, 1, 1)[/code]).
func set_instance_color(instance: int, color: Color) -> void:
	pass;

## Sets custom data for a specific instance. [param custom_data] is a [Color] type only to contain 4 floating-point numbers.
## [b]Note:[/b] Each number is stored in 32 bits in the Forward+ and Mobile rendering methods, but is packed into 16 bits in the Compatibility rendering method.
## For the custom data to be used, ensure that [member use_custom_data] is [code]true[/code].
## This custom instance data has to be manually accessed in your custom shader using [code]INSTANCE_CUSTOM[/code].
func set_instance_custom_data(instance: int, custom_data: Color) -> void:
	pass;

## Sets the [Transform3D] for a specific instance.
func set_instance_transform(instance: int, transform: Transform3D) -> void:
	pass;

## Sets the [Transform2D] for a specific instance.
func set_instance_transform_2d(instance: int, transform: Transform2D) -> void:
	pass;


func get_buffer() -> PackedFloat32Array:
	return buffer

func set_buffer(value: PackedFloat32Array) -> void:
	buffer = value

func _get_color_array() -> PackedColorArray:
	return color_array

func _set_color_array(value: PackedColorArray) -> void:
	color_array = value

func get_custom_aabb() -> AABB:
	return custom_aabb

func set_custom_aabb(value: AABB) -> void:
	custom_aabb = value

func _get_custom_data_array() -> PackedColorArray:
	return custom_data_array

func _set_custom_data_array(value: PackedColorArray) -> void:
	custom_data_array = value

func get_instance_count() -> int:
	return instance_count

func set_instance_count(value: int) -> void:
	instance_count = value

func get_mesh() -> Mesh:
	return mesh

func set_mesh(value: Mesh) -> void:
	mesh = value

func get_physics_interpolation_quality() -> int:
	return physics_interpolation_quality

func set_physics_interpolation_quality(value: int) -> void:
	physics_interpolation_quality = value

func _get_transform_2d_array() -> PackedVector2Array:
	return transform_2d_array

func _set_transform_2d_array(value: PackedVector2Array) -> void:
	transform_2d_array = value

func _get_transform_array() -> PackedVector3Array:
	return transform_array

func _set_transform_array(value: PackedVector3Array) -> void:
	transform_array = value

func get_transform_format() -> int:
	return transform_format

func set_transform_format(value: int) -> void:
	transform_format = value

func is_using_colors() -> bool:
	return use_colors

func set_use_colors(value: bool) -> void:
	use_colors = value

func is_using_custom_data() -> bool:
	return use_custom_data

func set_use_custom_data(value: bool) -> void:
	use_custom_data = value

func get_visible_instance_count() -> int:
	return visible_instance_count

func set_visible_instance_count(value: int) -> void:
	visible_instance_count = value

