extends VisualInstance3D
class_name LightmapGI

## Computes and stores baked lightmaps for fast global illumination.
##
## The [LightmapGI] node is used to compute and store baked lightmaps. Lightmaps are used to provide high-quality indirect lighting with very little light leaking. [LightmapGI] can also provide rough reflections using spherical harmonics if [member directional] is enabled. Dynamic objects can receive indirect lighting thanks to [i]light probes[/i], which can be automatically placed by setting [member generate_probes_subdiv] to a value other than [constant GENERATE_PROBES_DISABLED]. Additional lightmap probes can also be added by creating [LightmapProbe] nodes. The downside is that lightmaps are fully static and cannot be baked in an exported project. Baking a [LightmapGI] node is also slower compared to [VoxelGI].
## [b]Procedural generation:[/b] Lightmap baking functionality is only available in the editor. This means [LightmapGI] is not suited to procedurally generated or user-built levels. For procedurally generated or user-built levels, use [VoxelGI] or SDFGI instead (see [member Environment.sdfgi_enabled]).
## [b]Performance:[/b] [LightmapGI] provides the best possible run-time performance for global illumination. It is suitable for low-end hardware including integrated graphics and mobile devices.
## [b]Note:[/b] Due to how lightmaps work, most properties only have a visible effect once lightmaps are baked again.
## [b]Note:[/b] Lightmap baking on [CSGShape3D]s and [PrimitiveMesh]es is not supported, as these cannot store UV2 data required for baking.
## [b]Note:[/b] If no custom lightmappers are installed, [LightmapGI] can only be baked from devices that support the Forward+ or Mobile renderers.
## [b]Note:[/b] The [LightmapGI] node only bakes light data for child nodes of its parent. Nodes further up the hierarchy of the scene will not be baked.
##
## @tutorial(Using Lightmap global illumination): https://docs.godotengine.org/en/stable/tutorials/3d/global_illumination/using_lightmap_gi.html


## Low bake quality (fastest bake times). The quality of this preset can be adjusted by changing [member ProjectSettings.rendering/lightmapping/bake_quality/low_quality_ray_count] and [member ProjectSettings.rendering/lightmapping/bake_quality/low_quality_probe_ray_count].
## Medium bake quality (fast bake times). The quality of this preset can be adjusted by changing [member ProjectSettings.rendering/lightmapping/bake_quality/medium_quality_ray_count] and [member ProjectSettings.rendering/lightmapping/bake_quality/medium_quality_probe_ray_count].
## High bake quality (slow bake times). The quality of this preset can be adjusted by changing [member ProjectSettings.rendering/lightmapping/bake_quality/high_quality_ray_count] and [member ProjectSettings.rendering/lightmapping/bake_quality/high_quality_probe_ray_count].
## Highest bake quality (slowest bake times). The quality of this preset can be adjusted by changing [member ProjectSettings.rendering/lightmapping/bake_quality/ultra_quality_ray_count] and [member ProjectSettings.rendering/lightmapping/bake_quality/ultra_quality_probe_ray_count].
## Don't generate lightmap probes for lighting dynamic objects.
## Lowest level of subdivision (fastest bake times, smallest file sizes).
## Low level of subdivision (fast bake times, small file sizes).
## High level of subdivision (slow bake times, large file sizes).
## Highest level of subdivision (slowest bake times, largest file sizes).
## Lightmap baking was successful.
## Lightmap baking failed because the root node for the edited scene could not be accessed.
## Lightmap baking failed as the lightmap data resource is embedded in a foreign resource.
## Lightmap baking failed as there is no lightmapper available in this Godot build.
## Lightmap baking failed as the [LightmapGIData] save path isn't configured in the resource.
## Lightmap baking failed as there are no meshes whose [member GeometryInstance3D.gi_mode] is [constant GeometryInstance3D.GI_MODE_STATIC] and with valid UV2 mapping in the current scene. You may need to select 3D scenes in the Import dock and change their global illumination mode accordingly.
## Lightmap baking failed as the lightmapper failed to analyze some of the meshes marked as static for baking.
## Lightmap baking failed as the resulting image couldn't be saved or imported by Godot after it was saved.
## The user aborted the lightmap baking operation (typically by clicking the [b]Cancel[/b] button in the progress dialog).
## Lightmap baking failed as the maximum texture size is too small to fit some of the meshes marked for baking.
## Lightmap baking failed as the lightmap is too small.
## Lightmap baking failed as the lightmap was unable to fit into an atlas.
## Ignore environment lighting when baking lightmaps.
## Use the scene's environment lighting when baking lightmaps.
## [b]Note:[/b] If baking lightmaps in a scene with no [WorldEnvironment] node, this will act like [constant ENVIRONMENT_MODE_DISABLED]. The editor's preview sky and sun is [i]not[/i] taken into account by [LightmapGI] when baking lightmaps.
## Use [member environment_custom_sky] as a source of environment lighting when baking lightmaps.
## Use [member environment_custom_color] multiplied by [member environment_custom_energy] as a constant source of environment lighting when baking lightmaps.

#enum BakeQuality
enum {
    BAKE_QUALITY_LOW = 0,
    BAKE_QUALITY_MEDIUM = 1,
    BAKE_QUALITY_HIGH = 2,
    BAKE_QUALITY_ULTRA = 3,
}
#enum GenerateProbes
enum {
    GENERATE_PROBES_DISABLED = 0,
    GENERATE_PROBES_SUBDIV_4 = 1,
    GENERATE_PROBES_SUBDIV_8 = 2,
    GENERATE_PROBES_SUBDIV_16 = 3,
    GENERATE_PROBES_SUBDIV_32 = 4,
}
#enum BakeError
enum {
    BAKE_ERROR_OK = 0,
    BAKE_ERROR_NO_SCENE_ROOT = 1,
    BAKE_ERROR_FOREIGN_DATA = 2,
    BAKE_ERROR_NO_LIGHTMAPPER = 3,
    BAKE_ERROR_NO_SAVE_PATH = 4,
    BAKE_ERROR_NO_MESHES = 5,
    BAKE_ERROR_MESHES_INVALID = 6,
    BAKE_ERROR_CANT_CREATE_IMAGE = 7,
    BAKE_ERROR_USER_ABORTED = 8,
    BAKE_ERROR_TEXTURE_SIZE_TOO_SMALL = 9,
    BAKE_ERROR_LIGHTMAP_TOO_SMALL = 10,
    BAKE_ERROR_ATLAS_TOO_SMALL = 11,
}
#enum EnvironmentMode
enum {
    ENVIRONMENT_MODE_DISABLED = 0,
    ENVIRONMENT_MODE_SCENE = 1,
    ENVIRONMENT_MODE_CUSTOM_SKY = 2,
    ENVIRONMENT_MODE_CUSTOM_COLOR = 3,
}
## The bias to use when computing shadows. Increasing [member bias] can fix shadow acne on the resulting baked lightmap, but can introduce peter-panning (shadows not connecting to their casters). Real-time [Light3D] shadows are not affected by this [member bias] property.
var bias: float:
	get = get_bias, set = set_bias

## The energy multiplier for each bounce. Higher values will make indirect lighting brighter. A value of [code]1.0[/code] represents physically accurate behavior, but higher values can be used to make indirect lighting propagate more visibly when using a low number of bounces. This can be used to speed up bake times by lowering the number of [member bounces] then increasing [member bounce_indirect_energy].
## [b]Note:[/b] [member bounce_indirect_energy] only has an effect if [member bounces] is set to a value greater than or equal to [code]1[/code].
var bounce_indirect_energy: float:
	get = get_bounce_indirect_energy, set = set_bounce_indirect_energy

## Number of light bounces that are taken into account during baking. Higher values result in brighter, more realistic lighting, at the cost of longer bake times. If set to [code]0[/code], only environment lighting, direct light and emissive lighting is baked.
var bounces: int:
	get = get_bounces, set = set_bounces

## The [CameraAttributes] resource that specifies exposure levels to bake at. Auto-exposure and non exposure properties will be ignored. Exposure settings should be used to reduce the dynamic range present when baking. If exposure is too high, the [LightmapGI] will have banding artifacts or may have over-exposure artifacts.
var camera_attributes: CameraAttributes:
	get = get_camera_attributes, set = set_camera_attributes

## The distance in pixels from which the denoiser samples. Lower values preserve more details, but may give blotchy results if the lightmap quality is not high enough. Only effective if [member use_denoiser] is [code]true[/code] and [member ProjectSettings.rendering/lightmapping/denoising/denoiser] is set to JNLM.
var denoiser_range: int:
	get = get_denoiser_range, set = set_denoiser_range

## The strength of denoising step applied to the generated lightmaps. Only effective if [member use_denoiser] is [code]true[/code] and [member ProjectSettings.rendering/lightmapping/denoising/denoiser] is set to JNLM.
var denoiser_strength: float:
	get = get_denoiser_strength, set = set_denoiser_strength

## If [code]true[/code], bakes lightmaps to contain directional information as spherical harmonics. This results in more realistic lighting appearance, especially with normal mapped materials and for lights that have their direct light baked ([member Light3D.light_bake_mode] set to [constant Light3D.BAKE_STATIC] and with [member Light3D.editor_only] set to [code]false[/code]). The directional information is also used to provide rough reflections for static and dynamic objects. This has a small run-time performance cost as the shader has to perform more work to interpret the direction information from the lightmap. Directional lightmaps also take longer to bake and result in larger file sizes.
## [b]Note:[/b] The property's name has no relationship with [DirectionalLight3D]. [member directional] works with all light types.
var directional: bool:
	get = is_directional, set = set_directional

## The color to use for environment lighting. Only effective if [member environment_mode] is [constant ENVIRONMENT_MODE_CUSTOM_COLOR].
var environment_custom_color: Color:
	get = get_environment_custom_color, set = set_environment_custom_color

## The color multiplier to use for environment lighting. Only effective if [member environment_mode] is [constant ENVIRONMENT_MODE_CUSTOM_COLOR].
var environment_custom_energy: float:
	get = get_environment_custom_energy, set = set_environment_custom_energy

## The sky to use as a source of environment lighting. Only effective if [member environment_mode] is [constant ENVIRONMENT_MODE_CUSTOM_SKY].
var environment_custom_sky: Sky:
	get = get_environment_custom_sky, set = set_environment_custom_sky

## The environment mode to use when baking lightmaps.
var environment_mode: int:
	get = get_environment_mode, set = set_environment_mode

## The level of subdivision to use when automatically generating [LightmapProbe]s for dynamic object lighting. Higher values result in more accurate indirect lighting on dynamic objects, at the cost of longer bake times and larger file sizes.
## [b]Note:[/b] Automatically generated [LightmapProbe]s are not visible as nodes in the Scene tree dock, and cannot be modified this way after they are generated.
## [b]Note:[/b] Regardless of [member generate_probes_subdiv], direct lighting on dynamic objects is always applied using [Light3D] nodes in real-time.
var generate_probes_subdiv: int:
	get = get_generate_probes, set = set_generate_probes

## If [code]true[/code], ignore environment lighting when baking lightmaps.
var interior: bool:
	get = is_interior, set = set_interior

## The [LightmapGIData] associated to this [LightmapGI] node. This resource is automatically created after baking, and is not meant to be created manually.
var light_data: LightmapGIData:
	get = get_light_data, set = set_light_data

## The maximum texture size for the generated texture atlas. Higher values will result in fewer slices being generated, but may not work on all hardware as a result of hardware limitations on texture sizes. Leave [member max_texture_size] at its default value of [code]16384[/code] if unsure.
var max_texture_size: int:
	get = get_max_texture_size, set = set_max_texture_size

## The quality preset to use when baking lightmaps. This affects bake times, but output file sizes remain mostly identical across quality levels.
## To further speed up bake times, decrease [member bounces], disable [member use_denoiser] and/or decrease [member texel_scale].
## To further increase quality, enable [member supersampling] and/or increase [member texel_scale].
var quality: int:
	get = get_bake_quality, set = set_bake_quality

## The shadowmasking policy to use for directional shadows on static objects that are baked with this [LightmapGI] instance.
## Shadowmasking allows [DirectionalLight3D] nodes to cast shadows even outside the range defined by their [member DirectionalLight3D.directional_shadow_max_distance] property. This is done by baking a texture that contains a shadowmap for the directional light, then using this texture according to the current shadowmask mode.
## [b]Note:[/b] The shadowmask texture is only created if [member shadowmask_mode] is not [constant LightmapGIData.SHADOWMASK_MODE_NONE]. To see a difference, you need to bake lightmaps again after switching from [constant LightmapGIData.SHADOWMASK_MODE_NONE] to any other mode.
var shadowmask_mode: int:
	get = get_shadowmask_mode, set = set_shadowmask_mode

## If [code]true[/code], lightmaps are baked with the texel scale multiplied with [member supersampling_factor] and downsampled before saving the lightmap (so the effective texel density is identical to having supersampling disabled).
## Supersampling provides increased lightmap quality with less noise, smoother shadows and better shadowing of small-scale features in objects. However, it may result in significantly increased bake times and memory usage while baking lightmaps. Padding is automatically adjusted to avoid increasing light leaking.
var supersampling: bool:
	get = is_supersampling_enabled, set = set_supersampling_enabled

## The factor by which the texel density is multiplied for supersampling. For best results, use an integer value. While fractional values are allowed, they can result in increased light leaking and a blurry lightmap.
## Higher values may result in better quality, but also increase bake times and memory usage while baking.
## See [member supersampling] for more information.
var supersampling_factor: float:
	get = get_supersampling_factor, set = set_supersampling_factor

## Scales the lightmap texel density of all meshes for the current bake. This is a multiplier that builds upon the existing lightmap texel size defined in each imported 3D scene, along with the per-mesh density multiplier (which is designed to be used when the same mesh is used at different scales). Lower values will result in faster bake times.
## For example, doubling [member texel_scale] doubles the lightmap texture resolution for all objects [i]on each axis[/i], so it will [i]quadruple[/i] the texel count.
var texel_scale: float:
	get = get_texel_scale, set = set_texel_scale

## If [code]true[/code], uses a GPU-based denoising algorithm on the generated lightmap. This eliminates most noise within the generated lightmap at the cost of longer bake times. File sizes are generally not impacted significantly by the use of a denoiser, although lossless compression may do a better job at compressing a denoised image.
var use_denoiser: bool:
	get = is_using_denoiser, set = set_use_denoiser

## If [code]true[/code], a texture with the lighting information will be generated to speed up the generation of indirect lighting at the cost of some accuracy. The geometry might exhibit extra light leak artifacts when using low resolution lightmaps or UVs that stretch the lightmap significantly across surfaces. Leave [member use_texture_for_bounces] at its default value of [code]true[/code] if unsure.
## [b]Note:[/b] [member use_texture_for_bounces] only has an effect if [member bounces] is set to a value greater than or equal to [code]1[/code].
var use_texture_for_bounces: bool:
	get = is_using_texture_for_bounces, set = set_use_texture_for_bounces




func get_bias() -> float:
	return bias

func set_bias(value: float) -> void:
	bias = value

func get_bounce_indirect_energy() -> float:
	return bounce_indirect_energy

func set_bounce_indirect_energy(value: float) -> void:
	bounce_indirect_energy = value

func get_bounces() -> int:
	return bounces

func set_bounces(value: int) -> void:
	bounces = value

func get_camera_attributes() -> CameraAttributes:
	return camera_attributes

func set_camera_attributes(value: CameraAttributes) -> void:
	camera_attributes = value

func get_denoiser_range() -> int:
	return denoiser_range

func set_denoiser_range(value: int) -> void:
	denoiser_range = value

func get_denoiser_strength() -> float:
	return denoiser_strength

func set_denoiser_strength(value: float) -> void:
	denoiser_strength = value

func is_directional() -> bool:
	return directional

func set_directional(value: bool) -> void:
	directional = value

func get_environment_custom_color() -> Color:
	return environment_custom_color

func set_environment_custom_color(value: Color) -> void:
	environment_custom_color = value

func get_environment_custom_energy() -> float:
	return environment_custom_energy

func set_environment_custom_energy(value: float) -> void:
	environment_custom_energy = value

func get_environment_custom_sky() -> Sky:
	return environment_custom_sky

func set_environment_custom_sky(value: Sky) -> void:
	environment_custom_sky = value

func get_environment_mode() -> int:
	return environment_mode

func set_environment_mode(value: int) -> void:
	environment_mode = value

func get_generate_probes() -> int:
	return generate_probes_subdiv

func set_generate_probes(value: int) -> void:
	generate_probes_subdiv = value

func is_interior() -> bool:
	return interior

func set_interior(value: bool) -> void:
	interior = value

func get_light_data() -> LightmapGIData:
	return light_data

func set_light_data(value: LightmapGIData) -> void:
	light_data = value

func get_max_texture_size() -> int:
	return max_texture_size

func set_max_texture_size(value: int) -> void:
	max_texture_size = value

func get_bake_quality() -> int:
	return quality

func set_bake_quality(value: int) -> void:
	quality = value

func get_shadowmask_mode() -> int:
	return shadowmask_mode

func set_shadowmask_mode(value: int) -> void:
	shadowmask_mode = value

func is_supersampling_enabled() -> bool:
	return supersampling

func set_supersampling_enabled(value: bool) -> void:
	supersampling = value

func get_supersampling_factor() -> float:
	return supersampling_factor

func set_supersampling_factor(value: float) -> void:
	supersampling_factor = value

func get_texel_scale() -> float:
	return texel_scale

func set_texel_scale(value: float) -> void:
	texel_scale = value

func is_using_denoiser() -> bool:
	return use_denoiser

func set_use_denoiser(value: bool) -> void:
	use_denoiser = value

func is_using_texture_for_bounces() -> bool:
	return use_texture_for_bounces

func set_use_texture_for_bounces(value: bool) -> void:
	use_texture_for_bounces = value

