extends GPUParticlesCollision3D
class_name GPUParticlesCollisionHeightField3D

## A real-time heightmap-shaped 3D particle collision shape affecting [GPUParticles3D] nodes.
##
## A real-time heightmap-shaped 3D particle collision shape affecting [GPUParticles3D] nodes.
## Heightmap shapes allow for efficiently representing collisions for convex and concave objects with a single "floor" (such as terrain). This is less flexible than [GPUParticlesCollisionSDF3D], but it doesn't require a baking step.
## [GPUParticlesCollisionHeightField3D] can also be regenerated in real-time when it is moved, when the camera moves, or even continuously. This makes [GPUParticlesCollisionHeightField3D] a good choice for weather effects such as rain and snow and games with highly dynamic geometry. However, this class is limited since heightmaps cannot represent overhangs (e.g. indoors or caves).
## [b]Note:[/b] [member ParticleProcessMaterial.collision_mode] must be [code]true[/code] on the [GPUParticles3D]'s process material for collision to work.
## [b]Note:[/b] Particle collision only affects [GPUParticles3D], not [CPUParticles3D].


## Generate a 256×256 heightmap. Intended for small-scale scenes, or larger scenes with no distant particles.
## Generate a 512×512 heightmap. Intended for medium-scale scenes, or larger scenes with no distant particles.
## Generate a 1024×1024 heightmap. Intended for large scenes with distant particles.
## Generate a 2048×2048 heightmap. Intended for very large scenes with distant particles.
## Generate a 4096×4096 heightmap. Intended for huge scenes with distant particles.
## Generate a 8192×8192 heightmap. Intended for gigantic scenes with distant particles.
## Represents the size of the [enum Resolution] enum.
## Only update the heightmap when the [GPUParticlesCollisionHeightField3D] node is moved, or when the camera moves if [member follow_camera_enabled] is [code]true[/code]. An update can be forced by slightly moving the [GPUParticlesCollisionHeightField3D] in any direction, or by calling [method RenderingServer.particles_collision_height_field_update].
## Update the heightmap every frame. This has a significant performance cost. This update should only be used when geometry that particles can collide with changes significantly during gameplay.

#enum Resolution
enum {
    RESOLUTION_256 = 0,
    RESOLUTION_512 = 1,
    RESOLUTION_1024 = 2,
    RESOLUTION_2048 = 3,
    RESOLUTION_4096 = 4,
    RESOLUTION_8192 = 5,
    RESOLUTION_MAX = 6,
}
#enum UpdateMode
enum {
    UPDATE_MODE_WHEN_MOVED = 0,
    UPDATE_MODE_ALWAYS = 1,
}
## If [code]true[/code], the [GPUParticlesCollisionHeightField3D] will follow the current camera in global space. The [GPUParticlesCollisionHeightField3D] does not need to be a child of the [Camera3D] node for this to work.
## Following the camera has a performance cost, as it will force the heightmap to update whenever the camera moves. Consider lowering [member resolution] to improve performance if [member follow_camera_enabled] is [code]true[/code].
var follow_camera_enabled: bool:
	get = is_follow_camera_enabled, set = set_follow_camera_enabled

## The visual layers to account for when updating the heightmap. Only [MeshInstance3D]s whose [member VisualInstance3D.layers] match with this [member heightfield_mask] will be included in the heightmap collision update. By default, all 20 user-visible layers are taken into account for updating the heightmap collision.
## [b]Note:[/b] Since the [member heightfield_mask] allows for 32 layers to be stored in total, there are an additional 12 layers that are only used internally by the engine and aren't exposed in the editor. Setting [member heightfield_mask] using a script allows you to toggle those reserved layers, which can be useful for editor plugins.
## To adjust [member heightfield_mask] more easily using a script, use [method get_heightfield_mask_value] and [method set_heightfield_mask_value].
var heightfield_mask: int:
	get = get_heightfield_mask, set = set_heightfield_mask

## Higher resolutions can represent small details more accurately in large scenes, at the cost of lower performance. If [member update_mode] is [constant UPDATE_MODE_ALWAYS], consider using the lowest resolution possible.
var resolution: int:
	get = get_resolution, set = set_resolution

## The collision heightmap's size in 3D units. To improve heightmap quality, [member size] should be set as small as possible while covering the parts of the scene you need.
var size: Vector3:
	get = get_size, set = set_size

## The update policy to use for the generated heightmap.
var update_mode: int:
	get = get_update_mode, set = set_update_mode



## Returns [code]true[/code] if the specified layer of the [member heightfield_mask] is enabled, given a [param layer_number] between [code]1[/code] and [code]20[/code], inclusive.
func get_heightfield_mask_value(layer_number: int) -> bool:
	pass;

## Based on [param value], enables or disables the specified layer in the [member heightfield_mask], given a [param layer_number] between [code]1[/code] and [code]20[/code], inclusive.
func set_heightfield_mask_value(layer_number: int, value: bool) -> void:
	pass;


func is_follow_camera_enabled() -> bool:
	return follow_camera_enabled

func set_follow_camera_enabled(value: bool) -> void:
	follow_camera_enabled = value

func get_heightfield_mask() -> int:
	return heightfield_mask

func set_heightfield_mask(value: int) -> void:
	heightfield_mask = value

func get_resolution() -> int:
	return resolution

func set_resolution(value: int) -> void:
	resolution = value

func get_size() -> Vector3:
	return size

func set_size(value: Vector3) -> void:
	size = value

func get_update_mode() -> int:
	return update_mode

func set_update_mode(value: int) -> void:
	update_mode = value

