extends RefCounted
class_name EditorTranslationParserPlugin

## Plugin for adding custom parsers to extract strings that are to be translated from custom files (.csv, .json etc.).
##
## [EditorTranslationParserPlugin] is invoked when a file is being parsed to extract strings that require translation. To define the parsing and string extraction logic, override the [method _parse_file] method in script.
## The return value should be an [Array] of [PackedStringArray]s, one for each extracted translatable string. Each entry should contain [code][msgid, msgctxt, msgid_plural, comment][/code], where all except [code]msgid[/code] are optional. Empty strings will be ignored.
## The extracted strings will be written into a POT file selected by user under "POT Generation" in "Localization" tab in "Project Settings" menu.
## Below shows an example of a custom parser that extracts strings from a CSV file to write into a POT.
## [codeblocks]
## [gdscript]
## @tool
## extends EditorTranslationParserPlugin
## func _parse_file(path):
## var ret: Array[PackedStringArray] = []
## var file = FileAccess.open(path, FileAccess.READ)
## var text = file.get_as_text()
## var split_strs = text.split(",", false)
## for s in split_strs:
## ret.append(PackedStringArray([s]))
## #print("Extracted string: " + s)
## return ret
## func _get_recognized_extensions():
## return ["csv"]
## [/gdscript]
## [csharp]
## using Godot;
## [Tool]
## public partial class CustomParser : EditorTranslationParserPlugin
## {
## public override Godot.Collections.Array<string[]> _ParseFile(string path)
## {
## Godot.Collections.Array<string[]> ret;
## using var file = FileAccess.Open(path, FileAccess.ModeFlags.Read);
## string text = file.GetAsText();
## string[] splitStrs = text.Split(",", allowEmpty: false);
## foreach (string s in splitStrs)
## {
## ret.Add([s]);
## //GD.Print($"Extracted string: {s}");
## }
## return ret;
## }
## public override string[] _GetRecognizedExtensions()
## {
## return ["csv"];
## }
## }
## [/csharp]
## [/codeblocks]
## To add a translatable string associated with a context, plural, or comment:
## [codeblocks]
## [gdscript]
## # This will add a message with msgid "Test 1", msgctxt "context", msgid_plural "test 1 plurals", and comment "test 1 comment".
## ret.append(PackedStringArray(["Test 1", "context", "test 1 plurals", "test 1 comment"]))
## # This will add a message with msgid "A test without context" and msgid_plural "plurals".
## ret.append(PackedStringArray(["A test without context", "", "plurals"]))
## # This will add a message with msgid "Only with context" and msgctxt "a friendly context".
## ret.append(PackedStringArray(["Only with context", "a friendly context"]))
## [/gdscript]
## [csharp]
## // This will add a message with msgid "Test 1", msgctxt "context", msgid_plural "test 1 plurals", and comment "test 1 comment".
## ret.Add(["Test 1", "context", "test 1 plurals", "test 1 comment"]);
## // This will add a message with msgid "A test without context" and msgid_plural "plurals".
## ret.Add(["A test without context", "", "plurals"]);
## // This will add a message with msgid "Only with context" and msgctxt "a friendly context".
## ret.Add(["Only with context", "a friendly context"]);
## [/csharp]
## [/codeblocks]
## [b]Note:[/b] If you override parsing logic for standard script types (GDScript, C#, etc.), it would be better to load the [code]path[/code] argument using [method ResourceLoader.load]. This is because built-in scripts are loaded as [Resource] type, not [FileAccess] type. For example:
## [codeblocks]
## [gdscript]
## func _parse_file(path):
## var res = ResourceLoader.load(path, "Script")
## var text = res.source_code
## # Parsing logic.
## func _get_recognized_extensions():
## return ["gd"]
## [/gdscript]
## [csharp]
## public override Godot.Collections.Array<string[]> _ParseFile(string path)
## {
## var res = ResourceLoader.Load<Script>(path, "Script");
## string text = res.SourceCode;
## // Parsing logic.
## }
## public override string[] _GetRecognizedExtensions()
## {
## return ["gd"];
## }
## [/csharp]
## [/codeblocks]
## To use [EditorTranslationParserPlugin], register it using the [method EditorPlugin.add_translation_parser_plugin] method first.





## Gets the list of file extensions to associate with this parser, e.g. [code]["csv"][/code].
func _get_recognized_extensions() -> PackedStringArray:
	pass;

## Override this method to define a custom parsing logic to extract the translatable strings.
func _parse_file(path: String) -> Array[PackedStringArray]:
	pass;


