extends RefCounted
class_name EditorContextMenuPlugin

## Plugin for adding custom context menus in the editor.
##
## [EditorContextMenuPlugin] allows for the addition of custom options in the editor's context menu.
## Currently, context menus are supported for three commonly used areas: the file system, scene tree, and editor script list panel.


## Context menu of Scene dock. [method _popup_menu] will be called with a list of paths to currently selected nodes, while option callback will receive the list of currently selected nodes.
## Context menu of FileSystem dock. [method _popup_menu] and option callback will be called with list of paths of the currently selected files.
## Context menu of Script editor's script tabs. [method _popup_menu] will be called with the path to the currently edited script, while option callback will receive reference to that script.
## The "Create..." submenu of FileSystem dock's context menu. [method _popup_menu] and option callback will be called with list of paths of the currently selected files.
## Context menu of Script editor's code editor. [method _popup_menu] will be called with the path to the [CodeEdit] node. You can fetch it using this code:
## [codeblock]
## func _popup_menu(paths):
## var code_edit = Engine.get_main_loop().root.get_node(paths[0]);
## [/codeblock]
## The option callback will receive reference to that node. You can use [CodeEdit] methods to perform symbol lookups etc.
## Context menu of scene tabs. [method _popup_menu] will be called with the path of the clicked scene, or empty [PackedStringArray] if the menu was opened on empty space. The option callback will receive the path of the clicked scene, or empty [String] if none was clicked.
## Context menu of 2D editor's basic right-click menu. [method _popup_menu] will be called with paths to all [CanvasItem] nodes under the cursor. You can fetch them using this code:
## [codeblock]
## func _popup_menu(paths):
## var canvas_item = Engine.get_main_loop().root.get_node(paths[0]); # Replace 0 with the desired index.
## [/codeblock]
## The paths array is empty if there weren't any nodes under cursor. The option callback will receive a typed array of [CanvasItem] nodes.

#enum ContextMenuSlot
enum {
    CONTEXT_SLOT_SCENE_TREE = 0,
    CONTEXT_SLOT_FILESYSTEM = 1,
    CONTEXT_SLOT_SCRIPT_EDITOR = 2,
    CONTEXT_SLOT_FILESYSTEM_CREATE = 3,
    CONTEXT_SLOT_SCRIPT_EDITOR_CODE = 4,
    CONTEXT_SLOT_SCENE_TABS = 5,
    CONTEXT_SLOT_2D_EDITOR = 6,
}


## Called when creating a context menu, custom options can be added by using the [method add_context_menu_item] or [method add_context_menu_item_from_shortcut] functions. [param paths] contains currently selected paths (depending on menu), which can be used to conditionally add options.
func _popup_menu(paths: PackedStringArray) -> void:
	pass;

## Add custom option to the context menu of the plugin's specified slot. When the option is activated, [param callback] will be called. Callback should take single [Array] argument; array contents depend on context menu slot.
## [codeblock]
## func _popup_menu(paths):
## add_context_menu_item("File Custom options", handle, ICON)
## [/codeblock]
## If you want to assign shortcut to the menu item, use [method add_context_menu_item_from_shortcut] instead.
func add_context_menu_item(name: String, callback: Callable, icon: Texture2D = null) -> void:
	pass;

## Add custom option to the context menu of the plugin's specified slot. The option will have the [param shortcut] assigned and reuse its callback. The shortcut has to be registered beforehand with [method add_menu_shortcut].
## [codeblock]
## func _init():
## add_menu_shortcut(SHORTCUT, handle)
## func _popup_menu(paths):
## add_context_menu_item_from_shortcut("File Custom options", SHORTCUT, ICON)
## [/codeblock]
func add_context_menu_item_from_shortcut(name: String, shortcut: Shortcut, icon: Texture2D = null) -> void:
	pass;

## Add a submenu to the context menu of the plugin's specified slot. The submenu is not automatically handled, you need to connect to its signals yourself. Also the submenu is freed on every popup, so provide a new [PopupMenu] every time.
## [codeblock]
## func _popup_menu(paths):
## var popup_menu = PopupMenu.new()
## popup_menu.add_item("Blue")
## popup_menu.add_item("White")
## popup_menu.id_pressed.connect(_on_color_submenu_option)
## add_context_submenu_item("Set Node Color", popup_menu)
## [/codeblock]
func add_context_submenu_item(name: String, menu: PopupMenu, icon: Texture2D = null) -> void:
	pass;

## Registers a shortcut associated with the plugin's context menu. This method should be called once (e.g. in plugin's [method Object._init]). [param callback] will be called when user presses the specified [param shortcut] while the menu's context is in effect (e.g. FileSystem dock is focused). Callback should take single [Array] argument; array contents depend on context menu slot.
## [codeblock]
## func _init():
## add_menu_shortcut(SHORTCUT, handle)
## [/codeblock]
func add_menu_shortcut(shortcut: Shortcut, callback: Callable) -> void:
	pass;


