extends Node3D
class_name Camera3D

## Camera node, displays from a point of view.
##
## [Camera3D] is a special node that displays what is visible from its current location. Cameras register themselves in the nearest [Viewport] node (when ascending the tree). Only one camera can be active per viewport. If no viewport is available ascending the tree, the camera will register in the global viewport. In other words, a camera just provides 3D display capabilities to a [Viewport], and, without one, a scene registered in that [Viewport] (or higher viewports) can't be displayed.
##
## @tutorial(Third Person Shooter (TPS) Demo): https://godotengine.org/asset-library/asset/2710


## Perspective projection. Objects on the screen becomes smaller when they are far away.
## Orthogonal projection, also known as orthographic projection. Objects remain the same size on the screen no matter how far away they are.
## Frustum projection. This mode allows adjusting [member frustum_offset] to create "tilted frustum" effects.
## Preserves the horizontal aspect ratio; also known as Vert- scaling. This is usually the best option for projects running in portrait mode, as taller aspect ratios will benefit from a wider vertical FOV.
## Preserves the vertical aspect ratio; also known as Hor+ scaling. This is usually the best option for projects running in landscape mode, as wider aspect ratios will automatically benefit from a wider horizontal FOV.
## Disables [url=https://en.wikipedia.org/wiki/Doppler_effect]Doppler effect[/url] simulation (default).
## Simulate [url=https://en.wikipedia.org/wiki/Doppler_effect]Doppler effect[/url] by tracking positions of objects that are changed in [code]_process[/code]. Changes in the relative velocity of this camera compared to those objects affect how audio is perceived (changing the audio's [member AudioStreamPlayer3D.pitch_scale]).
## Simulate [url=https://en.wikipedia.org/wiki/Doppler_effect]Doppler effect[/url] by tracking positions of objects that are changed in [code]_physics_process[/code]. Changes in the relative velocity of this camera compared to those objects affect how audio is perceived (changing the audio's [member AudioStreamPlayer3D.pitch_scale]).

#enum ProjectionType
enum {
    PROJECTION_PERSPECTIVE = 0,
    PROJECTION_ORTHOGONAL = 1,
    PROJECTION_FRUSTUM = 2,
}
#enum KeepAspect
enum {
    KEEP_WIDTH = 0,
    KEEP_HEIGHT = 1,
}
#enum DopplerTracking
enum {
    DOPPLER_TRACKING_DISABLED = 0,
    DOPPLER_TRACKING_IDLE_STEP = 1,
    DOPPLER_TRACKING_PHYSICS_STEP = 2,
}
## The [CameraAttributes] to use for this camera.
var attributes: CameraAttributes:
	get = get_attributes, set = set_attributes

## The [Compositor] to use for this camera.
var compositor: Compositor:
	get = get_compositor, set = set_compositor

## The culling mask that describes which [member VisualInstance3D.layers] are rendered by this camera. By default, all 20 user-visible layers are rendered.
## [b]Note:[/b] Since the [member cull_mask] allows for 32 layers to be stored in total, there are an additional 12 layers that are only used internally by the engine and aren't exposed in the editor. Setting [member cull_mask] using a script allows you to toggle those reserved layers, which can be useful for editor plugins.
## To adjust [member cull_mask] more easily using a script, use [method get_cull_mask_value] and [method set_cull_mask_value].
## [b]Note:[/b] [VoxelGI], SDFGI and [LightmapGI] will always take all layers into account to determine what contributes to global illumination. If this is an issue, set [member GeometryInstance3D.gi_mode] to [constant GeometryInstance3D.GI_MODE_DISABLED] for meshes and [member Light3D.light_bake_mode] to [constant Light3D.BAKE_DISABLED] for lights to exclude them from global illumination.
var cull_mask: int:
	get = get_cull_mask, set = set_cull_mask

## If [code]true[/code], the ancestor [Viewport] is currently using this camera.
## If multiple cameras are in the scene, one will always be made current. For example, if two [Camera3D] nodes are present in the scene and only one is current, setting one camera's [member current] to [code]false[/code] will cause the other camera to be made current.
var current: bool:
	get = is_current, set = set_current

## If not [constant DOPPLER_TRACKING_DISABLED], this camera will simulate the [url=https://en.wikipedia.org/wiki/Doppler_effect]Doppler effect[/url] for objects changed in particular [code]_process[/code] methods. See [enum DopplerTracking] for possible values.
var doppler_tracking: int:
	get = get_doppler_tracking, set = set_doppler_tracking

## The [Environment] to use for this camera.
var environment: Environment:
	get = get_environment, set = set_environment

## The distance to the far culling boundary for this camera relative to its local Z axis. Higher values allow the camera to see further away, while decreasing [member far] can improve performance if it results in objects being partially or fully culled.
var far: float:
	get = get_far, set = set_far

## The camera's field of view angle (in degrees). Only applicable in perspective mode. Since [member keep_aspect] locks one axis, [member fov] sets the other axis' field of view angle.
## For reference, the default vertical field of view value ([code]75.0[/code]) is equivalent to a horizontal FOV of:
## - ~91.31 degrees in a 4:3 viewport
## - ~101.67 degrees in a 16:10 viewport
## - ~107.51 degrees in a 16:9 viewport
## - ~121.63 degrees in a 21:9 viewport
var fov: float:
	get = get_fov, set = set_fov

## The camera's frustum offset. This can be changed from the default to create "tilted frustum" effects such as [url=https://zdoom.org/wiki/Y-shearing]Y-shearing[/url].
## [b]Note:[/b] Only effective if [member projection] is [constant PROJECTION_FRUSTUM].
var frustum_offset: Vector2:
	get = get_frustum_offset, set = set_frustum_offset

## The horizontal (X) offset of the camera viewport.
var h_offset: float:
	get = get_h_offset, set = set_h_offset

## The axis to lock during [member fov]/[member size] adjustments. Can be either [constant KEEP_WIDTH] or [constant KEEP_HEIGHT].
var keep_aspect: int:
	get = get_keep_aspect_mode, set = set_keep_aspect_mode

## The distance to the near culling boundary for this camera relative to its local Z axis. Lower values allow the camera to see objects more up close to its origin, at the cost of lower precision across the [i]entire[/i] range. Values lower than the default can lead to increased Z-fighting.
var near: float:
	get = get_near, set = set_near

## The camera's projection mode. In [constant PROJECTION_PERSPECTIVE] mode, objects' Z distance from the camera's local space scales their perceived size.
var projection: int:
	get = get_projection, set = set_projection

## The camera's size in meters measured as the diameter of the width or height, depending on [member keep_aspect]. Only applicable in orthogonal and frustum modes.
var size: float:
	get = get_size, set = set_size

## The vertical (Y) offset of the camera viewport.
var v_offset: float:
	get = get_v_offset, set = set_v_offset



## If this is the current camera, remove it from being current. If [param enable_next] is [code]true[/code], request to make the next camera current, if any.
func clear_current(enable_next: bool = true) -> void:
	pass;

## Returns the projection matrix that this camera uses to render to its associated viewport. The camera must be part of the scene tree to function.
func get_camera_projection() -> Projection:
	pass;

## Returns the camera's RID from the [RenderingServer].
func get_camera_rid() -> RID:
	pass;

## Returns the transform of the camera plus the vertical ([member v_offset]) and horizontal ([member h_offset]) offsets; and any other adjustments made to the position and orientation of the camera by subclassed cameras such as [XRCamera3D].
func get_camera_transform() -> Transform3D:
	pass;

## Returns whether or not the specified layer of the [member cull_mask] is enabled, given a [param layer_number] between 1 and 20.
func get_cull_mask_value(layer_number: int) -> bool:
	pass;

## Returns the camera's frustum planes in world space units as an array of [Plane]s in the following order: near, far, left, top, right, bottom. Not to be confused with [member frustum_offset].
func get_frustum() -> Array[Plane]:
	pass;

## Returns the RID of a pyramid shape encompassing the camera's view frustum, ignoring the camera's near plane. The tip of the pyramid represents the position of the camera.
func get_pyramid_shape_rid() -> RID:
	pass;

## Returns [code]true[/code] if the given position is behind the camera (the blue part of the linked diagram). [url=https://raw.githubusercontent.com/godotengine/godot-docs/master/img/camera3d_position_frustum.png]See this diagram[/url] for an overview of position query methods.
## [b]Note:[/b] A position which returns [code]false[/code] may still be outside the camera's field of view.
func is_position_behind(world_point: Vector3) -> bool:
	pass;

## Returns [code]true[/code] if the given position is inside the camera's frustum (the green part of the linked diagram). [url=https://raw.githubusercontent.com/godotengine/godot-docs/master/img/camera3d_position_frustum.png]See this diagram[/url] for an overview of position query methods.
func is_position_in_frustum(world_point: Vector3) -> bool:
	pass;

## Makes this camera the current camera for the [Viewport] (see class description). If the camera node is outside the scene tree, it will attempt to become current once it's added.
func make_current() -> void:
	pass;

## Returns a normal vector from the screen point location directed along the camera. Orthogonal cameras are normalized. Perspective cameras account for perspective, screen width/height, etc.
func project_local_ray_normal(screen_point: Vector2) -> Vector3:
	pass;

## Returns the 3D point in world space that maps to the given 2D coordinate in the [Viewport] rectangle on a plane that is the given [param z_depth] distance into the scene away from the camera.
func project_position(screen_point: Vector2, z_depth: float) -> Vector3:
	pass;

## Returns a normal vector in world space, that is the result of projecting a point on the [Viewport] rectangle by the inverse camera projection. This is useful for casting rays in the form of (origin, normal) for object intersection or picking.
func project_ray_normal(screen_point: Vector2) -> Vector3:
	pass;

## Returns a 3D position in world space, that is the result of projecting a point on the [Viewport] rectangle by the inverse camera projection. This is useful for casting rays in the form of (origin, normal) for object intersection or picking.
func project_ray_origin(screen_point: Vector2) -> Vector3:
	pass;

## Based on [param value], enables or disables the specified layer in the [member cull_mask], given a [param layer_number] between 1 and 20.
func set_cull_mask_value(layer_number: int, value: bool) -> void:
	pass;

## Sets the camera projection to frustum mode (see [constant PROJECTION_FRUSTUM]), by specifying a [param size], an [param offset], and the [param z_near] and [param z_far] clip planes in world space units. See also [member frustum_offset].
func set_frustum(size: float, offset: Vector2, z_near: float, z_far: float) -> void:
	pass;

## Sets the camera projection to orthogonal mode (see [constant PROJECTION_ORTHOGONAL]), by specifying a [param size], and the [param z_near] and [param z_far] clip planes in world space units. (As a hint, 2D games often use this projection, with values specified in pixels.)
func set_orthogonal(size: float, z_near: float, z_far: float) -> void:
	pass;

## Sets the camera projection to perspective mode (see [constant PROJECTION_PERSPECTIVE]), by specifying a [param fov] (field of view) angle in degrees, and the [param z_near] and [param z_far] clip planes in world space units.
func set_perspective(fov: float, z_near: float, z_far: float) -> void:
	pass;

## Returns the 2D coordinate in the [Viewport] rectangle that maps to the given 3D point in world space.
## [b]Note:[/b] When using this to position GUI elements over a 3D viewport, use [method is_position_behind] to prevent them from appearing if the 3D point is behind the camera:
## [codeblock]
## # This code block is part of a script that inherits from Node3D.
## # `control` is a reference to a node inheriting from Control.
## control.visible = not get_viewport().get_camera_3d().is_position_behind(global_transform.origin)
## control.position = get_viewport().get_camera_3d().unproject_position(global_transform.origin)
## [/codeblock]
func unproject_position(world_point: Vector3) -> Vector2:
	pass;


func get_attributes() -> CameraAttributes:
	return attributes

func set_attributes(value: CameraAttributes) -> void:
	attributes = value

func get_compositor() -> Compositor:
	return compositor

func set_compositor(value: Compositor) -> void:
	compositor = value

func get_cull_mask() -> int:
	return cull_mask

func set_cull_mask(value: int) -> void:
	cull_mask = value

func is_current() -> bool:
	return current

func set_current(value: bool) -> void:
	current = value

func get_doppler_tracking() -> int:
	return doppler_tracking

func set_doppler_tracking(value: int) -> void:
	doppler_tracking = value

func get_environment() -> Environment:
	return environment

func set_environment(value: Environment) -> void:
	environment = value

func get_far() -> float:
	return far

func set_far(value: float) -> void:
	far = value

func get_fov() -> float:
	return fov

func set_fov(value: float) -> void:
	fov = value

func get_frustum_offset() -> Vector2:
	return frustum_offset

func set_frustum_offset(value: Vector2) -> void:
	frustum_offset = value

func get_h_offset() -> float:
	return h_offset

func set_h_offset(value: float) -> void:
	h_offset = value

func get_keep_aspect_mode() -> int:
	return keep_aspect

func set_keep_aspect_mode(value: int) -> void:
	keep_aspect = value

func get_near() -> float:
	return near

func set_near(value: float) -> void:
	near = value

func get_projection() -> int:
	return projection

func set_projection(value: int) -> void:
	projection = value

func get_size() -> float:
	return size

func set_size(value: float) -> void:
	size = value

func get_v_offset() -> float:
	return v_offset

func set_v_offset(value: float) -> void:
	v_offset = value

