extends Control
class_name BaseButton

## Abstract base class for GUI buttons.
##
## [BaseButton] is an abstract base class for GUI buttons. It doesn't display anything by itself.


## Emitted when the button starts being held down.
signal button_down
## Emitted when the button stops being held down.
signal button_up
## Emitted when the button is toggled or pressed. This is on [signal button_down] if [member action_mode] is [constant ACTION_MODE_BUTTON_PRESS] and on [signal button_up] otherwise.
## If you need to know the button's pressed state (and [member toggle_mode] is active), use [signal toggled] instead.
signal pressed
## Emitted when the button was just toggled between pressed and normal states (only if [member toggle_mode] is active). The new state is contained in the [param toggled_on] argument.
signal toggled(toggled_on: bool)
## The normal state (i.e. not pressed, not hovered, not toggled and enabled) of buttons.
## The state of buttons are pressed.
## The state of buttons are hovered.
## The state of buttons are disabled.
## The state of buttons are both hovered and pressed.
## Require just a press to consider the button clicked.
## Require a press and a subsequent release before considering the button clicked.

#enum DrawMode
enum {
    DRAW_NORMAL = 0,
    DRAW_PRESSED = 1,
    DRAW_HOVER = 2,
    DRAW_DISABLED = 3,
    DRAW_HOVER_PRESSED = 4,
}
#enum ActionMode
enum {
    ACTION_MODE_BUTTON_PRESS = 0,
    ACTION_MODE_BUTTON_RELEASE = 1,
}
## Determines when the button is considered clicked, one of the [enum ActionMode] constants.
var action_mode: int:
	get = get_action_mode, set = set_action_mode

## The [ButtonGroup] associated with the button. Not to be confused with node groups.
## [b]Note:[/b] The button will be configured as a radio button if a [ButtonGroup] is assigned to it.
var button_group: ButtonGroup:
	get = get_button_group, set = set_button_group

## Binary mask to choose which mouse buttons this button will respond to.
## To allow both left-click and right-click, use [code]MOUSE_BUTTON_MASK_LEFT | MOUSE_BUTTON_MASK_RIGHT[/code].
var button_mask: int:
	get = get_button_mask, set = set_button_mask

## If [code]true[/code], the button's state is pressed. Means the button is pressed down or toggled (if [member toggle_mode] is active). Only works if [member toggle_mode] is [code]true[/code].
## [b]Note:[/b] Changing the value of [member button_pressed] will result in [signal toggled] to be emitted. If you want to change the pressed state without emitting that signal, use [method set_pressed_no_signal].
var button_pressed: bool:
	get = is_pressed, set = set_pressed

## If [code]true[/code], the button is in disabled state and can't be clicked or toggled.
var disabled: bool:
	get = is_disabled, set = set_disabled

var focus_mode: int:
	get = get_focus_mode, set = set_focus_mode

## If [code]true[/code], the button stays pressed when moving the cursor outside the button while pressing it.
## [b]Note:[/b] This property only affects the button's visual appearance. Signals will be emitted at the same moment regardless of this property's value.
var keep_pressed_outside: bool:
	get = is_keep_pressed_outside, set = set_keep_pressed_outside

## [Shortcut] associated to the button.
var shortcut: Shortcut:
	get = get_shortcut, set = set_shortcut

## If [code]true[/code], the button will highlight for a short amount of time when its shortcut is activated. If [code]false[/code] and [member toggle_mode] is [code]false[/code], the shortcut will activate without any visual feedback.
var shortcut_feedback: bool:
	get = is_shortcut_feedback, set = set_shortcut_feedback

## If [code]true[/code], the button will add information about its shortcut in the tooltip.
## [b]Note:[/b] This property does nothing when the tooltip control is customized using [method Control._make_custom_tooltip].
var shortcut_in_tooltip: bool:
	get = is_shortcut_in_tooltip_enabled, set = set_shortcut_in_tooltip

## If [code]true[/code], the button is in toggle mode. Makes the button flip state between pressed and unpressed each time its area is clicked.
var toggle_mode: bool:
	get = is_toggle_mode, set = set_toggle_mode



## Called when the button is pressed. If you need to know the button's pressed state (and [member toggle_mode] is active), use [method _toggled] instead.
func _pressed() -> void:
	pass;

## Called when the button is toggled (only if [member toggle_mode] is active).
func _toggled(toggled_on: bool) -> void:
	pass;

## Returns the visual state used to draw the button. This is useful mainly when implementing your own draw code by either overriding _draw() or connecting to "draw" signal. The visual state of the button is defined by the [enum DrawMode] enum.
func get_draw_mode() -> int:
	pass;

## Returns [code]true[/code] if the mouse has entered the button and has not left it yet.
func is_hovered() -> bool:
	pass;

## Changes the [member button_pressed] state of the button, without emitting [signal toggled]. Use when you just want to change the state of the button without sending the pressed event (e.g. when initializing scene). Only works if [member toggle_mode] is [code]true[/code].
## [b]Note:[/b] This method doesn't unpress other buttons in [member button_group].
func set_pressed_no_signal(pressed: bool) -> void:
	pass;


func get_action_mode() -> int:
	return action_mode

func set_action_mode(value: int) -> void:
	action_mode = value

func get_button_group() -> ButtonGroup:
	return button_group

func set_button_group(value: ButtonGroup) -> void:
	button_group = value

func get_button_mask() -> int:
	return button_mask

func set_button_mask(value: int) -> void:
	button_mask = value

func is_pressed() -> bool:
	return button_pressed

func set_pressed(value: bool) -> void:
	button_pressed = value

func is_disabled() -> bool:
	return disabled

func set_disabled(value: bool) -> void:
	disabled = value

func get_focus_mode() -> int:
	return focus_mode

func set_focus_mode(value: int) -> void:
	focus_mode = value

func is_keep_pressed_outside() -> bool:
	return keep_pressed_outside

func set_keep_pressed_outside(value: bool) -> void:
	keep_pressed_outside = value

func get_shortcut() -> Shortcut:
	return shortcut

func set_shortcut(value: Shortcut) -> void:
	shortcut = value

func is_shortcut_feedback() -> bool:
	return shortcut_feedback

func set_shortcut_feedback(value: bool) -> void:
	shortcut_feedback = value

func is_shortcut_in_tooltip_enabled() -> bool:
	return shortcut_in_tooltip

func set_shortcut_in_tooltip(value: bool) -> void:
	shortcut_in_tooltip = value

func is_toggle_mode() -> bool:
	return toggle_mode

func set_toggle_mode(value: bool) -> void:
	toggle_mode = value

