extends Node
class_name AudioStreamPlayer

## A node for audio playback.
##
## The [AudioStreamPlayer] node plays an audio stream non-positionally. It is ideal for user interfaces, menus, or background music.
## To use this node, [member stream] needs to be set to a valid [AudioStream] resource. Playing more than one sound at the same time is also supported, see [member max_polyphony].
## If you need to play audio at a specific position, use [AudioStreamPlayer2D] or [AudioStreamPlayer3D] instead.
##
## @tutorial(Audio streams): https://docs.godotengine.org/en/stable/tutorials/audio/audio_streams.html
## @tutorial(2D Dodge The Creeps Demo): https://godotengine.org/asset-library/asset/2712
## @tutorial(Audio Device Changer Demo): https://godotengine.org/asset-library/asset/2758
## @tutorial(Audio Generator Demo): https://godotengine.org/asset-library/asset/2759
## @tutorial(Audio Microphone Record Demo): https://godotengine.org/asset-library/asset/2760
## @tutorial(Audio Spectrum Visualizer Demo): https://godotengine.org/asset-library/asset/2762


## Emitted when a sound finishes playing without interruptions. This signal is [i]not[/i] emitted when calling [method stop], or when exiting the tree while sounds are playing.
signal finished
## The audio will be played only on the first channel. This is the default.
## The audio will be played on all surround channels.
## The audio will be played on the second channel, which is usually the center.

#enum MixTarget
enum {
    MIX_TARGET_STEREO = 0,
    MIX_TARGET_SURROUND = 1,
    MIX_TARGET_CENTER = 2,
}
## If [code]true[/code], this node calls [method play] when entering the tree.
var autoplay: bool:
	get = is_autoplay_enabled, set = set_autoplay

## The target bus name. All sounds from this node will be playing on this bus.
## [b]Note:[/b] At runtime, if no bus with the given name exists, all sounds will fall back on [code]"Master"[/code]. See also [method AudioServer.get_bus_name].
var bus: StringName:
	get = get_bus, set = set_bus

## The maximum number of sounds this node can play at the same time. Calling [method play] after this value is reached will cut off the oldest sounds.
var max_polyphony: int:
	get = get_max_polyphony, set = set_max_polyphony

## The mix target channels, as one of the [enum MixTarget] constants. Has no effect when two speakers or less are detected (see [enum AudioServer.SpeakerMode]).
var mix_target: int:
	get = get_mix_target, set = set_mix_target

## The audio's pitch and tempo, as a multiplier of the [member stream]'s sample rate. A value of [code]2.0[/code] doubles the audio's pitch, while a value of [code]0.5[/code] halves the pitch.
var pitch_scale: float:
	get = get_pitch_scale, set = set_pitch_scale

## The playback type of the stream player. If set other than to the default value, it will force that playback type.
var playback_type: int:
	get = get_playback_type, set = set_playback_type

## If [code]true[/code], this node is playing sounds. Setting this property has the same effect as [method play] and [method stop].
var playing: bool:
	get = is_playing, set = set_playing

## The [AudioStream] resource to be played. Setting this property stops all currently playing sounds. If left empty, the [AudioStreamPlayer] does not work.
var stream: AudioStream:
	get = get_stream, set = set_stream

## If [code]true[/code], the sounds are paused. Setting [member stream_paused] to [code]false[/code] resumes all sounds.
## [b]Note:[/b] This property is automatically changed when exiting or entering the tree, or this node is paused (see [member Node.process_mode]).
var stream_paused: bool:
	get = get_stream_paused, set = set_stream_paused

## Volume of sound, in decibels. This is an offset of the [member stream]'s volume.
## [b]Note:[/b] To convert between decibel and linear energy (like most volume sliders do), use [member volume_linear], or [method @GlobalScope.db_to_linear] and [method @GlobalScope.linear_to_db].
var volume_db: float:
	get = get_volume_db, set = set_volume_db

## Volume of sound, as a linear value.
## [b]Note:[/b] This member modifies [member volume_db] for convenience. The returned value is equivalent to the result of [method @GlobalScope.db_to_linear] on [member volume_db]. Setting this member is equivalent to setting [member volume_db] to the result of [method @GlobalScope.linear_to_db] on a value.
var volume_linear: float:
	get = get_volume_linear, set = set_volume_linear



## Returns the position in the [AudioStream] of the latest sound, in seconds. Returns [code]0.0[/code] if no sounds are playing.
## [b]Note:[/b] The position is not always accurate, as the [AudioServer] does not mix audio every processed frame. To get more accurate results, add [method AudioServer.get_time_since_last_mix] to the returned position.
## [b]Note:[/b] This method always returns [code]0.0[/code] if the [member stream] is an [AudioStreamInteractive], since it can have multiple clips playing at once.
func get_playback_position() -> float:
	pass;

## Returns the latest [AudioStreamPlayback] of this node, usually the most recently created by [method play]. If no sounds are playing, this method fails and returns an empty playback.
func get_stream_playback() -> AudioStreamPlayback:
	pass;

## Returns [code]true[/code] if any sound is active, even if [member stream_paused] is set to [code]true[/code]. See also [member playing] and [method get_stream_playback].
func has_stream_playback() -> bool:
	pass;

## Plays a sound from the beginning, or the given [param from_position] in seconds.
func play(from_position: float = 0.0) -> void:
	pass;

## Restarts all sounds to be played from the given [param to_position], in seconds. Does nothing if no sounds are playing.
func seek(to_position: float) -> void:
	pass;

## Stops all sounds from this node.
func stop() -> void:
	pass;


func is_autoplay_enabled() -> bool:
	return autoplay

func set_autoplay(value: bool) -> void:
	autoplay = value

func get_bus() -> StringName:
	return bus

func set_bus(value: StringName) -> void:
	bus = value

func get_max_polyphony() -> int:
	return max_polyphony

func set_max_polyphony(value: int) -> void:
	max_polyphony = value

func get_mix_target() -> int:
	return mix_target

func set_mix_target(value: int) -> void:
	mix_target = value

func get_pitch_scale() -> float:
	return pitch_scale

func set_pitch_scale(value: float) -> void:
	pitch_scale = value

func get_playback_type() -> int:
	return playback_type

func set_playback_type(value: int) -> void:
	playback_type = value

func is_playing() -> bool:
	return playing

func set_playing(value: bool) -> void:
	playing = value

func get_stream() -> AudioStream:
	return stream

func set_stream(value: AudioStream) -> void:
	stream = value

func get_stream_paused() -> bool:
	return stream_paused

func set_stream_paused(value: bool) -> void:
	stream_paused = value

func get_volume_db() -> float:
	return volume_db

func set_volume_db(value: float) -> void:
	volume_db = value

func get_volume_linear() -> float:
	return volume_linear

func set_volume_linear(value: float) -> void:
	volume_linear = value

