extends AnimationRootNode
class_name AnimationNodeAnimation

## An input animation for an [AnimationNodeBlendTree].
##
## A resource to add to an [AnimationNodeBlendTree]. Only has one output port using the [member animation] property. Used as an input for [AnimationNode]s that blend animations together.
##
## @tutorial(Using AnimationTree): https://docs.godotengine.org/en/stable/tutorials/animation/animation_tree.html
## @tutorial(3D Platformer Demo): https://godotengine.org/asset-library/asset/2748
## @tutorial(Third Person Shooter (TPS) Demo): https://godotengine.org/asset-library/asset/2710


## Plays animation in forward direction.
## Plays animation in backward direction.

#enum PlayMode
enum {
    PLAY_MODE_FORWARD = 0,
    PLAY_MODE_BACKWARD = 1,
}
## If [code]true[/code], on receiving a request to play an animation from the start, the first frame is not drawn, but only processed, and playback starts from the next frame.
## See also the notes of [method AnimationPlayer.play].
var advance_on_start: bool:
	get = is_advance_on_start, set = set_advance_on_start

## Animation to use as an output. It is one of the animations provided by [member AnimationTree.anim_player].
var animation: StringName:
	get = get_animation, set = set_animation

## If [member use_custom_timeline] is [code]true[/code], override the loop settings of the original [Animation] resource with the value.
## [b]Note:[/b] If the [member Animation.loop_mode] isn't set to looping, the [method Animation.track_set_interpolation_loop_wrap] option will not be respected. If you cannot get the expected behavior, consider duplicating the [Animation] resource and changing the loop settings.
var loop_mode: int:
	get = get_loop_mode, set = set_loop_mode

## Determines the playback direction of the animation.
var play_mode: int:
	get = get_play_mode, set = set_play_mode

## If [member use_custom_timeline] is [code]true[/code], offset the start position of the animation.
## This is useful for adjusting which foot steps first in 3D walking animations.
var start_offset: float:
	get = get_start_offset, set = set_start_offset

## If [code]true[/code], scales the time so that the length specified in [member timeline_length] is one cycle.
## This is useful for matching the periods of walking and running animations.
## If [code]false[/code], the original animation length is respected. If you set the loop to [member loop_mode], the animation will loop in [member timeline_length].
var stretch_time_scale: bool:
	get = is_stretching_time_scale, set = set_stretch_time_scale

## If [member use_custom_timeline] is [code]true[/code], offset the start position of the animation.
var timeline_length: float:
	get = get_timeline_length, set = set_timeline_length

## If [code]true[/code], [AnimationNode] provides an animation based on the [Animation] resource with some parameters adjusted.
var use_custom_timeline: bool:
	get = is_using_custom_timeline, set = set_use_custom_timeline




func is_advance_on_start() -> bool:
	return advance_on_start

func set_advance_on_start(value: bool) -> void:
	advance_on_start = value

func get_animation() -> StringName:
	return animation

func set_animation(value: StringName) -> void:
	animation = value

func get_loop_mode() -> int:
	return loop_mode

func set_loop_mode(value: int) -> void:
	loop_mode = value

func get_play_mode() -> int:
	return play_mode

func set_play_mode(value: int) -> void:
	play_mode = value

func get_start_offset() -> float:
	return start_offset

func set_start_offset(value: float) -> void:
	start_offset = value

func is_stretching_time_scale() -> bool:
	return stretch_time_scale

func set_stretch_time_scale(value: bool) -> void:
	stretch_time_scale = value

func get_timeline_length() -> float:
	return timeline_length

func set_timeline_length(value: float) -> void:
	timeline_length = value

func is_using_custom_timeline() -> bool:
	return use_custom_timeline

func set_use_custom_timeline(value: bool) -> void:
	use_custom_timeline = value

