extends Node
class_name AnimationMixer

## Base class for [AnimationPlayer] and [AnimationTree].
##
## Base class for [AnimationPlayer] and [AnimationTree] to manage animation lists. It also has general properties and methods for playback and blending.
## After instantiating the playback information data within the extended class, the blending is processed by the [AnimationMixer].
##
## @tutorial(Migrating Animations from Godot 4.0 to 4.3): https://godotengine.org/article/migrating-animations-from-godot-4-0-to-4-3/


## Notifies when an animation finished playing.
## [b]Note:[/b] This signal is not emitted if an animation is looping.
signal animation_finished(anim_name: StringName)
## Notifies when the animation libraries have changed.
signal animation_libraries_updated
## Notifies when an animation list is changed.
signal animation_list_changed
## Notifies when an animation starts playing.
signal animation_started(anim_name: StringName)
## Notifies when the caches have been cleared, either automatically, or manually via [method clear_caches].
signal caches_cleared
## Notifies when the blending result related have been applied to the target objects.
signal mixer_applied
## Notifies when the property related process have been updated.
signal mixer_updated
## Process animation during physics frames (see [constant Node.NOTIFICATION_INTERNAL_PHYSICS_PROCESS]). This is especially useful when animating physics bodies.
## Process animation during process frames (see [constant Node.NOTIFICATION_INTERNAL_PROCESS]).
## Do not process animation. Use [method advance] to process the animation manually.
## Batch method calls during the animation process, then do the calls after events are processed. This avoids bugs involving deleting nodes or modifying the AnimationPlayer while playing.
## Make method calls immediately when reached in the animation.
## An [constant Animation.UPDATE_DISCRETE] track value takes precedence when blending [constant Animation.UPDATE_CONTINUOUS] or [constant Animation.UPDATE_CAPTURE] track values and [constant Animation.UPDATE_DISCRETE] track values.
## An [constant Animation.UPDATE_CONTINUOUS] or [constant Animation.UPDATE_CAPTURE] track value takes precedence when blending the [constant Animation.UPDATE_CONTINUOUS] or [constant Animation.UPDATE_CAPTURE] track values and the [constant Animation.UPDATE_DISCRETE] track values. This is the default behavior for [AnimationPlayer].
## Always treat the [constant Animation.UPDATE_DISCRETE] track value as [constant Animation.UPDATE_CONTINUOUS] with [constant Animation.INTERPOLATION_NEAREST]. This is the default behavior for [AnimationTree].
## If a value track has un-interpolatable type key values, it is internally converted to use [constant ANIMATION_CALLBACK_MODE_DISCRETE_RECESSIVE] with [constant Animation.UPDATE_DISCRETE].
## Un-interpolatable type list:
## - [constant @GlobalScope.TYPE_NIL]
## - [constant @GlobalScope.TYPE_NODE_PATH]
## - [constant @GlobalScope.TYPE_RID]
## - [constant @GlobalScope.TYPE_OBJECT]
## - [constant @GlobalScope.TYPE_CALLABLE]
## - [constant @GlobalScope.TYPE_SIGNAL]
## - [constant @GlobalScope.TYPE_DICTIONARY]
## - [constant @GlobalScope.TYPE_PACKED_BYTE_ARRAY]
## [constant @GlobalScope.TYPE_BOOL] and [constant @GlobalScope.TYPE_INT] are treated as [constant @GlobalScope.TYPE_FLOAT] during blending and rounded when the result is retrieved.
## It is same for arrays and vectors with them such as [constant @GlobalScope.TYPE_PACKED_INT32_ARRAY] or [constant @GlobalScope.TYPE_VECTOR2I], they are treated as [constant @GlobalScope.TYPE_PACKED_FLOAT32_ARRAY] or [constant @GlobalScope.TYPE_VECTOR2]. Also note that for arrays, the size is also interpolated.
## [constant @GlobalScope.TYPE_STRING] and [constant @GlobalScope.TYPE_STRING_NAME] are interpolated between character codes and lengths, but note that there is a difference in algorithm between interpolation between keys and interpolation by blending.

#enum AnimationCallbackModeProcess
enum {
    ANIMATION_CALLBACK_MODE_PROCESS_PHYSICS = 0,
    ANIMATION_CALLBACK_MODE_PROCESS_IDLE = 1,
    ANIMATION_CALLBACK_MODE_PROCESS_MANUAL = 2,
}
#enum AnimationCallbackModeMethod
enum {
    ANIMATION_CALLBACK_MODE_METHOD_DEFERRED = 0,
    ANIMATION_CALLBACK_MODE_METHOD_IMMEDIATE = 1,
}
#enum AnimationCallbackModeDiscrete
enum {
    ANIMATION_CALLBACK_MODE_DISCRETE_DOMINANT = 0,
    ANIMATION_CALLBACK_MODE_DISCRETE_RECESSIVE = 1,
    ANIMATION_CALLBACK_MODE_DISCRETE_FORCE_CONTINUOUS = 2,
}
## If [code]true[/code], the [AnimationMixer] will be processing.
var active: bool:
	get = is_active, set = set_active

## The number of possible simultaneous sounds for each of the assigned AudioStreamPlayers.
## For example, if this value is [code]32[/code] and the animation has two audio tracks, the two [AudioStreamPlayer]s assigned can play simultaneously up to [code]32[/code] voices each.
var audio_max_polyphony: int:
	get = get_audio_max_polyphony, set = set_audio_max_polyphony

## Ordinarily, tracks can be set to [constant Animation.UPDATE_DISCRETE] to update infrequently, usually when using nearest interpolation.
## However, when blending with [constant Animation.UPDATE_CONTINUOUS] several results are considered. The [member callback_mode_discrete] specify it explicitly. See also [enum AnimationCallbackModeDiscrete].
## To make the blended results look good, it is recommended to set this to [constant ANIMATION_CALLBACK_MODE_DISCRETE_FORCE_CONTINUOUS] to update every frame during blending. Other values exist for compatibility and they are fine if there is no blending, but not so, may produce artifacts.
var callback_mode_discrete: int:
	get = get_callback_mode_discrete, set = set_callback_mode_discrete

## The call mode used for "Call Method" tracks.
var callback_mode_method: int:
	get = get_callback_mode_method, set = set_callback_mode_method

## The process notification in which to update animations.
var callback_mode_process: int:
	get = get_callback_mode_process, set = set_callback_mode_process

## If [code]true[/code], the blending uses the deterministic algorithm. The total weight is not normalized and the result is accumulated with an initial value ([code]0[/code] or a [code]"RESET"[/code] animation if present).
## This means that if the total amount of blending is [code]0.0[/code], the result is equal to the [code]"RESET"[/code] animation.
## If the number of tracks between the blended animations is different, the animation with the missing track is treated as if it had the initial value.
## If [code]false[/code], The blend does not use the deterministic algorithm. The total weight is normalized and always [code]1.0[/code]. If the number of tracks between the blended animations is different, nothing is done about the animation that is missing a track.
## [b]Note:[/b] In [AnimationTree], the blending with [AnimationNodeAdd2], [AnimationNodeAdd3], [AnimationNodeSub2] or the weight greater than [code]1.0[/code] may produce unexpected results.
## For example, if [AnimationNodeAdd2] blends two nodes with the amount [code]1.0[/code], then total weight is [code]2.0[/code] but it will be normalized to make the total amount [code]1.0[/code] and the result will be equal to [AnimationNodeBlend2] with the amount [code]0.5[/code].
var deterministic: bool:
	get = is_deterministic, set = set_deterministic

## This is used by the editor. If set to [code]true[/code], the scene will be saved with the effects of the reset animation (the animation with the key [code]"RESET"[/code]) applied as if it had been seeked to time 0, with the editor keeping the values that the scene had before saving.
## This makes it more convenient to preview and edit animations in the editor, as changes to the scene will not be saved as long as they are set in the reset animation.
var reset_on_save: bool:
	get = is_reset_on_save_enabled, set = set_reset_on_save_enabled

## If [code]true[/code], [method get_root_motion_position] value is extracted as a local translation value before blending. In other words, it is treated like the translation is done after the rotation.
var root_motion_local: bool:
	get = is_root_motion_local, set = set_root_motion_local

## The path to the Animation track used for root motion. Paths must be valid scene-tree paths to a node, and must be specified starting from the parent node of the node that will reproduce the animation. The [member root_motion_track] uses the same format as [method Animation.track_set_path], but note that a bone must be specified.
## If the track has type [constant Animation.TYPE_POSITION_3D], [constant Animation.TYPE_ROTATION_3D], or [constant Animation.TYPE_SCALE_3D] the transformation will be canceled visually, and the animation will appear to stay in place. See also [method get_root_motion_position], [method get_root_motion_rotation], [method get_root_motion_scale], and [RootMotionView].
var root_motion_track: NodePath:
	get = get_root_motion_track, set = set_root_motion_track

## The node which node path references will travel from.
var root_node: NodePath:
	get = get_root_node, set = set_root_node



## A virtual function for processing after getting a key during playback.
func _post_process_key_value(animation: Animation, track: int, value: Variant, object_id: int, object_sub_idx: int) -> Variant:
	pass;

## Adds [param library] to the animation player, under the key [param name].
## AnimationMixer has a global library by default with an empty string as key. For adding an animation to the global library:
## [codeblocks]
## [gdscript]
## var global_library = mixer.get_animation_library("")
## global_library.add_animation("animation_name", animation_resource)
## [/gdscript]
## [/codeblocks]
func add_animation_library(name: StringName, library: AnimationLibrary) -> int:
	pass;

## Manually advance the animations by the specified time (in seconds).
func advance(delta: float) -> void:
	pass;

## If the animation track specified by [param name] has an option [constant Animation.UPDATE_CAPTURE], stores current values of the objects indicated by the track path as a cache. If there is already a captured cache, the old cache is discarded.
## After this it will interpolate with current animation blending result during the playback process for the time specified by [param duration], working like a crossfade.
## You can specify [param trans_type] as the curve for the interpolation. For better results, it may be appropriate to specify [constant Tween.TRANS_LINEAR] for cases where the first key of the track begins with a non-zero value or where the key value does not change, and [constant Tween.TRANS_QUAD] for cases where the key value changes linearly.
func capture(name: StringName, duration: float, trans_type: int = 0, ease_type: int = 0) -> void:
	pass;

## [AnimationMixer] caches animated nodes. It may not notice if a node disappears; [method clear_caches] forces it to update the cache again.
func clear_caches() -> void:
	pass;

## Returns the key of [param animation] or an empty [StringName] if not found.
func find_animation(animation: Animation) -> StringName:
	pass;

## Returns the key for the [AnimationLibrary] that contains [param animation] or an empty [StringName] if not found.
func find_animation_library(animation: Animation) -> StringName:
	pass;

## Returns the [Animation] with the key [param name]. If the animation does not exist, [code]null[/code] is returned and an error is logged.
func get_animation(name: StringName) -> Animation:
	pass;

## Returns the first [AnimationLibrary] with key [param name] or [code]null[/code] if not found.
## To get the [AnimationMixer]'s global animation library, use [code]get_animation_library("")[/code].
func get_animation_library(name: StringName) -> AnimationLibrary:
	pass;

## Returns the list of stored library keys.
func get_animation_library_list() -> Array[StringName]:
	pass;

## Returns the list of stored animation keys.
func get_animation_list() -> PackedStringArray:
	pass;

## Retrieve the motion delta of position with the [member root_motion_track] as a [Vector3] that can be used elsewhere.
## If [member root_motion_track] is not a path to a track of type [constant Animation.TYPE_POSITION_3D], returns [code]Vector3(0, 0, 0)[/code].
## See also [member root_motion_track] and [RootMotionView].
## The most basic example is applying position to [CharacterBody3D]:
## [codeblocks]
## [gdscript]
## var current_rotation
## func _process(delta):
## if Input.is_action_just_pressed("animate"):
## current_rotation = get_quaternion()
## state_machine.travel("Animate")
## var velocity = current_rotation * animation_tree.get_root_motion_position() / delta
## set_velocity(velocity)
## move_and_slide()
## [/gdscript]
## [/codeblocks]
## By using this in combination with [method get_root_motion_rotation_accumulator], you can apply the root motion position more correctly to account for the rotation of the node.
## [codeblocks]
## [gdscript]
## func _process(delta):
## if Input.is_action_just_pressed("animate"):
## state_machine.travel("Animate")
## set_quaternion(get_quaternion() * animation_tree.get_root_motion_rotation())
## var velocity = (animation_tree.get_root_motion_rotation_accumulator().inverse() * get_quaternion()) * animation_tree.get_root_motion_position() / delta
## set_velocity(velocity)
## move_and_slide()
## [/gdscript]
## [/codeblocks]
## If [member root_motion_local] is [code]true[/code], return the pre-multiplied translation value with the inverted rotation.
## In this case, the code can be written as follows:
## [codeblocks]
## [gdscript]
## func _process(delta):
## if Input.is_action_just_pressed("animate"):
## state_machine.travel("Animate")
## set_quaternion(get_quaternion() * animation_tree.get_root_motion_rotation())
## var velocity = get_quaternion() * animation_tree.get_root_motion_position() / delta
## set_velocity(velocity)
## move_and_slide()
## [/gdscript]
## [/codeblocks]
func get_root_motion_position() -> Vector3:
	pass;

## Retrieve the blended value of the position tracks with the [member root_motion_track] as a [Vector3] that can be used elsewhere.
## This is useful in cases where you want to respect the initial key values of the animation.
## For example, if an animation with only one key [code]Vector3(0, 0, 0)[/code] is played in the previous frame and then an animation with only one key [code]Vector3(1, 0, 1)[/code] is played in the next frame, the difference can be calculated as follows:
## [codeblocks]
## [gdscript]
## var prev_root_motion_position_accumulator
## func _process(delta):
## if Input.is_action_just_pressed("animate"):
## state_machine.travel("Animate")
## var current_root_motion_position_accumulator = animation_tree.get_root_motion_position_accumulator()
## var difference = current_root_motion_position_accumulator - prev_root_motion_position_accumulator
## prev_root_motion_position_accumulator = current_root_motion_position_accumulator
## transform.origin += difference
## [/gdscript]
## [/codeblocks]
## However, if the animation loops, an unintended discrete change may occur, so this is only useful for some simple use cases.
func get_root_motion_position_accumulator() -> Vector3:
	pass;

## Retrieve the motion delta of rotation with the [member root_motion_track] as a [Quaternion] that can be used elsewhere.
## If [member root_motion_track] is not a path to a track of type [constant Animation.TYPE_ROTATION_3D], returns [code]Quaternion(0, 0, 0, 1)[/code].
## See also [member root_motion_track] and [RootMotionView].
## The most basic example is applying rotation to [CharacterBody3D]:
## [codeblocks]
## [gdscript]
## func _process(delta):
## if Input.is_action_just_pressed("animate"):
## state_machine.travel("Animate")
## set_quaternion(get_quaternion() * animation_tree.get_root_motion_rotation())
## [/gdscript]
## [/codeblocks]
func get_root_motion_rotation() -> Quaternion:
	pass;

## Retrieve the blended value of the rotation tracks with the [member root_motion_track] as a [Quaternion] that can be used elsewhere.
## This is necessary to apply the root motion position correctly, taking rotation into account. See also [method get_root_motion_position].
## Also, this is useful in cases where you want to respect the initial key values of the animation.
## For example, if an animation with only one key [code]Quaternion(0, 0, 0, 1)[/code] is played in the previous frame and then an animation with only one key [code]Quaternion(0, 0.707, 0, 0.707)[/code] is played in the next frame, the difference can be calculated as follows:
## [codeblocks]
## [gdscript]
## var prev_root_motion_rotation_accumulator
## func _process(delta):
## if Input.is_action_just_pressed("animate"):
## state_machine.travel("Animate")
## var current_root_motion_rotation_accumulator = animation_tree.get_root_motion_rotation_accumulator()
## var difference = prev_root_motion_rotation_accumulator.inverse() * current_root_motion_rotation_accumulator
## prev_root_motion_rotation_accumulator = current_root_motion_rotation_accumulator
## transform.basis *=  Basis(difference)
## [/gdscript]
## [/codeblocks]
## However, if the animation loops, an unintended discrete change may occur, so this is only useful for some simple use cases.
func get_root_motion_rotation_accumulator() -> Quaternion:
	pass;

## Retrieve the motion delta of scale with the [member root_motion_track] as a [Vector3] that can be used elsewhere.
## If [member root_motion_track] is not a path to a track of type [constant Animation.TYPE_SCALE_3D], returns [code]Vector3(0, 0, 0)[/code].
## See also [member root_motion_track] and [RootMotionView].
## The most basic example is applying scale to [CharacterBody3D]:
## [codeblocks]
## [gdscript]
## var current_scale = Vector3(1, 1, 1)
## var scale_accum = Vector3(1, 1, 1)
## func _process(delta):
## if Input.is_action_just_pressed("animate"):
## current_scale = get_scale()
## scale_accum = Vector3(1, 1, 1)
## state_machine.travel("Animate")
## scale_accum += animation_tree.get_root_motion_scale()
## set_scale(current_scale * scale_accum)
## [/gdscript]
## [/codeblocks]
func get_root_motion_scale() -> Vector3:
	pass;

## Retrieve the blended value of the scale tracks with the [member root_motion_track] as a [Vector3] that can be used elsewhere.
## For example, if an animation with only one key [code]Vector3(1, 1, 1)[/code] is played in the previous frame and then an animation with only one key [code]Vector3(2, 2, 2)[/code] is played in the next frame, the difference can be calculated as follows:
## [codeblocks]
## [gdscript]
## var prev_root_motion_scale_accumulator
## func _process(delta):
## if Input.is_action_just_pressed("animate"):
## state_machine.travel("Animate")
## var current_root_motion_scale_accumulator = animation_tree.get_root_motion_scale_accumulator()
## var difference = current_root_motion_scale_accumulator - prev_root_motion_scale_accumulator
## prev_root_motion_scale_accumulator = current_root_motion_scale_accumulator
## transform.basis = transform.basis.scaled(difference)
## [/gdscript]
## [/codeblocks]
## However, if the animation loops, an unintended discrete change may occur, so this is only useful for some simple use cases.
func get_root_motion_scale_accumulator() -> Vector3:
	pass;

## Returns [code]true[/code] if the [AnimationMixer] stores an [Animation] with key [param name].
func has_animation(name: StringName) -> bool:
	pass;

## Returns [code]true[/code] if the [AnimationMixer] stores an [AnimationLibrary] with key [param name].
func has_animation_library(name: StringName) -> bool:
	pass;

## Removes the [AnimationLibrary] associated with the key [param name].
func remove_animation_library(name: StringName) -> void:
	pass;

## Moves the [AnimationLibrary] associated with the key [param name] to the key [param newname].
func rename_animation_library(name: StringName, newname: StringName) -> void:
	pass;


func is_active() -> bool:
	return active

func set_active(value: bool) -> void:
	active = value

func get_audio_max_polyphony() -> int:
	return audio_max_polyphony

func set_audio_max_polyphony(value: int) -> void:
	audio_max_polyphony = value

func get_callback_mode_discrete() -> int:
	return callback_mode_discrete

func set_callback_mode_discrete(value: int) -> void:
	callback_mode_discrete = value

func get_callback_mode_method() -> int:
	return callback_mode_method

func set_callback_mode_method(value: int) -> void:
	callback_mode_method = value

func get_callback_mode_process() -> int:
	return callback_mode_process

func set_callback_mode_process(value: int) -> void:
	callback_mode_process = value

func is_deterministic() -> bool:
	return deterministic

func set_deterministic(value: bool) -> void:
	deterministic = value

func is_reset_on_save_enabled() -> bool:
	return reset_on_save

func set_reset_on_save_enabled(value: bool) -> void:
	reset_on_save = value

func is_root_motion_local() -> bool:
	return root_motion_local

func set_root_motion_local(value: bool) -> void:
	root_motion_local = value

func get_root_motion_track() -> NodePath:
	return root_motion_track

func set_root_motion_track(value: NodePath) -> void:
	root_motion_track = value

func get_root_node() -> NodePath:
	return root_node

func set_root_node(value: NodePath) -> void:
	root_node = value

