extends RefCounted
class_name XMLParser

## Provides a low-level interface for creating parsers for XML files.
##
## Provides a low-level interface for creating parsers for [url=https://en.wikipedia.org/wiki/XML]XML[/url] files. This class can serve as base to make custom XML parsers.
## To parse XML, you must open a file with the [method open] method or a buffer with the [method open_buffer] method. Then, the [method read] method must be called to parse the next nodes. Most of the methods take into consideration the currently parsed node.
## Here is an example of using [XMLParser] to parse an SVG file (which is based on XML), printing each element and its attributes as a dictionary:
## [codeblocks]
## [gdscript]
## var parser = XMLParser.new()
## parser.open("path/to/file.svg")
## while parser.read() != ERR_FILE_EOF:
## if parser.get_node_type() == XMLParser.NODE_ELEMENT:
## var node_name = parser.get_node_name()
## var attributes_dict = {}
## for idx in range(parser.get_attribute_count()):
## attributes_dict[parser.get_attribute_name(idx)] = parser.get_attribute_value(idx)
## print("The ", node_name, " element has the following attributes: ", attributes_dict)
## [/gdscript]
## [csharp]
## var parser = new XmlParser();
## parser.Open("path/to/file.svg");
## while (parser.Read() != Error.FileEof)
## {
## if (parser.GetNodeType() == XmlParser.NodeType.Element)
## {
## var nodeName = parser.GetNodeName();
## var attributesDict = new Godot.Collections.Dictionary();
## for (int idx = 0; idx < parser.GetAttributeCount(); idx++)
## {
## attributesDict[parser.GetAttributeName(idx)] = parser.GetAttributeValue(idx);
## }
## GD.Print($"The {nodeName} element has the following attributes: {attributesDict}");
## }
## }
## [/csharp]
## [/codeblocks]


## There's no node (no file or buffer opened).
## An element node type, also known as a tag, e.g. [code]<title>[/code].
## An end of element node type, e.g. [code]</title>[/code].
## A text node type, i.e. text that is not inside an element. This includes whitespace.
## A comment node type, e.g. [code]<!--A comment-->[/code].
## A node type for CDATA (Character Data) sections, e.g. [code]<![CDATA[CDATA section]]>[/code].
## An unknown node type.

#enum NodeType
enum {
    NODE_NONE = 0,
    NODE_ELEMENT = 1,
    NODE_ELEMENT_END = 2,
    NODE_TEXT = 3,
    NODE_COMMENT = 4,
    NODE_CDATA = 5,
    NODE_UNKNOWN = 6,
}


## Returns the number of attributes in the currently parsed element.
## [b]Note:[/b] If this method is used while the currently parsed node is not [constant NODE_ELEMENT] or [constant NODE_ELEMENT_END], this count will not be updated and will still reflect the last element.
func get_attribute_count() -> int:
	pass;

## Returns the name of an attribute of the currently parsed element, specified by the [param idx] index.
func get_attribute_name(idx: int) -> String:
	pass;

## Returns the value of an attribute of the currently parsed element, specified by the [param idx] index.
func get_attribute_value(idx: int) -> String:
	pass;

## Returns the current line in the parsed file, counting from 0.
func get_current_line() -> int:
	pass;

## Returns the value of an attribute of the currently parsed element, specified by its [param name]. This method will raise an error if the element has no such attribute.
func get_named_attribute_value(name: String) -> String:
	pass;

## Returns the value of an attribute of the currently parsed element, specified by its [param name]. This method will return an empty string if the element has no such attribute.
func get_named_attribute_value_safe(name: String) -> String:
	pass;

## Returns the contents of a text node. This method will raise an error if the current parsed node is of any other type.
func get_node_data() -> String:
	pass;

## Returns the name of a node. This method will raise an error if the currently parsed node is a text node.
## [b]Note:[/b] The content of a [constant NODE_CDATA] node and the comment string of a [constant NODE_COMMENT] node are also considered names.
func get_node_name() -> String:
	pass;

## Returns the byte offset of the currently parsed node since the beginning of the file or buffer. This is usually equivalent to the number of characters before the read position.
func get_node_offset() -> int:
	pass;

## Returns the type of the current node. Compare with [enum NodeType] constants.
func get_node_type() -> int:
	pass;

## Returns [code]true[/code] if the currently parsed element has an attribute with the [param name].
func has_attribute(name: String) -> bool:
	pass;

## Returns [code]true[/code] if the currently parsed element is empty, e.g. [code]<element />[/code].
func is_empty() -> bool:
	pass;

## Opens an XML [param file] for parsing. This method returns an error code.
func open(file: String) -> int:
	pass;

## Opens an XML raw [param buffer] for parsing. This method returns an error code.
func open_buffer(buffer: PackedByteArray) -> int:
	pass;

## Parses the next node in the file. This method returns an error code.
func read() -> int:
	pass;

## Moves the buffer cursor to a certain offset (since the beginning) and reads the next node there. This method returns an error code.
func seek(position: int) -> int:
	pass;

## Skips the current section. If the currently parsed node contains more inner nodes, they will be ignored and the cursor will go to the closing of the current element.
func skip_section() -> void:
	pass;


