class_name Transform2D

## A 2×3 matrix representing a 2D transformation.
##
## The [Transform2D] built-in [Variant] type is a 2×3 [url=https://en.wikipedia.org/wiki/Matrix_(mathematics)]matrix[/url] representing a transformation in 2D space. It contains three [Vector2] values: [member x], [member y], and [member origin]. Together, they can represent translation, rotation, scale, and skew.
## The [member x] and [member y] axes form a 2×2 matrix, known as the transform's [b]basis[/b]. The length of each axis ([method Vector2.length]) influences the transform's scale, while the direction of all axes influence the rotation. Usually, both axes are perpendicular to one another. However, when you rotate one axis individually, the transform becomes skewed. Applying a skewed transform to a 2D sprite will make the sprite appear distorted.
## For a general introduction, see the [url=$DOCS_URL/tutorials/math/matrices_and_transforms.html]Matrices and transforms[/url] tutorial.
## [b]Note:[/b] Unlike [Transform3D], there is no 2D equivalent to the [Basis] type. All mentions of "basis" refer to the [member x] and [member y] components of [Transform2D].
##
## @tutorial(Math documentation index): https://docs.godotengine.org/en/stable/tutorials/math/index.html
## @tutorial(Matrices and transforms): https://docs.godotengine.org/en/stable/tutorials/math/matrices_and_transforms.html
## @tutorial(Matrix Transform Demo): https://godotengine.org/asset-library/asset/2787
## @tutorial(2.5D Game Demo): https://godotengine.org/asset-library/asset/2783


## The identity [Transform2D]. A transform with no translation, no rotation, and its scale being [code]1[/code]. When multiplied by another [Variant] such as [Rect2] or another [Transform2D], no transformation occurs. This means that:
## - The [member x] points right ([constant Vector2.RIGHT]);
## - The [member y] points up ([constant Vector2.UP]).
## [codeblock]
## var transform = Transform2D.IDENTITY
## print("| X | Y | Origin")
## print("| %s | %s | %s" % [transform.x.x, transform.y.x, transform.origin.x])
## print("| %s | %s | %s" % [transform.x.y, transform.y.y, transform.origin.y])
## # Prints:
## # | X | Y | Origin
## # | 1 | 0 | 0
## # | 0 | 1 | 0
## [/codeblock]
## This is identical to creating [constructor Transform2D] without any parameters. This constant can be used to make your code clearer, and for consistency with C#.
const IDENTITY = Transform2D(1, 0, 0, 1, 0, 0);

## When any transform is multiplied by [constant FLIP_X], it negates all components of the [member x] axis (the X column).
## When [constant FLIP_X] is multiplied by any basis, it negates the [member Vector2.x] component of all axes (the X row).
const FLIP_X = Transform2D(-1, 0, 0, 1, 0, 0);

## When any transform is multiplied by [constant FLIP_Y], it negates all components of the [member y] axis (the Y column).
## When [constant FLIP_Y] is multiplied by any basis, it negates the [member Vector2.y] component of all axes (the Y row).
const FLIP_Y = Transform2D(1, 0, 0, -1, 0, 0);


## The translation offset of this transform, and the column [code]2[/code] of the matrix. In 2D space, this can be seen as the position.
var origin: Vector2

## The transform basis's X axis, and the column [code]0[/code] of the matrix. Combined with [member y], this represents the transform's rotation, scale, and skew.
## On the identity transform, this vector points right ([constant Vector2.RIGHT]).
var x: Vector2

## The transform basis's Y axis, and the column [code]1[/code] of the matrix. Combined with [member x], this represents the transform's rotation, scale, and skew.
## On the identity transform, this vector points up ([constant Vector2.UP]).
var y: Vector2


## Constructs a [Transform2D] identical to [constant IDENTITY].
func Transform2D() -> Transform2D:
	pass;

## Constructs a [Transform2D] as a copy of the given [Transform2D].
func Transform2D(from: Transform2D) -> Transform2D:
	pass;

## Constructs a [Transform2D] from a given angle (in radians) and position.
func Transform2D(rotation: float, position: Vector2) -> Transform2D:
	pass;

## Constructs a [Transform2D] from a given angle (in radians), scale, skew (in radians), and position.
func Transform2D(rotation: float, scale: Vector2, skew: float, position: Vector2) -> Transform2D:
	pass;

## Constructs a [Transform2D] from 3 [Vector2] values representing [member x], [member y], and the [member origin] (the three matrix columns).
func Transform2D(x_axis: Vector2, y_axis: Vector2, origin: Vector2) -> Transform2D:
	pass;


## Returns the inverted version of this transform. Unlike [method inverse], this method works with almost any basis, including non-uniform ones, but is slower. See also [method inverse].
## [b]Note:[/b] For this method to return correctly, the transform's basis needs to have a determinant that is not exactly [code]0[/code] (see [method determinant]).
func affine_inverse() -> Transform2D:
	pass;

## Returns a copy of the [param v] vector, transformed (multiplied) by the transform basis's matrix. Unlike the multiplication operator ([code]*[/code]), this method ignores the [member origin].
func basis_xform(v: Vector2) -> Vector2:
	pass;

## Returns a copy of the [param v] vector, transformed (multiplied) by the inverse transform basis's matrix (see [method inverse]). This method ignores the [member origin].
## [b]Note:[/b] This method assumes that this transform's basis is [i]orthonormal[/i] (see [method orthonormalized]). If the basis is not orthonormal, [code]transform.affine_inverse().basis_xform(vector)[/code] should be used instead (see [method affine_inverse]).
func basis_xform_inv(v: Vector2) -> Vector2:
	pass;

## Returns the [url=https://en.wikipedia.org/wiki/Determinant]determinant[/url] of this transform basis's matrix. For advanced math, this number can be used to determine a few attributes:
## - If the determinant is exactly [code]0[/code], the basis is not invertible (see [method inverse]).
## - If the determinant is a negative number, the basis represents a negative scale.
## [b]Note:[/b] If the basis's scale is the same for every axis, its determinant is always that scale by the power of 2.
func determinant() -> float:
	pass;

## Returns this transform's translation. Equivalent to [member origin].
func get_origin() -> Vector2:
	pass;

## Returns this transform's rotation (in radians). This is equivalent to [member x]'s angle (see [method Vector2.angle]).
func get_rotation() -> float:
	pass;

## Returns the length of both [member x] and [member y], as a [Vector2]. If this transform's basis is not skewed, this value is the scaling factor. It is not affected by rotation.
## [codeblocks]
## [gdscript]
## var my_transform = Transform2D(
## Vector2(2, 0),
## Vector2(0, 4),
## Vector2(0, 0)
## )
## # Rotating the Transform2D in any way preserves its scale.
## my_transform = my_transform.rotated(TAU / 2)
## print(my_transform.get_scale()) # Prints (2, 4).
## [/gdscript]
## [csharp]
## var myTransform = new Transform2D(
## Vector3(2.0f, 0.0f),
## Vector3(0.0f, 4.0f),
## Vector3(0.0f, 0.0f)
## );
## // Rotating the Transform2D in any way preserves its scale.
## myTransform = myTransform.Rotated(Mathf.Tau / 2.0f);
## GD.Print(myTransform.GetScale()); // Prints (2, 4, 8).
## [/csharp]
## [/codeblocks]
## [b]Note:[/b] If the value returned by [method determinant] is negative, the scale is also negative.
func get_scale() -> Vector2:
	pass;

## Returns this transform's skew (in radians).
func get_skew() -> float:
	pass;

## Returns the result of the linear interpolation between this transform and [param xform] by the given [param weight].
## The [param weight] should be between [code]0.0[/code] and [code]1.0[/code] (inclusive). Values outside this range are allowed and can be used to perform [i]extrapolation[/i] instead.
func interpolate_with(xform: Transform2D, weight: float) -> Transform2D:
	pass;

## Returns the [url=https://en.wikipedia.org/wiki/Invertible_matrix]inverted version of this transform[/url].
## [b]Note:[/b] For this method to return correctly, the transform's basis needs to be [i]orthonormal[/i] (see [method orthonormalized]). That means, the basis should only represent a rotation. If it does not, use [method affine_inverse] instead.
func inverse() -> Transform2D:
	pass;

## Returns [code]true[/code] if this transform's basis is conformal. A conformal basis is both [i]orthogonal[/i] (the axes are perpendicular to each other) and [i]uniform[/i] (the axes share the same length). This method can be especially useful during physics calculations.
func is_conformal() -> bool:
	pass;

## Returns [code]true[/code] if this transform and [param xform] are approximately equal, by running [method @GlobalScope.is_equal_approx] on each component.
func is_equal_approx(xform: Transform2D) -> bool:
	pass;

## Returns [code]true[/code] if this transform is finite, by calling [method @GlobalScope.is_finite] on each component.
func is_finite() -> bool:
	pass;

## Returns a copy of the transform rotated such that the rotated X-axis points towards the [param target] position, in global space.
func looking_at(target: Vector2 = Vector2(0, 0)) -> Transform2D:
	pass;

## Returns a copy of this transform with its basis orthonormalized. An orthonormal basis is both [i]orthogonal[/i] (the axes are perpendicular to each other) and [i]normalized[/i] (the axes have a length of [code]1[/code]), which also means it can only represent rotation.
func orthonormalized() -> Transform2D:
	pass;

## Returns a copy of the transform rotated by the given [param angle] (in radians).
## This method is an optimized version of multiplying the given transform [code]X[/code] with a corresponding rotation transform [code]R[/code] from the left, i.e., [code]R * X[/code].
## This can be seen as transforming with respect to the global/parent frame.
func rotated(angle: float) -> Transform2D:
	pass;

## Returns a copy of the transform rotated by the given [param angle] (in radians).
## This method is an optimized version of multiplying the given transform [code]X[/code] with a corresponding rotation transform [code]R[/code] from the right, i.e., [code]X * R[/code].
## This can be seen as transforming with respect to the local frame.
func rotated_local(angle: float) -> Transform2D:
	pass;

## Returns a copy of the transform scaled by the given [param scale] factor.
## This method is an optimized version of multiplying the given transform [code]X[/code] with a corresponding scaling transform [code]S[/code] from the left, i.e., [code]S * X[/code].
## This can be seen as transforming with respect to the global/parent frame.
func scaled(scale: Vector2) -> Transform2D:
	pass;

## Returns a copy of the transform scaled by the given [param scale] factor.
## This method is an optimized version of multiplying the given transform [code]X[/code] with a corresponding scaling transform [code]S[/code] from the right, i.e., [code]X * S[/code].
## This can be seen as transforming with respect to the local frame.
func scaled_local(scale: Vector2) -> Transform2D:
	pass;

## Returns a copy of the transform translated by the given [param offset].
## This method is an optimized version of multiplying the given transform [code]X[/code] with a corresponding translation transform [code]T[/code] from the left, i.e., [code]T * X[/code].
## This can be seen as transforming with respect to the global/parent frame.
func translated(offset: Vector2) -> Transform2D:
	pass;

## Returns a copy of the transform translated by the given [param offset].
## This method is an optimized version of multiplying the given transform [code]X[/code] with a corresponding translation transform [code]T[/code] from the right, i.e., [code]X * T[/code].
## This can be seen as transforming with respect to the local frame.
func translated_local(offset: Vector2) -> Transform2D:
	pass;


