extends RefCounted
class_name TextServer

## A server interface for font management and text rendering.
##
## [TextServer] is the API backend for managing fonts and rendering text.
## [b]Note:[/b] This is a low-level API, consider using [TextLine], [TextParagraph], and [Font] classes instead.
## This is an abstract class, so to get the currently active [TextServer] instance, use the following code:
## [codeblocks]
## [gdscript]
## var ts = TextServerManager.get_primary_interface()
## [/gdscript]
## [csharp]
## var ts = TextServerManager.GetPrimaryInterface();
## [/csharp]
## [/codeblocks]


## Font glyphs are rasterized as 1-bit bitmaps.
## Font glyphs are rasterized as 8-bit grayscale anti-aliased bitmaps.
## Font glyphs are rasterized for LCD screens.
## LCD subpixel layout is determined by the value of [code]gui/theme/lcd_subpixel_layout[/code] project settings.
## LCD subpixel anti-aliasing mode is suitable only for rendering horizontal, unscaled text in 2D.
## Unknown or unsupported subpixel layout, LCD subpixel antialiasing is disabled.
## Horizontal RGB subpixel layout.
## Horizontal BGR subpixel layout.
## Vertical RGB subpixel layout.
## Vertical BGR subpixel layout.
## Represents the size of the [enum FontLCDSubpixelLayout] enum.
## Text direction is determined based on contents and current locale.
## Text is written from left to right.
## Text is written from right to left.
## Text writing direction is the same as base string writing direction. Used for BiDi override only.
## Text is written horizontally.
## Left to right text is written vertically from top to bottom.
## Right to left text is written vertically from bottom to top.
## Do not justify text.
## Justify text by adding and removing kashidas.
## Justify text by changing width of the spaces between the words.
## Remove trailing and leading spaces from the justified text.
## Only apply justification to the part of the text after the last tab.
## Apply justification to the trimmed line with ellipsis.
## Do not apply justification to the last line of the paragraph.
## Do not apply justification to the last line of the paragraph with visible characters (takes precedence over [constant JUSTIFICATION_SKIP_LAST_LINE]).
## Always apply justification to the paragraphs with a single line ([constant JUSTIFICATION_SKIP_LAST_LINE] and [constant JUSTIFICATION_SKIP_LAST_LINE_WITH_VISIBLE_CHARS] are ignored).
## Autowrap is disabled.
## Wraps the text inside the node's bounding rectangle by allowing to break lines at arbitrary positions, which is useful when very limited space is available.
## Wraps the text inside the node's bounding rectangle by soft-breaking between words.
## Behaves similarly to [constant AUTOWRAP_WORD], but force-breaks a word if that single word does not fit in one line.
## Do not break the line.
## Break the line at the line mandatory break characters (e.g. [code]"\n"[/code]).
## Break the line between the words.
## Break the line between any unconnected graphemes.
## Should be used only in conjunction with [constant BREAK_WORD_BOUND], break the line between any unconnected graphemes, if it's impossible to break it between the words.
## Remove edge spaces from the broken line segments.
## Subtract first line indentation width from all lines after the first one.
## Trims text before the shaping. e.g, increasing [member Label.visible_characters] or [member RichTextLabel.visible_characters] value is visually identical to typing the text.
## Displays glyphs that are mapped to the first [member Label.visible_characters] or [member RichTextLabel.visible_characters] characters from the beginning of the text.
## Displays [member Label.visible_ratio] or [member RichTextLabel.visible_ratio] glyphs, starting from the left or from the right, depending on [member Control.layout_direction] value.
## Displays [member Label.visible_ratio] or [member RichTextLabel.visible_ratio] glyphs, starting from the left.
## Displays [member Label.visible_ratio] or [member RichTextLabel.visible_ratio] glyphs, starting from the right.
## No text trimming is performed.
## Trims the text per character.
## Trims the text per word.
## Trims the text per character and adds an ellipsis to indicate that parts are hidden.
## Trims the text per word and adds an ellipsis to indicate that parts are hidden.
## No trimming is performed.
## Trims the text when it exceeds the given width.
## Trims the text per word instead of per grapheme.
## Determines whether an ellipsis should be added at the end of the text.
## Determines whether the ellipsis at the end of the text is enforced and may not be hidden.
## Accounts for the text being justified before attempting to trim it (see [enum JustificationFlag]).
## Grapheme is supported by the font, and can be drawn.
## Grapheme is part of right-to-left or bottom-to-top run.
## Grapheme is not part of source text, it was added by justification process.
## Grapheme is whitespace.
## Grapheme is mandatory break point (e.g. [code]"\n"[/code]).
## Grapheme is optional break point (e.g. space).
## Grapheme is the tabulation character.
## Grapheme is kashida.
## Grapheme is punctuation character.
## Grapheme is underscore character.
## Grapheme is connected to the previous grapheme. Breaking line before this grapheme is not safe.
## It is safe to insert a U+0640 before this grapheme for elongation.
## Grapheme is an object replacement character for the embedded object.
## Grapheme is a soft hyphen.
## Disables font hinting (smoother but less crisp).
## Use the light font hinting mode.
## Use the default font hinting mode (crisper but less smooth).
## [b]Note:[/b] This hinting mode changes both horizontal and vertical glyph metrics. If applied to monospace font, some glyphs might have different width.
## Glyph horizontal position is rounded to the whole pixel size, each glyph is rasterized once.
## Glyph horizontal position is rounded based on font size.
## - To one quarter of the pixel size if font size is smaller or equal to [constant SUBPIXEL_POSITIONING_ONE_QUARTER_MAX_SIZE].
## - To one half of the pixel size if font size is smaller or equal to [constant SUBPIXEL_POSITIONING_ONE_HALF_MAX_SIZE].
## - To the whole pixel size for larger fonts.
## Glyph horizontal position is rounded to one half of the pixel size, each glyph is rasterized up to two times.
## Glyph horizontal position is rounded to one quarter of the pixel size, each glyph is rasterized up to four times.
## Maximum font size which will use one half of the pixel subpixel positioning in [constant SUBPIXEL_POSITIONING_AUTO] mode.
## Maximum font size which will use one quarter of the pixel subpixel positioning in [constant SUBPIXEL_POSITIONING_AUTO] mode.
## TextServer supports simple text layouts.
## TextServer supports bidirectional text layouts.
## TextServer supports vertical layouts.
## TextServer supports complex text shaping.
## TextServer supports justification using kashidas.
## TextServer supports complex line/word breaking rules (e.g. dictionary based).
## TextServer supports loading bitmap fonts.
## TextServer supports loading dynamic (TrueType, OpeType, etc.) fonts.
## TextServer supports multichannel signed distance field dynamic font rendering.
## TextServer supports loading system fonts.
## TextServer supports variable fonts.
## TextServer supports locale dependent and context sensitive case conversion.
## TextServer require external data file for some features, see [method load_support_data].
## TextServer supports UAX #31 identifier validation, see [method is_valid_identifier].
## TextServer supports [url=https://unicode.org/reports/tr36/]Unicode Technical Report #36[/url] and [url=https://unicode.org/reports/tr39/]Unicode Technical Standard #39[/url] based spoof detection features.
## Contour point is on the curve.
## Contour point isn't on the curve, but serves as a control point for a conic (quadratic) Bézier arc.
## Contour point isn't on the curve, but serves as a control point for a cubic Bézier arc.
## Spacing for each glyph.
## Spacing for the space character.
## Spacing at the top of the line.
## Spacing at the bottom of the line.
## Represents the size of the [enum SpacingType] enum.
## Font is bold.
## Font is italic or oblique.
## Font have fixed-width characters.
## Use default Unicode BiDi algorithm.
## BiDi override for URI.
## BiDi override for file path.
## BiDi override for email.
## BiDi override for lists. Structured text options: list separator [String].
## BiDi override for GDScript.
## User defined structured text BiDi override function.
## Bitmap font is not scaled.
## Bitmap font is scaled to the closest integer multiple of the font's fixed size. This is the recommended option for pixel art fonts.
## Bitmap font is scaled to an arbitrary (fractional) size. This is the recommended option for non-pixel art fonts.

#enum FontAntialiasing
enum {
    FONT_ANTIALIASING_NONE = 0,
    FONT_ANTIALIASING_GRAY = 1,
    FONT_ANTIALIASING_LCD = 2,
}
#enum FontLCDSubpixelLayout
enum {
    FONT_LCD_SUBPIXEL_LAYOUT_NONE = 0,
    FONT_LCD_SUBPIXEL_LAYOUT_HRGB = 1,
    FONT_LCD_SUBPIXEL_LAYOUT_HBGR = 2,
    FONT_LCD_SUBPIXEL_LAYOUT_VRGB = 3,
    FONT_LCD_SUBPIXEL_LAYOUT_VBGR = 4,
    FONT_LCD_SUBPIXEL_LAYOUT_MAX = 5,
}
#enum Direction
enum {
    DIRECTION_AUTO = 0,
    DIRECTION_LTR = 1,
    DIRECTION_RTL = 2,
    DIRECTION_INHERITED = 3,
}
#enum Orientation
enum {
    ORIENTATION_HORIZONTAL = 0,
    ORIENTATION_VERTICAL = 1,
}
#enum JustificationFlag
enum {
    JUSTIFICATION_NONE = 0,
    JUSTIFICATION_KASHIDA = 1,
    JUSTIFICATION_WORD_BOUND = 2,
    JUSTIFICATION_TRIM_EDGE_SPACES = 4,
    JUSTIFICATION_AFTER_LAST_TAB = 8,
    JUSTIFICATION_CONSTRAIN_ELLIPSIS = 16,
    JUSTIFICATION_SKIP_LAST_LINE = 32,
    JUSTIFICATION_SKIP_LAST_LINE_WITH_VISIBLE_CHARS = 64,
    JUSTIFICATION_DO_NOT_SKIP_SINGLE_LINE = 128,
}
#enum AutowrapMode
enum {
    AUTOWRAP_OFF = 0,
    AUTOWRAP_ARBITRARY = 1,
    AUTOWRAP_WORD = 2,
    AUTOWRAP_WORD_SMART = 3,
}
#enum LineBreakFlag
enum {
    BREAK_NONE = 0,
    BREAK_MANDATORY = 1,
    BREAK_WORD_BOUND = 2,
    BREAK_GRAPHEME_BOUND = 4,
    BREAK_ADAPTIVE = 8,
    BREAK_TRIM_EDGE_SPACES = 16,
    BREAK_TRIM_INDENT = 32,
}
#enum VisibleCharactersBehavior
enum {
    VC_CHARS_BEFORE_SHAPING = 0,
    VC_CHARS_AFTER_SHAPING = 1,
    VC_GLYPHS_AUTO = 2,
    VC_GLYPHS_LTR = 3,
    VC_GLYPHS_RTL = 4,
}
#enum OverrunBehavior
enum {
    OVERRUN_NO_TRIMMING = 0,
    OVERRUN_TRIM_CHAR = 1,
    OVERRUN_TRIM_WORD = 2,
    OVERRUN_TRIM_ELLIPSIS = 3,
    OVERRUN_TRIM_WORD_ELLIPSIS = 4,
}
#enum TextOverrunFlag
enum {
    OVERRUN_NO_TRIM = 0,
    OVERRUN_TRIM = 1,
    OVERRUN_TRIM_WORD_ONLY = 2,
    OVERRUN_ADD_ELLIPSIS = 4,
    OVERRUN_ENFORCE_ELLIPSIS = 8,
    OVERRUN_JUSTIFICATION_AWARE = 16,
}
#enum GraphemeFlag
enum {
    GRAPHEME_IS_VALID = 1,
    GRAPHEME_IS_RTL = 2,
    GRAPHEME_IS_VIRTUAL = 4,
    GRAPHEME_IS_SPACE = 8,
    GRAPHEME_IS_BREAK_HARD = 16,
    GRAPHEME_IS_BREAK_SOFT = 32,
    GRAPHEME_IS_TAB = 64,
    GRAPHEME_IS_ELONGATION = 128,
    GRAPHEME_IS_PUNCTUATION = 256,
    GRAPHEME_IS_UNDERSCORE = 512,
    GRAPHEME_IS_CONNECTED = 1024,
    GRAPHEME_IS_SAFE_TO_INSERT_TATWEEL = 2048,
    GRAPHEME_IS_EMBEDDED_OBJECT = 4096,
    GRAPHEME_IS_SOFT_HYPHEN = 8192,
}
#enum Hinting
enum {
    HINTING_NONE = 0,
    HINTING_LIGHT = 1,
    HINTING_NORMAL = 2,
}
#enum SubpixelPositioning
enum {
    SUBPIXEL_POSITIONING_DISABLED = 0,
    SUBPIXEL_POSITIONING_AUTO = 1,
    SUBPIXEL_POSITIONING_ONE_HALF = 2,
    SUBPIXEL_POSITIONING_ONE_QUARTER = 3,
    SUBPIXEL_POSITIONING_ONE_HALF_MAX_SIZE = 20,
    SUBPIXEL_POSITIONING_ONE_QUARTER_MAX_SIZE = 16,
}
#enum Feature
enum {
    FEATURE_SIMPLE_LAYOUT = 1,
    FEATURE_BIDI_LAYOUT = 2,
    FEATURE_VERTICAL_LAYOUT = 4,
    FEATURE_SHAPING = 8,
    FEATURE_KASHIDA_JUSTIFICATION = 16,
    FEATURE_BREAK_ITERATORS = 32,
    FEATURE_FONT_BITMAP = 64,
    FEATURE_FONT_DYNAMIC = 128,
    FEATURE_FONT_MSDF = 256,
    FEATURE_FONT_SYSTEM = 512,
    FEATURE_FONT_VARIABLE = 1024,
    FEATURE_CONTEXT_SENSITIVE_CASE_CONVERSION = 2048,
    FEATURE_USE_SUPPORT_DATA = 4096,
    FEATURE_UNICODE_IDENTIFIERS = 8192,
    FEATURE_UNICODE_SECURITY = 16384,
}
#enum ContourPointTag
enum {
    CONTOUR_CURVE_TAG_ON = 1,
    CONTOUR_CURVE_TAG_OFF_CONIC = 0,
    CONTOUR_CURVE_TAG_OFF_CUBIC = 2,
}
#enum SpacingType
enum {
    SPACING_GLYPH = 0,
    SPACING_SPACE = 1,
    SPACING_TOP = 2,
    SPACING_BOTTOM = 3,
    SPACING_MAX = 4,
}
#enum FontStyle
enum {
    FONT_BOLD = 1,
    FONT_ITALIC = 2,
    FONT_FIXED_WIDTH = 4,
}
#enum StructuredTextParser
enum {
    STRUCTURED_TEXT_DEFAULT = 0,
    STRUCTURED_TEXT_URI = 1,
    STRUCTURED_TEXT_FILE = 2,
    STRUCTURED_TEXT_EMAIL = 3,
    STRUCTURED_TEXT_LIST = 4,
    STRUCTURED_TEXT_GDSCRIPT = 5,
    STRUCTURED_TEXT_CUSTOM = 6,
}
#enum FixedSizeScaleMode
enum {
    FIXED_SIZE_SCALE_DISABLE = 0,
    FIXED_SIZE_SCALE_INTEGER_ONLY = 1,
    FIXED_SIZE_SCALE_ENABLED = 2,
}


## Creates a new, empty font cache entry resource. To free the resulting resource, use the [method free_rid] method.
func create_font() -> RID:
	pass;

## Creates a new variation existing font which is reusing the same glyph cache and font data. To free the resulting resource, use the [method free_rid] method.
func create_font_linked_variation(font_rid: RID) -> RID:
	pass;

## Creates a new buffer for complex text layout, with the given [param direction] and [param orientation]. To free the resulting buffer, use [method free_rid] method.
## [b]Note:[/b] Direction is ignored if server does not support [constant FEATURE_BIDI_LAYOUT] feature (supported by [TextServerAdvanced]).
## [b]Note:[/b] Orientation is ignored if server does not support [constant FEATURE_VERTICAL_LAYOUT] feature (supported by [TextServerAdvanced]).
func create_shaped_text(direction: int = 0, orientation: int = 0) -> RID:
	pass;

## Draws box displaying character hexadecimal code. Used for replacing missing characters.
func draw_hex_code_box(canvas: RID, size: int, pos: Vector2, index: int, color: Color) -> void:
	pass;

## Removes all rendered glyph information from the cache entry.
## [b]Note:[/b] This function will not remove textures associated with the glyphs, use [method font_remove_texture] to remove them manually.
func font_clear_glyphs(font_rid: RID, size: Vector2i) -> void:
	pass;

## Removes all kerning overrides.
func font_clear_kerning_map(font_rid: RID, size: int) -> void:
	pass;

## Removes all font sizes from the cache entry.
func font_clear_size_cache(font_rid: RID) -> void:
	pass;

## Removes all textures from font cache entry.
## [b]Note:[/b] This function will not remove glyphs associated with the texture, use [method font_remove_glyph] to remove them manually.
func font_clear_textures(font_rid: RID, size: Vector2i) -> void:
	pass;

## Draws single glyph into a canvas item at the position, using [param font_rid] at the size [param size].
## [b]Note:[/b] Glyph index is specific to the font, use glyphs indices returned by [method shaped_text_get_glyphs] or [method font_get_glyph_index].
## [b]Note:[/b] If there are pending glyphs to render, calling this function might trigger the texture cache update.
func font_draw_glyph(font_rid: RID, canvas: RID, size: int, pos: Vector2, index: int, color: Color = Color(1, 1, 1, 1)) -> void:
	pass;

## Draws single glyph outline of size [param outline_size] into a canvas item at the position, using [param font_rid] at the size [param size].
## [b]Note:[/b] Glyph index is specific to the font, use glyphs indices returned by [method shaped_text_get_glyphs] or [method font_get_glyph_index].
## [b]Note:[/b] If there are pending glyphs to render, calling this function might trigger the texture cache update.
func font_draw_glyph_outline(font_rid: RID, canvas: RID, size: int, outline_size: int, pos: Vector2, index: int, color: Color = Color(1, 1, 1, 1)) -> void:
	pass;

## Returns font anti-aliasing mode.
func font_get_antialiasing(font_rid: RID) -> int:
	pass;

## Returns the font ascent (number of pixels above the baseline).
func font_get_ascent(font_rid: RID, size: int) -> float:
	pass;

## Returns extra baseline offset (as a fraction of font height).
func font_get_baseline_offset(font_rid: RID) -> float:
	pass;

## Returns character code associated with [param glyph_index], or [code]0[/code] if [param glyph_index] is invalid. See [method font_get_glyph_index].
func font_get_char_from_glyph_index(font_rid: RID, size: int, glyph_index: int) -> int:
	pass;

## Returns the font descent (number of pixels below the baseline).
func font_get_descent(font_rid: RID, size: int) -> float:
	pass;

## Returns whether the font's embedded bitmap loading is disabled.
func font_get_disable_embedded_bitmaps(font_rid: RID) -> bool:
	pass;

## Returns font embolden strength.
func font_get_embolden(font_rid: RID) -> float:
	pass;

## Returns number of faces in the TrueType / OpenType collection.
func font_get_face_count(font_rid: RID) -> int:
	pass;

## Returns an active face index in the TrueType / OpenType collection.
func font_get_face_index(font_rid: RID) -> int:
	pass;

## Returns bitmap font fixed size.
func font_get_fixed_size(font_rid: RID) -> int:
	pass;

## Returns bitmap font scaling mode.
func font_get_fixed_size_scale_mode(font_rid: RID) -> int:
	pass;

## Returns [code]true[/code] if font texture mipmap generation is enabled.
func font_get_generate_mipmaps(font_rid: RID) -> bool:
	pass;

## Returns the font oversampling factor, shared by all fonts in the TextServer.
func font_get_global_oversampling() -> float:
	pass;

## Returns glyph advance (offset of the next glyph).
## [b]Note:[/b] Advance for glyphs outlines is the same as the base glyph advance and is not saved.
func font_get_glyph_advance(font_rid: RID, size: int, glyph: int) -> Vector2:
	pass;

## Returns outline contours of the glyph as a [Dictionary] with the following contents:
## [code]points[/code]         - [PackedVector3Array], containing outline points. [code]x[/code] and [code]y[/code] are point coordinates. [code]z[/code] is the type of the point, using the [enum ContourPointTag] values.
## [code]contours[/code]       - [PackedInt32Array], containing indices the end points of each contour.
## [code]orientation[/code]    - [bool], contour orientation. If [code]true[/code], clockwise contours must be filled.
func font_get_glyph_contours(font: RID, size: int, index: int) -> Dictionary:
	pass;

## Returns the glyph index of a [param char], optionally modified by the [param variation_selector]. See [method font_get_char_from_glyph_index].
func font_get_glyph_index(font_rid: RID, size: int, char: int, variation_selector: int) -> int:
	pass;

## Returns list of rendered glyphs in the cache entry.
func font_get_glyph_list(font_rid: RID, size: Vector2i) -> PackedInt32Array:
	pass;

## Returns glyph offset from the baseline.
func font_get_glyph_offset(font_rid: RID, size: Vector2i, glyph: int) -> Vector2:
	pass;

## Returns size of the glyph.
func font_get_glyph_size(font_rid: RID, size: Vector2i, glyph: int) -> Vector2:
	pass;

## Returns index of the cache texture containing the glyph.
func font_get_glyph_texture_idx(font_rid: RID, size: Vector2i, glyph: int) -> int:
	pass;

## Returns resource ID of the cache texture containing the glyph.
## [b]Note:[/b] If there are pending glyphs to render, calling this function might trigger the texture cache update.
func font_get_glyph_texture_rid(font_rid: RID, size: Vector2i, glyph: int) -> RID:
	pass;

## Returns size of the cache texture containing the glyph.
## [b]Note:[/b] If there are pending glyphs to render, calling this function might trigger the texture cache update.
func font_get_glyph_texture_size(font_rid: RID, size: Vector2i, glyph: int) -> Vector2:
	pass;

## Returns rectangle in the cache texture containing the glyph.
func font_get_glyph_uv_rect(font_rid: RID, size: Vector2i, glyph: int) -> Rect2:
	pass;

## Returns the font hinting mode. Used by dynamic fonts only.
func font_get_hinting(font_rid: RID) -> int:
	pass;

## Returns kerning for the pair of glyphs.
func font_get_kerning(font_rid: RID, size: int, glyph_pair: Vector2i) -> Vector2:
	pass;

## Returns list of the kerning overrides.
func font_get_kerning_list(font_rid: RID, size: int) -> Array[Vector2i]:
	pass;

## Returns [code]true[/code] if support override is enabled for the [param language].
func font_get_language_support_override(font_rid: RID, language: String) -> bool:
	pass;

## Returns list of language support overrides.
func font_get_language_support_overrides(font_rid: RID) -> PackedStringArray:
	pass;

## Returns the width of the range around the shape between the minimum and maximum representable signed distance.
func font_get_msdf_pixel_range(font_rid: RID) -> int:
	pass;

## Returns source font size used to generate MSDF textures.
func font_get_msdf_size(font_rid: RID) -> int:
	pass;

## Returns font family name.
func font_get_name(font_rid: RID) -> String:
	pass;

## Returns font OpenType feature set override.
func font_get_opentype_feature_overrides(font_rid: RID) -> Dictionary:
	pass;

## Returns [Dictionary] with OpenType font name strings (localized font names, version, description, license information, sample text, etc.).
func font_get_ot_name_strings(font_rid: RID) -> Dictionary:
	pass;

## Returns font oversampling factor, if set to [code]0.0[/code] global oversampling factor is used instead. Used by dynamic fonts only.
func font_get_oversampling(font_rid: RID) -> float:
	pass;

## Returns scaling factor of the color bitmap font.
func font_get_scale(font_rid: RID, size: int) -> float:
	pass;

## Returns [code]true[/code] if support override is enabled for the [param script].
func font_get_script_support_override(font_rid: RID, script: String) -> bool:
	pass;

## Returns list of script support overrides.
func font_get_script_support_overrides(font_rid: RID) -> PackedStringArray:
	pass;

## Returns list of the font sizes in the cache. Each size is [Vector2i] with font size and outline size.
func font_get_size_cache_list(font_rid: RID) -> Array[Vector2i]:
	pass;

## Returns the spacing for [param spacing] (see [enum TextServer.SpacingType]) in pixels (not relative to the font size).
func font_get_spacing(font_rid: RID, spacing: int) -> int:
	pass;

## Returns font stretch amount, compared to a normal width. A percentage value between [code]50%[/code] and [code]200%[/code].
func font_get_stretch(font_rid: RID) -> int:
	pass;

## Returns font style flags, see [enum FontStyle].
func font_get_style(font_rid: RID) -> int:
	pass;

## Returns font style name.
func font_get_style_name(font_rid: RID) -> String:
	pass;

## Returns font subpixel glyph positioning mode.
func font_get_subpixel_positioning(font_rid: RID) -> int:
	pass;

## Returns a string containing all the characters available in the font.
func font_get_supported_chars(font_rid: RID) -> String:
	pass;

## Returns number of textures used by font cache entry.
func font_get_texture_count(font_rid: RID, size: Vector2i) -> int:
	pass;

## Returns font cache texture image data.
func font_get_texture_image(font_rid: RID, size: Vector2i, texture_index: int) -> Image:
	pass;

## Returns array containing glyph packing data.
func font_get_texture_offsets(font_rid: RID, size: Vector2i, texture_index: int) -> PackedInt32Array:
	pass;

## Returns 2D transform applied to the font outlines.
func font_get_transform(font_rid: RID) -> Transform2D:
	pass;

## Returns pixel offset of the underline below the baseline.
func font_get_underline_position(font_rid: RID, size: int) -> float:
	pass;

## Returns thickness of the underline in pixels.
func font_get_underline_thickness(font_rid: RID, size: int) -> float:
	pass;

## Returns variation coordinates for the specified font cache entry. See [method font_supported_variation_list] for more info.
func font_get_variation_coordinates(font_rid: RID) -> Dictionary:
	pass;

## Returns weight (boldness) of the font. A value in the [code]100...999[/code] range, normal font weight is [code]400[/code], bold font weight is [code]700[/code].
func font_get_weight(font_rid: RID) -> int:
	pass;

## Returns [code]true[/code] if a Unicode [param char] is available in the font.
func font_has_char(font_rid: RID, char: int) -> bool:
	pass;

## Returns [code]true[/code] if system fonts can be automatically used as fallbacks.
func font_is_allow_system_fallback(font_rid: RID) -> bool:
	pass;

## Returns [code]true[/code] if auto-hinting is supported and preferred over font built-in hinting. Used by dynamic fonts only.
func font_is_force_autohinter(font_rid: RID) -> bool:
	pass;

## Returns [code]true[/code], if font supports given language ([url=https://en.wikipedia.org/wiki/ISO_639-1]ISO 639[/url] code).
func font_is_language_supported(font_rid: RID, language: String) -> bool:
	pass;

## Returns [code]true[/code] if glyphs of all sizes are rendered using single multichannel signed distance field generated from the dynamic font vector data.
func font_is_multichannel_signed_distance_field(font_rid: RID) -> bool:
	pass;

## Returns [code]true[/code], if font supports given script (ISO 15924 code).
func font_is_script_supported(font_rid: RID, script: String) -> bool:
	pass;

## Removes specified rendered glyph information from the cache entry.
## [b]Note:[/b] This function will not remove textures associated with the glyphs, use [method font_remove_texture] to remove them manually.
func font_remove_glyph(font_rid: RID, size: Vector2i, glyph: int) -> void:
	pass;

## Removes kerning override for the pair of glyphs.
func font_remove_kerning(font_rid: RID, size: int, glyph_pair: Vector2i) -> void:
	pass;

## Remove language support override.
func font_remove_language_support_override(font_rid: RID, language: String) -> void:
	pass;

## Removes script support override.
func font_remove_script_support_override(font_rid: RID, script: String) -> void:
	pass;

## Removes specified font size from the cache entry.
func font_remove_size_cache(font_rid: RID, size: Vector2i) -> void:
	pass;

## Removes specified texture from the cache entry.
## [b]Note:[/b] This function will not remove glyphs associated with the texture, remove them manually, using [method font_remove_glyph].
func font_remove_texture(font_rid: RID, size: Vector2i, texture_index: int) -> void:
	pass;

## Renders specified glyph to the font cache texture.
func font_render_glyph(font_rid: RID, size: Vector2i, index: int) -> void:
	pass;

## Renders the range of characters to the font cache texture.
func font_render_range(font_rid: RID, size: Vector2i, start: int, end: int) -> void:
	pass;

## If set to [code]true[/code], system fonts can be automatically used as fallbacks.
func font_set_allow_system_fallback(font_rid: RID, allow_system_fallback: bool) -> void:
	pass;

## Sets font anti-aliasing mode.
func font_set_antialiasing(font_rid: RID, antialiasing: int) -> void:
	pass;

## Sets the font ascent (number of pixels above the baseline).
func font_set_ascent(font_rid: RID, size: int, ascent: float) -> void:
	pass;

## Sets extra baseline offset (as a fraction of font height).
func font_set_baseline_offset(font_rid: RID, baseline_offset: float) -> void:
	pass;

## Sets font source data, e.g contents of the dynamic font source file.
func font_set_data(font_rid: RID, data: PackedByteArray) -> void:
	pass;

## Sets the font descent (number of pixels below the baseline).
func font_set_descent(font_rid: RID, size: int, descent: float) -> void:
	pass;

## If set to [code]true[/code], embedded font bitmap loading is disabled (bitmap-only and color fonts ignore this property).
func font_set_disable_embedded_bitmaps(font_rid: RID, disable_embedded_bitmaps: bool) -> void:
	pass;

## Sets font embolden strength. If [param strength] is not equal to zero, emboldens the font outlines. Negative values reduce the outline thickness.
func font_set_embolden(font_rid: RID, strength: float) -> void:
	pass;

## Sets an active face index in the TrueType / OpenType collection.
func font_set_face_index(font_rid: RID, face_index: int) -> void:
	pass;

## Sets bitmap font fixed size. If set to value greater than zero, same cache entry will be used for all font sizes.
func font_set_fixed_size(font_rid: RID, fixed_size: int) -> void:
	pass;

## Sets bitmap font scaling mode. This property is used only if [code]fixed_size[/code] is greater than zero.
func font_set_fixed_size_scale_mode(font_rid: RID, fixed_size_scale_mode: int) -> void:
	pass;

## If set to [code]true[/code] auto-hinting is preferred over font built-in hinting.
func font_set_force_autohinter(font_rid: RID, force_autohinter: bool) -> void:
	pass;

## If set to [code]true[/code] font texture mipmap generation is enabled.
func font_set_generate_mipmaps(font_rid: RID, generate_mipmaps: bool) -> void:
	pass;

## Sets oversampling factor, shared by all font in the TextServer.
## [b]Note:[/b] This value can be automatically changed by display server.
func font_set_global_oversampling(oversampling: float) -> void:
	pass;

## Sets glyph advance (offset of the next glyph).
## [b]Note:[/b] Advance for glyphs outlines is the same as the base glyph advance and is not saved.
func font_set_glyph_advance(font_rid: RID, size: int, glyph: int, advance: Vector2) -> void:
	pass;

## Sets glyph offset from the baseline.
func font_set_glyph_offset(font_rid: RID, size: Vector2i, glyph: int, offset: Vector2) -> void:
	pass;

## Sets size of the glyph.
func font_set_glyph_size(font_rid: RID, size: Vector2i, glyph: int, gl_size: Vector2) -> void:
	pass;

## Sets index of the cache texture containing the glyph.
func font_set_glyph_texture_idx(font_rid: RID, size: Vector2i, glyph: int, texture_idx: int) -> void:
	pass;

## Sets rectangle in the cache texture containing the glyph.
func font_set_glyph_uv_rect(font_rid: RID, size: Vector2i, glyph: int, uv_rect: Rect2) -> void:
	pass;

## Sets font hinting mode. Used by dynamic fonts only.
func font_set_hinting(font_rid: RID, hinting: int) -> void:
	pass;

## Sets kerning for the pair of glyphs.
func font_set_kerning(font_rid: RID, size: int, glyph_pair: Vector2i, kerning: Vector2) -> void:
	pass;

## Adds override for [method font_is_language_supported].
func font_set_language_support_override(font_rid: RID, language: String, supported: bool) -> void:
	pass;

## Sets the width of the range around the shape between the minimum and maximum representable signed distance.
func font_set_msdf_pixel_range(font_rid: RID, msdf_pixel_range: int) -> void:
	pass;

## Sets source font size used to generate MSDF textures.
func font_set_msdf_size(font_rid: RID, msdf_size: int) -> void:
	pass;

## If set to [code]true[/code], glyphs of all sizes are rendered using single multichannel signed distance field generated from the dynamic font vector data. MSDF rendering allows displaying the font at any scaling factor without blurriness, and without incurring a CPU cost when the font size changes (since the font no longer needs to be rasterized on the CPU). As a downside, font hinting is not available with MSDF. The lack of font hinting may result in less crisp and less readable fonts at small sizes.
## [b]Note:[/b] MSDF font rendering does not render glyphs with overlapping shapes correctly. Overlapping shapes are not valid per the OpenType standard, but are still commonly found in many font files, especially those converted by Google Fonts. To avoid issues with overlapping glyphs, consider downloading the font file directly from the type foundry instead of relying on Google Fonts.
func font_set_multichannel_signed_distance_field(font_rid: RID, msdf: bool) -> void:
	pass;

## Sets the font family name.
func font_set_name(font_rid: RID, name: String) -> void:
	pass;

## Sets font OpenType feature set override.
func font_set_opentype_feature_overrides(font_rid: RID, overrides: Dictionary) -> void:
	pass;

## Sets font oversampling factor, if set to [code]0.0[/code] global oversampling factor is used instead. Used by dynamic fonts only.
func font_set_oversampling(font_rid: RID, oversampling: float) -> void:
	pass;

## Sets scaling factor of the color bitmap font.
func font_set_scale(font_rid: RID, size: int, scale: float) -> void:
	pass;

## Adds override for [method font_is_script_supported].
func font_set_script_support_override(font_rid: RID, script: String, supported: bool) -> void:
	pass;

## Sets the spacing for [param spacing] (see [enum TextServer.SpacingType]) to [param value] in pixels (not relative to the font size).
func font_set_spacing(font_rid: RID, spacing: int, value: int) -> void:
	pass;

## Sets font stretch amount, compared to a normal width. A percentage value between [code]50%[/code] and [code]200%[/code].
## [b]Note:[/b] This value is used for font matching only and will not affect font rendering. Use [method font_set_face_index], [method font_set_variation_coordinates], or [method font_set_transform] instead.
func font_set_stretch(font_rid: RID, weight: int) -> void:
	pass;

## Sets the font style flags, see [enum FontStyle].
## [b]Note:[/b] This value is used for font matching only and will not affect font rendering. Use [method font_set_face_index], [method font_set_variation_coordinates], [method font_set_embolden], or [method font_set_transform] instead.
func font_set_style(font_rid: RID, style: int) -> void:
	pass;

## Sets the font style name.
func font_set_style_name(font_rid: RID, name: String) -> void:
	pass;

## Sets font subpixel glyph positioning mode.
func font_set_subpixel_positioning(font_rid: RID, subpixel_positioning: int) -> void:
	pass;

## Sets font cache texture image data.
func font_set_texture_image(font_rid: RID, size: Vector2i, texture_index: int, image: Image) -> void:
	pass;

## Sets array containing glyph packing data.
func font_set_texture_offsets(font_rid: RID, size: Vector2i, texture_index: int, offset: PackedInt32Array) -> void:
	pass;

## Sets 2D transform, applied to the font outlines, can be used for slanting, flipping, and rotating glyphs.
## For example, to simulate italic typeface by slanting, apply the following transform [code]Transform2D(1.0, slant, 0.0, 1.0, 0.0, 0.0)[/code].
func font_set_transform(font_rid: RID, transform: Transform2D) -> void:
	pass;

## Sets pixel offset of the underline below the baseline.
func font_set_underline_position(font_rid: RID, size: int, underline_position: float) -> void:
	pass;

## Sets thickness of the underline in pixels.
func font_set_underline_thickness(font_rid: RID, size: int, underline_thickness: float) -> void:
	pass;

## Sets variation coordinates for the specified font cache entry. See [method font_supported_variation_list] for more info.
func font_set_variation_coordinates(font_rid: RID, variation_coordinates: Dictionary) -> void:
	pass;

## Sets weight (boldness) of the font. A value in the [code]100...999[/code] range, normal font weight is [code]400[/code], bold font weight is [code]700[/code].
## [b]Note:[/b] This value is used for font matching only and will not affect font rendering. Use [method font_set_face_index], [method font_set_variation_coordinates], or [method font_set_embolden] instead.
func font_set_weight(font_rid: RID, weight: int) -> void:
	pass;

## Returns the dictionary of the supported OpenType features.
func font_supported_feature_list(font_rid: RID) -> Dictionary:
	pass;

## Returns the dictionary of the supported OpenType variation coordinates.
func font_supported_variation_list(font_rid: RID) -> Dictionary:
	pass;

## Converts a number from the Western Arabic (0..9) to the numeral systems used in [param language].
## If [param language] is omitted, the active locale will be used.
func format_number(number: String, language: String = "") -> String:
	pass;

## Frees an object created by this [TextServer].
func free_rid(rid: RID) -> void:
	pass;

## Returns text server features, see [enum Feature].
func get_features() -> int:
	pass;

## Returns size of the replacement character (box with character hexadecimal code that is drawn in place of invalid characters).
func get_hex_code_box_size(size: int, index: int) -> Vector2:
	pass;

## Returns the name of the server interface.
func get_name() -> String:
	pass;

## Returns default TextServer database (e.g. ICU break iterators and dictionaries) filename.
func get_support_data_filename() -> String:
	pass;

## Returns TextServer database (e.g. ICU break iterators and dictionaries) description.
func get_support_data_info() -> String:
	pass;

## Returns [code]true[/code] if [param rid] is valid resource owned by this text server.
func has(rid: RID) -> bool:
	pass;

## Returns [code]true[/code] if the server supports a feature.
func has_feature(feature: int) -> bool:
	pass;

## Returns index of the first string in [param dict] which is visually confusable with the [param string], or [code]-1[/code] if none is found.
## [b]Note:[/b] This method doesn't detect invisible characters, for spoof detection use it in combination with [method spoof_check].
## [b]Note:[/b] Always returns [code]-1[/code] if the server does not support the [constant FEATURE_UNICODE_SECURITY] feature.
func is_confusable(string: String, dict: PackedStringArray) -> int:
	pass;

## Returns [code]true[/code] if locale is right-to-left.
func is_locale_right_to_left(locale: String) -> bool:
	pass;

## Returns [code]true[/code] if [param string] is a valid identifier.
## If the text server supports the [constant FEATURE_UNICODE_IDENTIFIERS] feature, a valid identifier must:
## - Conform to normalization form C.
## - Begin with a Unicode character of class XID_Start or [code]"_"[/code].
## - May contain Unicode characters of class XID_Continue in the other positions.
## - Use UAX #31 recommended scripts only (mixed scripts are allowed).
## If the [constant FEATURE_UNICODE_IDENTIFIERS] feature is not supported, a valid identifier must:
## - Begin with a Unicode character of class XID_Start or [code]"_"[/code].
## - May contain Unicode characters of class XID_Continue in the other positions.
func is_valid_identifier(string: String) -> bool:
	pass;

## Returns [code]true[/code] if the given code point is a valid letter, i.e. it belongs to the Unicode category "L".
func is_valid_letter(unicode: int) -> bool:
	pass;

## Loads optional TextServer database (e.g. ICU break iterators and dictionaries).
## [b]Note:[/b] This function should be called before any other TextServer functions used, otherwise it won't have any effect.
func load_support_data(filename: String) -> bool:
	pass;

## Converts readable feature, variation, script, or language name to OpenType tag.
func name_to_tag(name: String) -> int:
	pass;

## Converts [param number] from the numeral systems used in [param language] to Western Arabic (0..9).
func parse_number(number: String, language: String = "") -> String:
	pass;

## Default implementation of the BiDi algorithm override function. See [enum StructuredTextParser] for more info.
func parse_structured_text(parser_type: int, args: Array, text: String) -> Array[Vector3i]:
	pass;

## Returns percent sign used in the [param language].
func percent_sign(language: String = "") -> String:
	pass;

## Saves optional TextServer database (e.g. ICU break iterators and dictionaries) to the file.
## [b]Note:[/b] This function is used by during project export, to include TextServer database.
func save_support_data(filename: String) -> bool:
	pass;

## Returns number of text spans added using [method shaped_text_add_string] or [method shaped_text_add_object].
func shaped_get_span_count(shaped: RID) -> int:
	pass;

## Returns text span metadata.
func shaped_get_span_meta(shaped: RID, index: int) -> Variant:
	pass;

## Changes text span font, font size, and OpenType features, without changing the text.
func shaped_set_span_update_font(shaped: RID, index: int, fonts: Array[RID], size: int, opentype_features: Dictionary = {}) -> void:
	pass;

## Adds inline object to the text buffer, [param key] must be unique. In the text, object is represented as [param length] object replacement characters.
func shaped_text_add_object(shaped: RID, key: Variant, size: Vector2, inline_align: int = 5, length: int = 1, baseline: float = 0.0) -> bool:
	pass;

## Adds text span and font to draw it to the text buffer.
func shaped_text_add_string(shaped: RID, text: String, fonts: Array[RID], size: int, opentype_features: Dictionary = {}, language: String = "", meta: Variant = null) -> bool:
	pass;

## Clears text buffer (removes text and inline objects).
func shaped_text_clear(rid: RID) -> void:
	pass;

## Returns composite character position closest to the [param pos].
func shaped_text_closest_character_pos(shaped: RID, pos: int) -> int:
	pass;

## Draw shaped text into a canvas item at a given position, with [param color]. [param pos] specifies the leftmost point of the baseline (for horizontal layout) or topmost point of the baseline (for vertical layout).
func shaped_text_draw(shaped: RID, canvas: RID, pos: Vector2, clip_l: float = -1, clip_r: float = -1, color: Color = Color(1, 1, 1, 1)) -> void:
	pass;

## Draw the outline of the shaped text into a canvas item at a given position, with [param color]. [param pos] specifies the leftmost point of the baseline (for horizontal layout) or topmost point of the baseline (for vertical layout).
func shaped_text_draw_outline(shaped: RID, canvas: RID, pos: Vector2, clip_l: float = -1, clip_r: float = -1, outline_size: int = 1, color: Color = Color(1, 1, 1, 1)) -> void:
	pass;

## Adjusts text width to fit to specified width, returns new text width.
func shaped_text_fit_to_width(shaped: RID, width: float, justification_flags: int = 3) -> float:
	pass;

## Returns the text ascent (number of pixels above the baseline for horizontal layout or to the left of baseline for vertical).
## [b]Note:[/b] Overall ascent can be higher than font ascent, if some glyphs are displaced from the baseline.
func shaped_text_get_ascent(shaped: RID) -> float:
	pass;

## Returns shapes of the carets corresponding to the character offset [param position] in the text. Returned caret shape is 1 pixel wide rectangle.
func shaped_text_get_carets(shaped: RID, position: int) -> Dictionary:
	pass;

## Returns array of the composite character boundaries.
func shaped_text_get_character_breaks(shaped: RID) -> PackedInt32Array:
	pass;

## Returns ellipsis character used for text clipping.
func shaped_text_get_custom_ellipsis(shaped: RID) -> int:
	pass;

## Returns custom punctuation character list, used for word breaking. If set to empty string, server defaults are used.
func shaped_text_get_custom_punctuation(shaped: RID) -> String:
	pass;

## Returns the text descent (number of pixels below the baseline for horizontal layout or to the right of baseline for vertical).
## [b]Note:[/b] Overall descent can be higher than font descent, if some glyphs are displaced from the baseline.
func shaped_text_get_descent(shaped: RID) -> float:
	pass;

## Returns direction of the text.
func shaped_text_get_direction(shaped: RID) -> int:
	pass;

## Returns dominant direction of in the range of text.
func shaped_text_get_dominant_direction_in_range(shaped: RID, start: int, end: int) -> int:
	pass;

## Returns number of glyphs in the ellipsis.
func shaped_text_get_ellipsis_glyph_count(shaped: RID) -> int:
	pass;

## Returns array of the glyphs in the ellipsis.
func shaped_text_get_ellipsis_glyphs(shaped: RID) -> Array[Dictionary]:
	pass;

## Returns position of the ellipsis.
func shaped_text_get_ellipsis_pos(shaped: RID) -> int:
	pass;

## Returns number of glyphs in the buffer.
func shaped_text_get_glyph_count(shaped: RID) -> int:
	pass;

## Returns an array of glyphs in the visual order.
func shaped_text_get_glyphs(shaped: RID) -> Array[Dictionary]:
	pass;

## Returns composite character's bounds as offsets from the start of the line.
func shaped_text_get_grapheme_bounds(shaped: RID, pos: int) -> Vector2:
	pass;

## Returns direction of the text, inferred by the BiDi algorithm.
func shaped_text_get_inferred_direction(shaped: RID) -> int:
	pass;

## Breaks text to the lines and returns character ranges for each line.
func shaped_text_get_line_breaks(shaped: RID, width: float, start: int = 0, break_flags: int = 3) -> PackedInt32Array:
	pass;

## Breaks text to the lines and columns. Returns character ranges for each segment.
func shaped_text_get_line_breaks_adv(shaped: RID, width: PackedFloat32Array, start: int = 0, once: bool = true, break_flags: int = 3) -> PackedInt32Array:
	pass;

## Returns the glyph index of the inline object.
func shaped_text_get_object_glyph(shaped: RID, key: Variant) -> int:
	pass;

## Returns the character range of the inline object.
func shaped_text_get_object_range(shaped: RID, key: Variant) -> Vector2i:
	pass;

## Returns bounding rectangle of the inline object.
func shaped_text_get_object_rect(shaped: RID, key: Variant) -> Rect2:
	pass;

## Returns array of inline objects.
func shaped_text_get_objects(shaped: RID) -> Array:
	pass;

## Returns text orientation.
func shaped_text_get_orientation(shaped: RID) -> int:
	pass;

## Returns the parent buffer from which the substring originates.
func shaped_text_get_parent(shaped: RID) -> RID:
	pass;

## Returns [code]true[/code] if text buffer is configured to display control characters.
func shaped_text_get_preserve_control(shaped: RID) -> bool:
	pass;

## Returns [code]true[/code] if text buffer is configured to display hexadecimal codes in place of invalid characters.
## [b]Note:[/b] If set to [code]false[/code], nothing is displayed in place of invalid characters.
func shaped_text_get_preserve_invalid(shaped: RID) -> bool:
	pass;

## Returns substring buffer character range in the parent buffer.
func shaped_text_get_range(shaped: RID) -> Vector2i:
	pass;

## Returns selection rectangles for the specified character range.
func shaped_text_get_selection(shaped: RID, start: int, end: int) -> PackedVector2Array:
	pass;

## Returns size of the text.
func shaped_text_get_size(shaped: RID) -> Vector2:
	pass;

## Returns extra spacing added between glyphs or lines in pixels.
func shaped_text_get_spacing(shaped: RID, spacing: int) -> int:
	pass;

## Returns the position of the overrun trim.
func shaped_text_get_trim_pos(shaped: RID) -> int:
	pass;

## Returns pixel offset of the underline below the baseline.
func shaped_text_get_underline_position(shaped: RID) -> float:
	pass;

## Returns thickness of the underline.
func shaped_text_get_underline_thickness(shaped: RID) -> float:
	pass;

## Returns width (for horizontal layout) or height (for vertical) of the text.
func shaped_text_get_width(shaped: RID) -> float:
	pass;

## Breaks text into words and returns array of character ranges. Use [param grapheme_flags] to set what characters are used for breaking (see [enum GraphemeFlag]).
func shaped_text_get_word_breaks(shaped: RID, grapheme_flags: int = 264, skip_grapheme_flags: int = 4) -> PackedInt32Array:
	pass;

## Returns [code]true[/code] if text buffer contains any visible characters.
func shaped_text_has_visible_chars(shaped: RID) -> bool:
	pass;

## Returns grapheme index at the specified pixel offset at the baseline, or [code]-1[/code] if none is found.
func shaped_text_hit_test_grapheme(shaped: RID, coords: float) -> int:
	pass;

## Returns caret character offset at the specified pixel offset at the baseline. This function always returns a valid position.
func shaped_text_hit_test_position(shaped: RID, coords: float) -> int:
	pass;

## Returns [code]true[/code] if buffer is successfully shaped.
func shaped_text_is_ready(shaped: RID) -> bool:
	pass;

## Returns composite character end position closest to the [param pos].
func shaped_text_next_character_pos(shaped: RID, pos: int) -> int:
	pass;

## Returns grapheme end position closest to the [param pos].
func shaped_text_next_grapheme_pos(shaped: RID, pos: int) -> int:
	pass;

## Trims text if it exceeds the given width.
func shaped_text_overrun_trim_to_width(shaped: RID, width: float = 0, overrun_trim_flags: int = 0) -> void:
	pass;

## Returns composite character start position closest to the [param pos].
func shaped_text_prev_character_pos(shaped: RID, pos: int) -> int:
	pass;

## Returns grapheme start position closest to the [param pos].
func shaped_text_prev_grapheme_pos(shaped: RID, pos: int) -> int:
	pass;

## Sets new size and alignment of embedded object.
func shaped_text_resize_object(shaped: RID, key: Variant, size: Vector2, inline_align: int = 5, baseline: float = 0.0) -> bool:
	pass;

## Overrides BiDi for the structured text.
## Override ranges should cover full source text without overlaps. BiDi algorithm will be used on each range separately.
func shaped_text_set_bidi_override(shaped: RID, override: Array) -> void:
	pass;

## Sets ellipsis character used for text clipping.
func shaped_text_set_custom_ellipsis(shaped: RID, char: int) -> void:
	pass;

## Sets custom punctuation character list, used for word breaking. If set to empty string, server defaults are used.
func shaped_text_set_custom_punctuation(shaped: RID, punct: String) -> void:
	pass;

## Sets desired text direction. If set to [constant DIRECTION_AUTO], direction will be detected based on the buffer contents and current locale.
## [b]Note:[/b] Direction is ignored if server does not support [constant FEATURE_BIDI_LAYOUT] feature (supported by [TextServerAdvanced]).
func shaped_text_set_direction(shaped: RID, direction: int = 0) -> void:
	pass;

## Sets desired text orientation.
## [b]Note:[/b] Orientation is ignored if server does not support [constant FEATURE_VERTICAL_LAYOUT] feature (supported by [TextServerAdvanced]).
func shaped_text_set_orientation(shaped: RID, orientation: int = 0) -> void:
	pass;

## If set to [code]true[/code] text buffer will display control characters.
func shaped_text_set_preserve_control(shaped: RID, enabled: bool) -> void:
	pass;

## If set to [code]true[/code] text buffer will display invalid characters as hexadecimal codes, otherwise nothing is displayed.
func shaped_text_set_preserve_invalid(shaped: RID, enabled: bool) -> void:
	pass;

## Sets extra spacing added between glyphs or lines in pixels.
func shaped_text_set_spacing(shaped: RID, spacing: int, value: int) -> void:
	pass;

## Shapes buffer if it's not shaped. Returns [code]true[/code] if the string is shaped successfully.
## [b]Note:[/b] It is not necessary to call this function manually, buffer will be shaped automatically as soon as any of its output data is requested.
func shaped_text_shape(shaped: RID) -> bool:
	pass;

## Returns text glyphs in the logical order.
func shaped_text_sort_logical(shaped: RID) -> Array[Dictionary]:
	pass;

## Returns text buffer for the substring of the text in the [param shaped] text buffer (including inline objects).
func shaped_text_substr(shaped: RID, start: int, length: int) -> RID:
	pass;

## Aligns shaped text to the given tab-stops.
func shaped_text_tab_align(shaped: RID, tab_stops: PackedFloat32Array) -> float:
	pass;

## Returns [code]true[/code] if [param string] is likely to be an attempt at confusing the reader.
## [b]Note:[/b] Always returns [code]false[/code] if the server does not support the [constant FEATURE_UNICODE_SECURITY] feature.
func spoof_check(string: String) -> bool:
	pass;

## Returns array of the composite character boundaries.
## [codeblock]
## var ts = TextServerManager.get_primary_interface()
## print(ts.string_get_word_breaks("Test ❤️‍🔥 Test")) # Prints [1, 2, 3, 4, 5, 9, 10, 11, 12, 13, 14]
## [/codeblock]
func string_get_character_breaks(string: String, language: String = "") -> PackedInt32Array:
	pass;

## Returns an array of the word break boundaries. Elements in the returned array are the offsets of the start and end of words. Therefore the length of the array is always even.
## When [param chars_per_line] is greater than zero, line break boundaries are returned instead.
## [codeblock]
## var ts = TextServerManager.get_primary_interface()
## print(ts.string_get_word_breaks("The Godot Engine, 4")) # Prints [0, 3, 4, 9, 10, 16, 18, 19], which corresponds to the following substrings: "The", "Godot", "Engine", "4"
## print(ts.string_get_word_breaks("The Godot Engine, 4", "en", 5)) # Prints [0, 3, 4, 9, 10, 15, 15, 19], which corresponds to the following substrings: "The", "Godot", "Engin", "e, 4"
## print(ts.string_get_word_breaks("The Godot Engine, 4", "en", 10)) # Prints [0, 9, 10, 19], which corresponds to the following substrings: "The Godot", "Engine, 4"
## [/codeblock]
func string_get_word_breaks(string: String, language: String = "", chars_per_line: int = 0) -> PackedInt32Array:
	pass;

## Returns the string converted to lowercase.
## [b]Note:[/b] Casing is locale dependent and context sensitive if server support [constant FEATURE_CONTEXT_SENSITIVE_CASE_CONVERSION] feature (supported by [TextServerAdvanced]).
## [b]Note:[/b] The result may be longer or shorter than the original.
func string_to_lower(string: String, language: String = "") -> String:
	pass;

## Returns the string converted to title case.
## [b]Note:[/b] Casing is locale dependent and context sensitive if server support [constant FEATURE_CONTEXT_SENSITIVE_CASE_CONVERSION] feature (supported by [TextServerAdvanced]).
## [b]Note:[/b] The result may be longer or shorter than the original.
func string_to_title(string: String, language: String = "") -> String:
	pass;

## Returns the string converted to uppercase.
## [b]Note:[/b] Casing is locale dependent and context sensitive if server support [constant FEATURE_CONTEXT_SENSITIVE_CASE_CONVERSION] feature (supported by [TextServerAdvanced]).
## [b]Note:[/b] The result may be longer or shorter than the original.
func string_to_upper(string: String, language: String = "") -> String:
	pass;

## Strips diacritics from the string.
## [b]Note:[/b] The result may be longer or shorter than the original.
func strip_diacritics(string: String) -> String:
	pass;

## Converts OpenType tag to readable feature, variation, script, or language name.
func tag_to_name(tag: int) -> String:
	pass;


