class_name Quaternion

## A unit quaternion used for representing 3D rotations.
##
## The [Quaternion] built-in [Variant] type is a 4D data structure that represents rotation in the form of a [url=https://en.wikipedia.org/wiki/Quaternions_and_spatial_rotation]Hamilton convention quaternion[/url]. Compared to the [Basis] type which can store both rotation and scale, quaternions can [i]only[/i] store rotation.
## A [Quaternion] is composed by 4 floating-point components: [member w], [member x], [member y], and [member z]. These components are very compact in memory, and because of this some operations are more efficient and less likely to cause floating-point errors. Methods such as [method get_angle], [method get_axis], and [method slerp] are faster than their [Basis] counterparts.
## For a great introduction to quaternions, see [url=https://www.youtube.com/watch?v=d4EgbgTm0Bg]this video by 3Blue1Brown[/url]. You do not need to know the math behind quaternions, as Godot provides several helper methods that handle it for you. These include [method slerp] and [method spherical_cubic_interpolate], as well as the [code]*[/code] operator.
## [b]Note:[/b] Quaternions must be normalized before being used for rotation (see [method normalized]).
## [b]Note:[/b] Similarly to [Vector2] and [Vector3], the components of a quaternion use 32-bit precision by default, unlike [float] which is always 64-bit. If double precision is needed, compile the engine with the option [code]precision=double[/code].
##
## @tutorial(3Blue1Brown's video on Quaternions): https://www.youtube.com/watch?v=d4EgbgTm0Bg
## @tutorial(Online Quaternion Visualization): https://quaternions.online/
## @tutorial(Using 3D transforms): https://docs.godotengine.org/en/stable/tutorials/3d/using_transforms.html#interpolating-with-quaternions
## @tutorial(Third Person Shooter (TPS) Demo): https://godotengine.org/asset-library/asset/2710
## @tutorial(Advanced Quaternion Visualization): https://iwatake2222.github.io/rotation_master/rotation_master.html


## The identity quaternion, representing no rotation. This has the same rotation as [constant Basis.IDENTITY].
## If a [Vector3] is rotated (multiplied) by this quaternion, it does not change.
const IDENTITY = Quaternion(0, 0, 0, 1);


## W component of the quaternion. This is the "real" part.
## [b]Note:[/b] Quaternion components should usually not be manipulated directly.
var w: float

## X component of the quaternion. This is the value along the "imaginary" [code]i[/code] axis.
## [b]Note:[/b] Quaternion components should usually not be manipulated directly.
var x: float

## Y component of the quaternion. This is the value along the "imaginary" [code]j[/code] axis.
## [b]Note:[/b] Quaternion components should usually not be manipulated directly.
var y: float

## Z component of the quaternion. This is the value along the "imaginary" [code]k[/code] axis.
## [b]Note:[/b] Quaternion components should usually not be manipulated directly.
var z: float


## Constructs a [Quaternion] identical to the [constant IDENTITY].
func Quaternion() -> Quaternion:
	pass;

## Constructs a [Quaternion] as a copy of the given [Quaternion].
func Quaternion(from: Quaternion) -> Quaternion:
	pass;

## Constructs a [Quaternion] representing the shortest arc between [param arc_from] and [param arc_to]. These can be imagined as two points intersecting a sphere's surface, with a radius of [code]1.0[/code].
func Quaternion(arc_from: Vector3, arc_to: Vector3) -> Quaternion:
	pass;

## Constructs a [Quaternion] representing rotation around the [param axis] by the given [param angle], in radians. The axis must be a normalized vector.
func Quaternion(axis: Vector3, angle: float) -> Quaternion:
	pass;

## Constructs a [Quaternion] from the given rotation [Basis].
## This constructor is faster than [method Basis.get_rotation_quaternion], but the given basis must be [i]orthonormalized[/i] (see [method Basis.orthonormalized]). Otherwise, the constructor fails and returns [constant IDENTITY].
func Quaternion(from: Basis) -> Quaternion:
	pass;

## Constructs a [Quaternion] defined by the given values.
## [b]Note:[/b] Only normalized quaternions represent rotation; if these values are not normalized, the new [Quaternion] will not be a valid rotation.
func Quaternion(x: float, y: float, z: float, w: float) -> Quaternion:
	pass;


## Returns the angle between this quaternion and [param to]. This is the magnitude of the angle you would need to rotate by to get from one to the other.
## [b]Note:[/b] The magnitude of the floating-point error for this method is abnormally high, so methods such as [code]is_zero_approx[/code] will not work reliably.
func angle_to(to: Quaternion) -> float:
	pass;

## Returns the dot product between this quaternion and [param with].
## This is equivalent to [code](quat.x * with.x) + (quat.y * with.y) + (quat.z * with.z) + (quat.w * with.w)[/code].
func dot(with: Quaternion) -> float:
	pass;

## Returns the exponential of this quaternion. The rotation axis of the result is the normalized rotation axis of this quaternion, the angle of the result is the length of the vector part of this quaternion.
func exp() -> Quaternion:
	pass;

## Constructs a new [Quaternion] from the given [Vector3] of [url=https://en.wikipedia.org/wiki/Euler_angles]Euler angles[/url], in radians. This method always uses the YXZ convention ([constant EULER_ORDER_YXZ]).
static func from_euler(euler: Vector3) -> Quaternion:
	pass;

## Returns the angle of the rotation represented by this quaternion.
## [b]Note:[/b] The quaternion must be normalized.
func get_angle() -> float:
	pass;

## Returns the rotation axis of the rotation represented by this quaternion.
func get_axis() -> Vector3:
	pass;

## Returns this quaternion's rotation as a [Vector3] of [url=https://en.wikipedia.org/wiki/Euler_angles]Euler angles[/url], in radians.
## The order of each consecutive rotation can be changed with [param order] (see [enum EulerOrder] constants). By default, the YXZ convention is used ([constant EULER_ORDER_YXZ]): Z (roll) is calculated first, then X (pitch), and lastly Y (yaw). When using the opposite method [method from_euler], this order is reversed.
func get_euler(order: int = 2) -> Vector3:
	pass;

## Returns the inverse version of this quaternion, inverting the sign of every component except [member w].
func inverse() -> Quaternion:
	pass;

## Returns [code]true[/code] if this quaternion and [param to] are approximately equal, by running [method @GlobalScope.is_equal_approx] on each component.
func is_equal_approx(to: Quaternion) -> bool:
	pass;

## Returns [code]true[/code] if this quaternion is finite, by calling [method @GlobalScope.is_finite] on each component.
func is_finite() -> bool:
	pass;

## Returns [code]true[/code] if this quaternion is normalized. See also [method normalized].
func is_normalized() -> bool:
	pass;

## Returns this quaternion's length, also called magnitude.
func length() -> float:
	pass;

## Returns this quaternion's length, squared.
## [b]Note:[/b] This method is faster than [method length], so prefer it if you only need to compare quaternion lengths.
func length_squared() -> float:
	pass;

## Returns the logarithm of this quaternion. Multiplies this quaternion's rotation axis by its rotation angle, and stores the result in the returned quaternion's vector part ([member x], [member y], and [member z]). The returned quaternion's real part ([member w]) is always [code]0.0[/code].
func log() -> Quaternion:
	pass;

## Returns a copy of this quaternion, normalized so that its length is [code]1.0[/code]. See also [method is_normalized].
func normalized() -> Quaternion:
	pass;

## Performs a spherical-linear interpolation with the [param to] quaternion, given a [param weight] and returns the result. Both this quaternion and [param to] must be normalized.
func slerp(to: Quaternion, weight: float) -> Quaternion:
	pass;

## Performs a spherical-linear interpolation with the [param to] quaternion, given a [param weight] and returns the result. Unlike [method slerp], this method does not check if the rotation path is smaller than 90 degrees. Both this quaternion and [param to] must be normalized.
func slerpni(to: Quaternion, weight: float) -> Quaternion:
	pass;

## Performs a spherical cubic interpolation between quaternions [param pre_a], this vector, [param b], and [param post_b], by the given amount [param weight].
func spherical_cubic_interpolate(b: Quaternion, pre_a: Quaternion, post_b: Quaternion, weight: float) -> Quaternion:
	pass;

## Performs a spherical cubic interpolation between quaternions [param pre_a], this vector, [param b], and [param post_b], by the given amount [param weight].
## It can perform smoother interpolation than [method spherical_cubic_interpolate] by the time values.
func spherical_cubic_interpolate_in_time(b: Quaternion, pre_a: Quaternion, post_b: Quaternion, weight: float, b_t: float, pre_a_t: float, post_b_t: float) -> Quaternion:
	pass;


