extends Object
class_name NavigationServer3D

## A server interface for low-level 3D navigation access.
##
## NavigationServer3D is the server that handles navigation maps, regions and agents. It does not handle A* navigation from [AStar3D].
## Maps are divided into regions, which are composed of navigation meshes. Together, they define the navigable areas in the 3D world.
## [b]Note:[/b] Most [NavigationServer3D] changes take effect after the next physics frame and not immediately. This includes all changes made to maps, regions or agents by navigation-related nodes in the scene tree or made through scripts.
## For two regions to be connected to each other, they must share a similar edge. An edge is considered connected to another if both of its two vertices are at a distance less than [code]edge_connection_margin[/code] to the respective other edge's vertex.
## You may assign navigation layers to regions with [method NavigationServer3D.region_set_navigation_layers], which then can be checked upon when requesting a path with [method NavigationServer3D.map_get_path]. This can be used to allow or deny certain areas for some objects.
## To use the collision avoidance system, you may use agents. You can set an agent's target velocity, then the servers will emit a callback with a modified velocity.
## [b]Note:[/b] The collision avoidance system ignores regions. Using the modified velocity directly may move an agent outside of the traversable area. This is a limitation of the collision avoidance system, any more complex situation may require the use of the physics engine.
## This server keeps tracks of any call and executes them during the sync phase. This means that you can request any change to the map, using any thread, without worrying.
##
## @tutorial(Using NavigationServer): https://docs.godotengine.org/en/stable/tutorials/navigation/navigation_using_navigationservers.html
## @tutorial(3D Navigation Demo): https://godotengine.org/asset-library/asset/2743


## Emitted when avoidance debug settings are changed. Only available in debug builds.
signal avoidance_debug_changed
## Emitted when a navigation map is updated, when a region moves or is modified.
signal map_changed(map: RID)
## Emitted when navigation debug settings are changed. Only available in debug builds.
signal navigation_debug_changed
## Constant to get the number of active navigation maps.
## Constant to get the number of active navigation regions.
## Constant to get the number of active navigation agents processing avoidance.
## Constant to get the number of active navigation links.
## Constant to get the number of navigation mesh polygons.
## Constant to get the number of navigation mesh polygon edges.
## Constant to get the number of navigation mesh polygon edges that were merged due to edge key overlap.
## Constant to get the number of navigation mesh polygon edges that are considered connected by edge proximity.
## Constant to get the number of navigation mesh polygon edges that could not be merged but may be still connected by edge proximity or with links.

#enum ProcessInfo
enum {
    INFO_ACTIVE_MAPS = 0,
    INFO_REGION_COUNT = 1,
    INFO_AGENT_COUNT = 2,
    INFO_LINK_COUNT = 3,
    INFO_POLYGON_COUNT = 4,
    INFO_EDGE_COUNT = 5,
    INFO_EDGE_MERGE_COUNT = 6,
    INFO_EDGE_CONNECTION_COUNT = 7,
    INFO_EDGE_FREE_COUNT = 8,
}


## Creates the agent.
func agent_create() -> RID:
	pass;

## Returns [code]true[/code] if the provided [param agent] has avoidance enabled.
func agent_get_avoidance_enabled(agent: RID) -> bool:
	pass;

## Returns the [code]avoidance_layers[/code] bitmask of the specified [param agent].
func agent_get_avoidance_layers(agent: RID) -> int:
	pass;

## Returns the [code]avoidance_mask[/code] bitmask of the specified [param agent].
func agent_get_avoidance_mask(agent: RID) -> int:
	pass;

## Returns the [code]avoidance_priority[/code] of the specified [param agent].
func agent_get_avoidance_priority(agent: RID) -> float:
	pass;

## Returns the [code]height[/code] of the specified [param agent].
func agent_get_height(agent: RID) -> float:
	pass;

## Returns the navigation map [RID] the requested [param agent] is currently assigned to.
func agent_get_map(agent: RID) -> RID:
	pass;

## Returns the maximum number of other agents the specified [param agent] takes into account in the navigation.
func agent_get_max_neighbors(agent: RID) -> int:
	pass;

## Returns the maximum speed of the specified [param agent].
func agent_get_max_speed(agent: RID) -> float:
	pass;

## Returns the maximum distance to other agents the specified [param agent] takes into account in the navigation.
func agent_get_neighbor_distance(agent: RID) -> float:
	pass;

## Returns [code]true[/code] if the specified [param agent] is paused.
func agent_get_paused(agent: RID) -> bool:
	pass;

## Returns the position of the specified [param agent] in world space.
func agent_get_position(agent: RID) -> Vector3:
	pass;

## Returns the radius of the specified [param agent].
func agent_get_radius(agent: RID) -> float:
	pass;

## Returns the minimal amount of time for which the specified [param agent]'s velocities that are computed by the simulation are safe with respect to other agents.
func agent_get_time_horizon_agents(agent: RID) -> float:
	pass;

## Returns the minimal amount of time for which the specified [param agent]'s velocities that are computed by the simulation are safe with respect to static avoidance obstacles.
func agent_get_time_horizon_obstacles(agent: RID) -> float:
	pass;

## Returns [code]true[/code] if the provided [param agent] uses avoidance in 3D space Vector3(x,y,z) instead of horizontal 2D Vector2(x,y) / Vector3(x,0.0,z).
func agent_get_use_3d_avoidance(agent: RID) -> bool:
	pass;

## Returns the velocity of the specified [param agent].
func agent_get_velocity(agent: RID) -> Vector3:
	pass;

## Return [code]true[/code] if the specified [param agent] has an avoidance callback.
func agent_has_avoidance_callback(agent: RID) -> bool:
	pass;

## Returns true if the map got changed the previous frame.
func agent_is_map_changed(agent: RID) -> bool:
	pass;

## Sets the callback [Callable] that gets called after each avoidance processing step for the [param agent]. The calculated [code]safe_velocity[/code] will be dispatched with a signal to the object just before the physics calculations.
## [b]Note:[/b] Created callbacks are always processed independently of the SceneTree state as long as the agent is on a navigation map and not freed. To disable the dispatch of a callback from an agent use [method agent_set_avoidance_callback] again with an empty [Callable].
func agent_set_avoidance_callback(agent: RID, callback: Callable) -> void:
	pass;

## If [param enabled] is [code]true[/code], the provided [param agent] calculates avoidance.
func agent_set_avoidance_enabled(agent: RID, enabled: bool) -> void:
	pass;

## Set the agent's [code]avoidance_layers[/code] bitmask.
func agent_set_avoidance_layers(agent: RID, layers: int) -> void:
	pass;

## Set the agent's [code]avoidance_mask[/code] bitmask.
func agent_set_avoidance_mask(agent: RID, mask: int) -> void:
	pass;

## Set the agent's [code]avoidance_priority[/code] with a [param priority] between 0.0 (lowest priority) to 1.0 (highest priority).
## The specified [param agent] does not adjust the velocity for other agents that would match the [code]avoidance_mask[/code] but have a lower [code]avoidance_priority[/code]. This in turn makes the other agents with lower priority adjust their velocities even more to avoid collision with this agent.
func agent_set_avoidance_priority(agent: RID, priority: float) -> void:
	pass;

## Updates the provided [param agent] [param height].
func agent_set_height(agent: RID, height: float) -> void:
	pass;

## Puts the agent in the map.
func agent_set_map(agent: RID, map: RID) -> void:
	pass;

## Sets the maximum number of other agents the agent takes into account in the navigation. The larger this number, the longer the running time of the simulation. If the number is too low, the simulation will not be safe.
func agent_set_max_neighbors(agent: RID, count: int) -> void:
	pass;

## Sets the maximum speed of the agent. Must be positive.
func agent_set_max_speed(agent: RID, max_speed: float) -> void:
	pass;

## Sets the maximum distance to other agents this agent takes into account in the navigation. The larger this number, the longer the running time of the simulation. If the number is too low, the simulation will not be safe.
func agent_set_neighbor_distance(agent: RID, distance: float) -> void:
	pass;

## If [param paused] is true the specified [param agent] will not be processed, e.g. calculate avoidance velocities or receive avoidance callbacks.
func agent_set_paused(agent: RID, paused: bool) -> void:
	pass;

## Sets the position of the agent in world space.
func agent_set_position(agent: RID, position: Vector3) -> void:
	pass;

## Sets the radius of the agent.
func agent_set_radius(agent: RID, radius: float) -> void:
	pass;

## The minimal amount of time for which the agent's velocities that are computed by the simulation are safe with respect to other agents. The larger this number, the sooner this agent will respond to the presence of other agents, but the less freedom this agent has in choosing its velocities. A too high value will slow down agents movement considerably. Must be positive.
func agent_set_time_horizon_agents(agent: RID, time_horizon: float) -> void:
	pass;

## The minimal amount of time for which the agent's velocities that are computed by the simulation are safe with respect to static avoidance obstacles. The larger this number, the sooner this agent will respond to the presence of static avoidance obstacles, but the less freedom this agent has in choosing its velocities. A too high value will slow down agents movement considerably. Must be positive.
func agent_set_time_horizon_obstacles(agent: RID, time_horizon: float) -> void:
	pass;

## Sets if the agent uses the 2D avoidance or the 3D avoidance while avoidance is enabled.
## If [code]true[/code] the agent calculates avoidance velocities in 3D for the xyz-axis, e.g. for games that take place in air, underwater or space. The 3D using agent only avoids other 3D avoidance using agent's. The 3D using agent only reacts to radius based avoidance obstacles. The 3D using agent ignores any vertices based obstacles. The 3D using agent only avoids other 3D using agent's.
## If [code]false[/code] the agent calculates avoidance velocities in 2D along the xz-axis ignoring the y-axis. The 2D using agent only avoids other 2D avoidance using agent's. The 2D using agent reacts to radius avoidance obstacles. The 2D using agent reacts to vertices based avoidance obstacles. The 2D using agent only avoids other 2D using agent's. 2D using agents will ignore other 2D using agents or obstacles that are below their current position or above their current position including the agents height in 2D avoidance.
func agent_set_use_3d_avoidance(agent: RID, enabled: bool) -> void:
	pass;

## Sets [param velocity] as the new wanted velocity for the specified [param agent]. The avoidance simulation will try to fulfill this velocity if possible but will modify it to avoid collision with other agent's and obstacles. When an agent is teleported to a new position use [method agent_set_velocity_forced] as well to reset the internal simulation velocity.
func agent_set_velocity(agent: RID, velocity: Vector3) -> void:
	pass;

## Replaces the internal velocity in the collision avoidance simulation with [param velocity] for the specified [param agent]. When an agent is teleported to a new position this function should be used in the same frame. If called frequently this function can get agents stuck.
func agent_set_velocity_forced(agent: RID, velocity: Vector3) -> void:
	pass;

## Bakes the provided [param navigation_mesh] with the data from the provided [param source_geometry_data]. After the process is finished the optional [param callback] will be called.
func bake_from_source_geometry_data(navigation_mesh: NavigationMesh, source_geometry_data: NavigationMeshSourceGeometryData3D, callback: Callable = Callable()) -> void:
	pass;

## Bakes the provided [param navigation_mesh] with the data from the provided [param source_geometry_data] as an async task running on a background thread. After the process is finished the optional [param callback] will be called.
func bake_from_source_geometry_data_async(navigation_mesh: NavigationMesh, source_geometry_data: NavigationMeshSourceGeometryData3D, callback: Callable = Callable()) -> void:
	pass;

## Destroys the given RID.
func free_rid(rid: RID) -> void:
	pass;

## Returns [code]true[/code] when the NavigationServer has debug enabled.
func get_debug_enabled() -> bool:
	pass;

## Returns all created navigation map [RID]s on the NavigationServer. This returns both 2D and 3D created navigation maps as there is technically no distinction between them.
func get_maps() -> Array[RID]:
	pass;

## Returns information about the current state of the NavigationServer. See [enum ProcessInfo] for a list of available states.
func get_process_info(process_info: int) -> int:
	pass;

## Returns [code]true[/code] when the provided navigation mesh is being baked on a background thread.
func is_baking_navigation_mesh(navigation_mesh: NavigationMesh) -> bool:
	pass;

## Create a new link between two positions on a map.
func link_create() -> RID:
	pass;

## Returns [code]true[/code] if the specified [param link] is enabled.
func link_get_enabled(link: RID) -> bool:
	pass;

## Returns the ending position of this [param link].
func link_get_end_position(link: RID) -> Vector3:
	pass;

## Returns the enter cost of this [param link].
func link_get_enter_cost(link: RID) -> float:
	pass;

## Returns the navigation map [RID] the requested [param link] is currently assigned to.
func link_get_map(link: RID) -> RID:
	pass;

## Returns the navigation layers for this [param link].
func link_get_navigation_layers(link: RID) -> int:
	pass;

## Returns the [code]ObjectID[/code] of the object which manages this link.
func link_get_owner_id(link: RID) -> int:
	pass;

## Returns the starting position of this [param link].
func link_get_start_position(link: RID) -> Vector3:
	pass;

## Returns the travel cost of this [param link].
func link_get_travel_cost(link: RID) -> float:
	pass;

## Returns whether this [param link] can be travelled in both directions.
func link_is_bidirectional(link: RID) -> bool:
	pass;

## Sets whether this [param link] can be travelled in both directions.
func link_set_bidirectional(link: RID, bidirectional: bool) -> void:
	pass;

## If [param enabled] is [code]true[/code], the specified [param link] will contribute to its current navigation map.
func link_set_enabled(link: RID, enabled: bool) -> void:
	pass;

## Sets the exit position for the [param link].
func link_set_end_position(link: RID, position: Vector3) -> void:
	pass;

## Sets the [param enter_cost] for this [param link].
func link_set_enter_cost(link: RID, enter_cost: float) -> void:
	pass;

## Sets the navigation map [RID] for the link.
func link_set_map(link: RID, map: RID) -> void:
	pass;

## Set the links's navigation layers. This allows selecting links from a path request (when using [method NavigationServer3D.map_get_path]).
func link_set_navigation_layers(link: RID, navigation_layers: int) -> void:
	pass;

## Set the [code]ObjectID[/code] of the object which manages this link.
func link_set_owner_id(link: RID, owner_id: int) -> void:
	pass;

## Sets the entry position for this [param link].
func link_set_start_position(link: RID, position: Vector3) -> void:
	pass;

## Sets the [param travel_cost] for this [param link].
func link_set_travel_cost(link: RID, travel_cost: float) -> void:
	pass;

## Create a new map.
func map_create() -> RID:
	pass;

## This function immediately forces synchronization of the specified navigation [param map] [RID]. By default navigation maps are only synchronized at the end of each physics frame. This function can be used to immediately (re)calculate all the navigation meshes and region connections of the navigation map. This makes it possible to query a navigation path for a changed map immediately and in the same frame (multiple times if needed).
## Due to technical restrictions the current NavigationServer command queue will be flushed. This means all already queued update commands for this physics frame will be executed, even those intended for other maps, regions and agents not part of the specified map. The expensive computation of the navigation meshes and region connections of a map will only be done for the specified map. Other maps will receive the normal synchronization at the end of the physics frame. Should the specified map receive changes after the forced update it will update again as well when the other maps receive their update.
## Avoidance processing and dispatch of the [code]safe_velocity[/code] signals is unaffected by this function and continues to happen for all maps and agents at the end of the physics frame.
## [b]Note:[/b] With great power comes great responsibility. This function should only be used by users that really know what they are doing and have a good reason for it. Forcing an immediate update of a navigation map requires locking the NavigationServer and flushing the entire NavigationServer command queue. Not only can this severely impact the performance of a game but it can also introduce bugs if used inappropriately without much foresight.
func map_force_update(map: RID) -> void:
	pass;

## Returns all navigation agents [RID]s that are currently assigned to the requested navigation [param map].
func map_get_agents(map: RID) -> Array[RID]:
	pass;

## Returns the map cell height used to rasterize the navigation mesh vertices on the Y axis.
func map_get_cell_height(map: RID) -> float:
	pass;

## Returns the map cell size used to rasterize the navigation mesh vertices on the XZ plane.
func map_get_cell_size(map: RID) -> float:
	pass;

## Returns the point closest to the provided [param to_point] on the navigation mesh surface.
func map_get_closest_point(map: RID, to_point: Vector3) -> Vector3:
	pass;

## Returns the normal for the point returned by [method map_get_closest_point].
func map_get_closest_point_normal(map: RID, to_point: Vector3) -> Vector3:
	pass;

## Returns the owner region RID for the point returned by [method map_get_closest_point].
func map_get_closest_point_owner(map: RID, to_point: Vector3) -> RID:
	pass;

## Returns the closest point between the navigation surface and the segment.
func map_get_closest_point_to_segment(map: RID, start: Vector3, end: Vector3, use_collision: bool = false) -> Vector3:
	pass;

## Returns the edge connection margin of the map. This distance is the minimum vertex distance needed to connect two edges from different regions.
func map_get_edge_connection_margin(map: RID) -> float:
	pass;

## Returns the current iteration id of the navigation map. Every time the navigation map changes and synchronizes the iteration id increases. An iteration id of 0 means the navigation map has never synchronized.
## [b]Note:[/b] The iteration id will wrap back to 1 after reaching its range limit.
func map_get_iteration_id(map: RID) -> int:
	pass;

## Returns the link connection radius of the map. This distance is the maximum range any link will search for navigation mesh polygons to connect to.
func map_get_link_connection_radius(map: RID) -> float:
	pass;

## Returns all navigation link [RID]s that are currently assigned to the requested navigation [param map].
func map_get_links(map: RID) -> Array[RID]:
	pass;

## Returns map's internal merge rasterizer cell scale.
func map_get_merge_rasterizer_cell_scale(map: RID) -> float:
	pass;

## Returns all navigation obstacle [RID]s that are currently assigned to the requested navigation [param map].
func map_get_obstacles(map: RID) -> Array[RID]:
	pass;

## Returns the navigation path to reach the destination from the origin. [param navigation_layers] is a bitmask of all region navigation layers that are allowed to be in the path.
func map_get_path(map: RID, origin: Vector3, destination: Vector3, optimize: bool, navigation_layers: int = 1) -> PackedVector3Array:
	pass;

## Returns a random position picked from all map region polygons with matching [param navigation_layers].
## If [param uniformly] is [code]true[/code], all map regions, polygons, and faces are weighted by their surface area (slower).
## If [param uniformly] is [code]false[/code], just a random region and a random polygon are picked (faster).
func map_get_random_point(map: RID, navigation_layers: int, uniformly: bool) -> Vector3:
	pass;

## Returns all navigation regions [RID]s that are currently assigned to the requested navigation [param map].
func map_get_regions(map: RID) -> Array[RID]:
	pass;

## Returns the map's up direction.
func map_get_up(map: RID) -> Vector3:
	pass;

## Returns true if the navigation [param map] allows navigation regions to use edge connections to connect with other navigation regions within proximity of the navigation map edge connection margin.
func map_get_use_edge_connections(map: RID) -> bool:
	pass;

## Returns true if the map is active.
func map_is_active(map: RID) -> bool:
	pass;

## Sets the map active.
func map_set_active(map: RID, active: bool) -> void:
	pass;

## Sets the map cell height used to rasterize the navigation mesh vertices on the Y axis. Must match with the cell height of the used navigation meshes.
func map_set_cell_height(map: RID, cell_height: float) -> void:
	pass;

## Sets the map cell size used to rasterize the navigation mesh vertices on the XZ plane. Must match with the cell size of the used navigation meshes.
func map_set_cell_size(map: RID, cell_size: float) -> void:
	pass;

## Set the map edge connection margin used to weld the compatible region edges.
func map_set_edge_connection_margin(map: RID, margin: float) -> void:
	pass;

## Set the map's link connection radius used to connect links to navigation polygons.
func map_set_link_connection_radius(map: RID, radius: float) -> void:
	pass;

## Set the map's internal merge rasterizer cell scale used to control merging sensitivity.
func map_set_merge_rasterizer_cell_scale(map: RID, scale: float) -> void:
	pass;

## Sets the map up direction.
func map_set_up(map: RID, up: Vector3) -> void:
	pass;

## Set the navigation [param map] edge connection use. If [param enabled] is [code]true[/code], the navigation map allows navigation regions to use edge connections to connect with other navigation regions within proximity of the navigation map edge connection margin.
func map_set_use_edge_connections(map: RID, enabled: bool) -> void:
	pass;

## Creates a new obstacle.
func obstacle_create() -> RID:
	pass;

## Returns [code]true[/code] if the provided [param obstacle] has avoidance enabled.
func obstacle_get_avoidance_enabled(obstacle: RID) -> bool:
	pass;

## Returns the [code]avoidance_layers[/code] bitmask of the specified [param obstacle].
func obstacle_get_avoidance_layers(obstacle: RID) -> int:
	pass;

## Returns the [code]height[/code] of the specified [param obstacle].
func obstacle_get_height(obstacle: RID) -> float:
	pass;

## Returns the navigation map [RID] the requested [param obstacle] is currently assigned to.
func obstacle_get_map(obstacle: RID) -> RID:
	pass;

## Returns [code]true[/code] if the specified [param obstacle] is paused.
func obstacle_get_paused(obstacle: RID) -> bool:
	pass;

## Returns the position of the specified [param obstacle] in world space.
func obstacle_get_position(obstacle: RID) -> Vector3:
	pass;

## Returns the radius of the specified dynamic [param obstacle].
func obstacle_get_radius(obstacle: RID) -> float:
	pass;

## Returns [code]true[/code] if the provided [param obstacle] uses avoidance in 3D space Vector3(x,y,z) instead of horizontal 2D Vector2(x,y) / Vector3(x,0.0,z).
func obstacle_get_use_3d_avoidance(obstacle: RID) -> bool:
	pass;

## Returns the velocity of the specified dynamic [param obstacle].
func obstacle_get_velocity(obstacle: RID) -> Vector3:
	pass;

## Returns the outline vertices for the specified [param obstacle].
func obstacle_get_vertices(obstacle: RID) -> PackedVector3Array:
	pass;

## If [param enabled] is [code]true[/code], the provided [param obstacle] affects avoidance using agents.
func obstacle_set_avoidance_enabled(obstacle: RID, enabled: bool) -> void:
	pass;

## Set the obstacles's [code]avoidance_layers[/code] bitmask.
func obstacle_set_avoidance_layers(obstacle: RID, layers: int) -> void:
	pass;

## Sets the [param height] for the [param obstacle]. In 3D agents will ignore obstacles that are above or below them while using 2D avoidance.
func obstacle_set_height(obstacle: RID, height: float) -> void:
	pass;

## Assigns the [param obstacle] to a navigation map.
func obstacle_set_map(obstacle: RID, map: RID) -> void:
	pass;

## If [param paused] is true the specified [param obstacle] will not be processed, e.g. affect avoidance velocities.
func obstacle_set_paused(obstacle: RID, paused: bool) -> void:
	pass;

## Updates the [param position] in world space for the [param obstacle].
func obstacle_set_position(obstacle: RID, position: Vector3) -> void:
	pass;

## Sets the radius of the dynamic obstacle.
func obstacle_set_radius(obstacle: RID, radius: float) -> void:
	pass;

## Sets if the [param obstacle] uses the 2D avoidance or the 3D avoidance while avoidance is enabled.
func obstacle_set_use_3d_avoidance(obstacle: RID, enabled: bool) -> void:
	pass;

## Sets [param velocity] of the dynamic [param obstacle]. Allows other agents to better predict the movement of the dynamic obstacle. Only works in combination with the radius of the obstacle.
func obstacle_set_velocity(obstacle: RID, velocity: Vector3) -> void:
	pass;

## Sets the outline vertices for the obstacle. If the vertices are winded in clockwise order agents will be pushed in by the obstacle, else they will be pushed out.
func obstacle_set_vertices(obstacle: RID, vertices: PackedVector3Array) -> void:
	pass;

## Parses the [SceneTree] for source geometry according to the properties of [param navigation_mesh]. Updates the provided [param source_geometry_data] resource with the resulting data. The resource can then be used to bake a navigation mesh with [method bake_from_source_geometry_data]. After the process is finished the optional [param callback] will be called.
## [b]Note:[/b] This function needs to run on the main thread or with a deferred call as the SceneTree is not thread-safe.
## [b]Performance:[/b] While convenient, reading data arrays from [Mesh] resources can affect the frame rate negatively. The data needs to be received from the GPU, stalling the [RenderingServer] in the process. For performance prefer the use of e.g. collision shapes or creating the data arrays entirely in code.
func parse_source_geometry_data(navigation_mesh: NavigationMesh, source_geometry_data: NavigationMeshSourceGeometryData3D, root_node: Node, callback: Callable = Callable()) -> void:
	pass;

## Queries a path in a given navigation map. Start and target position and other parameters are defined through [NavigationPathQueryParameters3D]. Updates the provided [NavigationPathQueryResult3D] result object with the path among other results requested by the query.
func query_path(parameters: NavigationPathQueryParameters3D, result: NavigationPathQueryResult3D) -> void:
	pass;

## Bakes the [param navigation_mesh] with bake source geometry collected starting from the [param root_node].
func region_bake_navigation_mesh(navigation_mesh: NavigationMesh, root_node: Node) -> void:
	pass;

## Creates a new region.
func region_create() -> RID:
	pass;

## Returns the ending point of a connection door. [param connection] is an index between 0 and the return value of [method region_get_connections_count].
func region_get_connection_pathway_end(region: RID, connection: int) -> Vector3:
	pass;

## Returns the starting point of a connection door. [param connection] is an index between 0 and the return value of [method region_get_connections_count].
func region_get_connection_pathway_start(region: RID, connection: int) -> Vector3:
	pass;

## Returns how many connections this [param region] has with other regions in the map.
func region_get_connections_count(region: RID) -> int:
	pass;

## Returns [code]true[/code] if the specified [param region] is enabled.
func region_get_enabled(region: RID) -> bool:
	pass;

## Returns the enter cost of this [param region].
func region_get_enter_cost(region: RID) -> float:
	pass;

## Returns the navigation map [RID] the requested [param region] is currently assigned to.
func region_get_map(region: RID) -> RID:
	pass;

## Returns the region's navigation layers.
func region_get_navigation_layers(region: RID) -> int:
	pass;

## Returns the [code]ObjectID[/code] of the object which manages this region.
func region_get_owner_id(region: RID) -> int:
	pass;

## Returns a random position picked from all region polygons with matching [param navigation_layers].
## If [param uniformly] is [code]true[/code], all region polygons and faces are weighted by their surface area (slower).
## If [param uniformly] is [code]false[/code], just a random polygon and face is picked (faster).
func region_get_random_point(region: RID, navigation_layers: int, uniformly: bool) -> Vector3:
	pass;

## Returns the global transformation of this [param region].
func region_get_transform(region: RID) -> Transform3D:
	pass;

## Returns the travel cost of this [param region].
func region_get_travel_cost(region: RID) -> float:
	pass;

## Returns true if the navigation [param region] is set to use edge connections to connect with other navigation regions within proximity of the navigation map edge connection margin.
func region_get_use_edge_connections(region: RID) -> bool:
	pass;

## Returns [code]true[/code] if the provided [param point] in world space is currently owned by the provided navigation [param region]. Owned in this context means that one of the region's navigation mesh polygon faces has a possible position at the closest distance to this point compared to all other navigation meshes from other navigation regions that are also registered on the navigation map of the provided region.
## If multiple navigation meshes have positions at equal distance the navigation region whose polygons are processed first wins the ownership. Polygons are processed in the same order that navigation regions were registered on the NavigationServer.
## [b]Note:[/b] If navigation meshes from different navigation regions overlap (which should be avoided in general) the result might not be what is expected.
func region_owns_point(region: RID, point: Vector3) -> bool:
	pass;

## If [param enabled] is [code]true[/code], the specified [param region] will contribute to its current navigation map.
func region_set_enabled(region: RID, enabled: bool) -> void:
	pass;

## Sets the [param enter_cost] for this [param region].
func region_set_enter_cost(region: RID, enter_cost: float) -> void:
	pass;

## Sets the map for the region.
func region_set_map(region: RID, map: RID) -> void:
	pass;

## Set the region's navigation layers. This allows selecting regions from a path request (when using [method NavigationServer3D.map_get_path]).
func region_set_navigation_layers(region: RID, navigation_layers: int) -> void:
	pass;

## Sets the navigation mesh for the region.
func region_set_navigation_mesh(region: RID, navigation_mesh: NavigationMesh) -> void:
	pass;

## Set the [code]ObjectID[/code] of the object which manages this region.
func region_set_owner_id(region: RID, owner_id: int) -> void:
	pass;

## Sets the global transformation for the region.
func region_set_transform(region: RID, transform: Transform3D) -> void:
	pass;

## Sets the [param travel_cost] for this [param region].
func region_set_travel_cost(region: RID, travel_cost: float) -> void:
	pass;

## If [param enabled] is [code]true[/code], the navigation [param region] will use edge connections to connect with other navigation regions within proximity of the navigation map edge connection margin.
func region_set_use_edge_connections(region: RID, enabled: bool) -> void:
	pass;

## Control activation of this server.
func set_active(active: bool) -> void:
	pass;

## If [code]true[/code] enables debug mode on the NavigationServer.
func set_debug_enabled(enabled: bool) -> void:
	pass;

## Returns a simplified version of [param path] with less critical path points removed. The simplification amount is in worlds units and controlled by [param epsilon]. The simplification uses a variant of Ramer-Douglas-Peucker algorithm for curve point decimation.
## Path simplification can be helpful to mitigate various path following issues that can arise with certain agent types and script behaviors. E.g. "steering" agents or avoidance in "open fields".
func simplify_path(path: PackedVector3Array, epsilon: float) -> PackedVector3Array:
	pass;

## Creates a new source geometry parser. If a [Callable] is set for the parser with [method source_geometry_parser_set_callback] the callback will be called for every single node that gets parsed whenever [method parse_source_geometry_data] is used.
func source_geometry_parser_create() -> RID:
	pass;

## Sets the [param callback] [Callable] for the specific source geometry [param parser]. The [Callable] will receive a call with the following parameters:
## - [code]navigation_mesh[/code] - The [NavigationMesh] reference used to define the parse settings. Do NOT edit or add directly to the navigation mesh.
## - [code]source_geometry_data[/code] - The [NavigationMeshSourceGeometryData3D] reference. Add custom source geometry for navigation mesh baking to this object.
## - [code]node[/code] - The [Node] that is parsed.
func source_geometry_parser_set_callback(parser: RID, callback: Callable) -> void:
	pass;


