extends Resource
class_name JSON

## Helper class for creating and parsing JSON data.
##
## The [JSON] class enables all data types to be converted to and from a JSON string. This is useful for serializing data, e.g. to save to a file or send over the network.
## [method stringify] is used to convert any data type into a JSON string.
## [method parse] is used to convert any existing JSON data into a [Variant] that can be used within Godot. If successfully parsed, use [member data] to retrieve the [Variant], and use [code]typeof[/code] to check if the Variant's type is what you expect. JSON Objects are converted into a [Dictionary], but JSON data can be used to store [Array]s, numbers, [String]s and even just a boolean.
## [b]Example[/b]
## [codeblock]
## var data_to_send = ["a", "b", "c"]
## var json_string = JSON.stringify(data_to_send)
## # Save data
## # ...
## # Retrieve data
## var json = JSON.new()
## var error = json.parse(json_string)
## if error == OK:
## var data_received = json.data
## if typeof(data_received) == TYPE_ARRAY:
## print(data_received) # Prints array
## else:
## print("Unexpected data")
## else:
## print("JSON Parse Error: ", json.get_error_message(), " in ", json_string, " at line ", json.get_error_line())
## [/codeblock]
## Alternatively, you can parse strings using the static [method parse_string] method, but it doesn't handle errors.
## [codeblock]
## var data = JSON.parse_string(json_string) # Returns null if parsing failed.
## [/codeblock]
## [b]Note:[/b] Both parse methods do not fully comply with the JSON specification:
## - Trailing commas in arrays or objects are ignored, instead of causing a parser error.
## - New line and tab characters are accepted in string literals, and are treated like their corresponding escape sequences [code]\n[/code] and [code]\t[/code].
## - Numbers are parsed using [method String.to_float] which is generally more lax than the JSON specification.
## - Certain errors, such as invalid Unicode sequences, do not cause a parser error. Instead, the string is cleansed and an error is logged to the console.



## Contains the parsed JSON data in [Variant] form.
var data: Variant:
	get = get_data, set = set_data



## Returns [code]0[/code] if the last call to [method parse] was successful, or the line number where the parse failed.
func get_error_line() -> int:
	pass;

## Returns an empty string if the last call to [method parse] was successful, or the error message if it failed.
func get_error_message() -> String:
	pass;

## Return the text parsed by [method parse] (requires passing [code]keep_text[/code] to [method parse]).
func get_parsed_text() -> String:
	pass;

## Attempts to parse the [param json_text] provided.
## Returns an [enum Error]. If the parse was successful, it returns [constant OK] and the result can be retrieved using [member data]. If unsuccessful, use [method get_error_line] and [method get_error_message] to identify the source of the failure.
## Non-static variant of [method parse_string], if you want custom error handling.
## The optional [param keep_text] argument instructs the parser to keep a copy of the original text. This text can be obtained later by using the [method get_parsed_text] function and is used when saving the resource (instead of generating new text from [member data]).
func parse(json_text: String, keep_text: bool = false) -> int:
	pass;

## Attempts to parse the [param json_string] provided and returns the parsed data. Returns [code]null[/code] if parse failed.
static func parse_string(json_string: String) -> Variant:
	pass;

## Converts a [Variant] var to JSON text and returns the result. Useful for serializing data to store or send over the network.
## [b]Note:[/b] The JSON specification does not define integer or float types, but only a [i]number[/i] type. Therefore, converting a Variant to JSON text will convert all numerical values to [float] types.
## [b]Note:[/b] If [param full_precision] is [code]true[/code], when stringifying floats, the unreliable digits are stringified in addition to the reliable digits to guarantee exact decoding.
## The [param indent] parameter controls if and how something is indented; its contents will be used where there should be an indent in the output. Even spaces like [code]"   "[/code] will work. [code]\t[/code] and [code]\n[/code] can also be used for a tab indent, or to make a newline for each indent respectively.
## [b]Example output:[/b]
## [codeblock]
## ## JSON.stringify(my_dictionary)
## {"name":"my_dictionary","version":"1.0.0","entities":[{"name":"entity_0","value":"value_0"},{"name":"entity_1","value":"value_1"}]}
## ## JSON.stringify(my_dictionary, "\t")
## {
## "name": "my_dictionary",
## "version": "1.0.0",
## "entities": [
## {
## "name": "entity_0",
## "value": "value_0"
## },
## {
## "name": "entity_1",
## "value": "value_1"
## }
## ]
## }
## ## JSON.stringify(my_dictionary, "...")
## {
## ..."name": "my_dictionary",
## ..."version": "1.0.0",
## ..."entities": [
## ......{
## ........."name": "entity_0",
## ........."value": "value_0"
## ......},
## ......{
## ........."name": "entity_1",
## ........."value": "value_1"
## ......}
## ...]
## }
## [/codeblock]
static func stringify(data: Variant, indent: String = "", sort_keys: bool = true, full_precision: bool = false) -> String:
	pass;


func get_data() -> Variant:
	return data

func set_data(value: Variant) -> void:
	data = value

