extends Node
class_name HTTPRequest

## A node with the ability to send HTTP(S) requests.
##
## A node with the ability to send HTTP requests. Uses [HTTPClient] internally.
## Can be used to make HTTP requests, i.e. download or upload files or web content via HTTP.
## [b]Warning:[/b] See the notes and warnings on [HTTPClient] for limitations, especially regarding TLS security.
## [b]Note:[/b] When exporting to Android, make sure to enable the [code]INTERNET[/code] permission in the Android export preset before exporting the project or using one-click deploy. Otherwise, network communication of any kind will be blocked by Android.
## [b]Example of contacting a REST API and printing one of its returned fields:[/b]
## [codeblocks]
## [gdscript]
## func _ready():
## # Create an HTTP request node and connect its completion signal.
## var http_request = HTTPRequest.new()
## add_child(http_request)
## http_request.request_completed.connect(self._http_request_completed)
## # Perform a GET request. The URL below returns JSON as of writing.
## var error = http_request.request("https://httpbin.org/get")
## if error != OK:
## push_error("An error occurred in the HTTP request.")
## # Perform a POST request. The URL below returns JSON as of writing.
## # Note: Don't make simultaneous requests using a single HTTPRequest node.
## # The snippet below is provided for reference only.
## var body = JSON.new().stringify({"name": "Godette"})
## error = http_request.request("https://httpbin.org/post", [], HTTPClient.METHOD_POST, body)
## if error != OK:
## push_error("An error occurred in the HTTP request.")
## # Called when the HTTP request is completed.
## func _http_request_completed(result, response_code, headers, body):
## var json = JSON.new()
## json.parse(body.get_string_from_utf8())
## var response = json.get_data()
## # Will print the user agent string used by the HTTPRequest node (as recognized by httpbin.org).
## print(response.headers["User-Agent"])
## [/gdscript]
## [csharp]
## public override void _Ready()
## {
## // Create an HTTP request node and connect its completion signal.
## var httpRequest = new HttpRequest();
## AddChild(httpRequest);
## httpRequest.RequestCompleted += HttpRequestCompleted;
## // Perform a GET request. The URL below returns JSON as of writing.
## Error error = httpRequest.Request("https://httpbin.org/get");
## if (error != Error.Ok)
## {
## GD.PushError("An error occurred in the HTTP request.");
## }
## // Perform a POST request. The URL below returns JSON as of writing.
## // Note: Don't make simultaneous requests using a single HTTPRequest node.
## // The snippet below is provided for reference only.
## string body = new Json().Stringify(new Godot.Collections.Dictionary
## {
## { "name", "Godette" }
## });
## error = httpRequest.Request("https://httpbin.org/post", null, HttpClient.Method.Post, body);
## if (error != Error.Ok)
## {
## GD.PushError("An error occurred in the HTTP request.");
## }
## }
## // Called when the HTTP request is completed.
## private void HttpRequestCompleted(long result, long responseCode, string[] headers, byte[] body)
## {
## var json = new Json();
## json.Parse(body.GetStringFromUtf8());
## var response = json.GetData().AsGodotDictionary();
## // Will print the user agent string used by the HTTPRequest node (as recognized by httpbin.org).
## GD.Print((response["headers"].AsGodotDictionary())["User-Agent"]);
## }
## [/csharp]
## [/codeblocks]
## [b]Example of loading and displaying an image using HTTPRequest:[/b]
## [codeblocks]
## [gdscript]
## func _ready():
## # Create an HTTP request node and connect its completion signal.
## var http_request = HTTPRequest.new()
## add_child(http_request)
## http_request.request_completed.connect(self._http_request_completed)
## # Perform the HTTP request. The URL below returns a PNG image as of writing.
## var error = http_request.request("https://via.placeholder.com/512")
## if error != OK:
## push_error("An error occurred in the HTTP request.")
## # Called when the HTTP request is completed.
## func _http_request_completed(result, response_code, headers, body):
## if result != HTTPRequest.RESULT_SUCCESS:
## push_error("Image couldn't be downloaded. Try a different image.")
## var image = Image.new()
## var error = image.load_png_from_buffer(body)
## if error != OK:
## push_error("Couldn't load the image.")
## var texture = ImageTexture.create_from_image(image)
## # Display the image in a TextureRect node.
## var texture_rect = TextureRect.new()
## add_child(texture_rect)
## texture_rect.texture = texture
## [/gdscript]
## [csharp]
## public override void _Ready()
## {
## // Create an HTTP request node and connect its completion signal.
## var httpRequest = new HttpRequest();
## AddChild(httpRequest);
## httpRequest.RequestCompleted += HttpRequestCompleted;
## // Perform the HTTP request. The URL below returns a PNG image as of writing.
## Error error = httpRequest.Request("https://via.placeholder.com/512");
## if (error != Error.Ok)
## {
## GD.PushError("An error occurred in the HTTP request.");
## }
## }
## // Called when the HTTP request is completed.
## private void HttpRequestCompleted(long result, long responseCode, string[] headers, byte[] body)
## {
## if (result != (long)HttpRequest.Result.Success)
## {
## GD.PushError("Image couldn't be downloaded. Try a different image.");
## }
## var image = new Image();
## Error error = image.LoadPngFromBuffer(body);
## if (error != Error.Ok)
## {
## GD.PushError("Couldn't load the image.");
## }
## var texture = ImageTexture.CreateFromImage(image);
## // Display the image in a TextureRect node.
## var textureRect = new TextureRect();
## AddChild(textureRect);
## textureRect.Texture = texture;
## }
## [/csharp]
## [/codeblocks]
## [b]Gzipped response bodies[/b]: HTTPRequest will automatically handle decompression of response bodies. A [code]Accept-Encoding[/code] header will be automatically added to each of your requests, unless one is already specified. Any response with a [code]Content-Encoding: gzip[/code] header will automatically be decompressed and delivered to you as uncompressed bytes.
##
## @tutorial(Making HTTP requests): https://docs.godotengine.org/en/stable/tutorials/networking/http_request_class.html
## @tutorial(TLS certificates): https://docs.godotengine.org/en/stable/tutorials/networking/ssl_certificates.html


## Emitted when a request is completed.
signal request_completed(result: int, response_code: int, headers: PackedStringArray, body: PackedByteArray)
## Request successful.
## Request failed while connecting.
## Request failed while resolving.
## Request failed due to connection (read/write) error.
## Request failed on TLS handshake.
## Request does not have a response (yet).
## Request exceeded its maximum size limit, see [member body_size_limit].
## Request failed (currently unused).
## HTTPRequest couldn't open the download file.
## HTTPRequest couldn't write to the download file.
## Request reached its maximum redirect limit, see [member max_redirects].
## Request failed due to a timeout. If you expect requests to take a long time, try increasing the value of [member timeout] or setting it to [code]0.0[/code] to remove the timeout completely.

#enum Result
enum {
    RESULT_SUCCESS = 0,
    RESULT_CHUNKED_BODY_SIZE_MISMATCH = 1,
    RESULT_CANT_CONNECT = 2,
    RESULT_CANT_RESOLVE = 3,
    RESULT_CONNECTION_ERROR = 4,
    RESULT_TLS_HANDSHAKE_ERROR = 5,
    RESULT_NO_RESPONSE = 6,
    RESULT_BODY_SIZE_LIMIT_EXCEEDED = 7,
    RESULT_BODY_DECOMPRESS_FAILED = 8,
    RESULT_REQUEST_FAILED = 9,
    RESULT_DOWNLOAD_FILE_CANT_OPEN = 10,
    RESULT_DOWNLOAD_FILE_WRITE_ERROR = 11,
    RESULT_REDIRECT_LIMIT_REACHED = 12,
    RESULT_TIMEOUT = 13,
}
## If [code]true[/code], this header will be added to each request: [code]Accept-Encoding: gzip, deflate[/code] telling servers that it's okay to compress response bodies.
## Any Response body declaring a [code]Content-Encoding[/code] of either [code]gzip[/code] or [code]deflate[/code] will then be automatically decompressed, and the uncompressed bytes will be delivered via [signal request_completed].
## If the user has specified their own [code]Accept-Encoding[/code] header, then no header will be added regardless of [member accept_gzip].
## If [code]false[/code] no header will be added, and no decompression will be performed on response bodies. The raw bytes of the response body will be returned via [signal request_completed].
var accept_gzip: bool:
	get = is_accepting_gzip, set = set_accept_gzip

## Maximum allowed size for response bodies. If the response body is compressed, this will be used as the maximum allowed size for the decompressed body.
var body_size_limit: int:
	get = get_body_size_limit, set = set_body_size_limit

## The size of the buffer used and maximum bytes to read per iteration. See [member HTTPClient.read_chunk_size].
## Set this to a lower value (e.g. 4096 for 4 KiB) when downloading small files to decrease memory usage at the cost of download speeds.
var download_chunk_size: int:
	get = get_download_chunk_size, set = set_download_chunk_size

## The file to download into. Will output any received file into it.
var download_file: String:
	get = get_download_file, set = set_download_file

## Maximum number of allowed redirects.
var max_redirects: int:
	get = get_max_redirects, set = set_max_redirects

## The duration to wait in seconds before a request times out. If [member timeout] is set to [code]0.0[/code] then the request will never time out. For simple requests, such as communication with a REST API, it is recommended that [member timeout] is set to a value suitable for the server response time (e.g. between [code]1.0[/code] and [code]10.0[/code]). This will help prevent unwanted timeouts caused by variation in server response times while still allowing the application to detect when a request has timed out. For larger requests such as file downloads it is suggested the [member timeout] be set to [code]0.0[/code], disabling the timeout functionality. This will help to prevent large transfers from failing due to exceeding the timeout value.
var timeout: float:
	get = get_timeout, set = set_timeout

## If [code]true[/code], multithreading is used to improve performance.
var use_threads: bool:
	get = is_using_threads, set = set_use_threads



## Cancels the current request.
func cancel_request() -> void:
	pass;

## Returns the response body length.
## [b]Note:[/b] Some Web servers may not send a body length. In this case, the value returned will be [code]-1[/code]. If using chunked transfer encoding, the body length will also be [code]-1[/code].
func get_body_size() -> int:
	pass;

## Returns the number of bytes this HTTPRequest downloaded.
func get_downloaded_bytes() -> int:
	pass;

## Returns the current status of the underlying [HTTPClient]. See [enum HTTPClient.Status].
func get_http_client_status() -> int:
	pass;

## Creates request on the underlying [HTTPClient]. If there is no configuration errors, it tries to connect using [method HTTPClient.connect_to_host] and passes parameters onto [method HTTPClient.request].
## Returns [constant OK] if request is successfully created. (Does not imply that the server has responded), [constant ERR_UNCONFIGURED] if not in the tree, [constant ERR_BUSY] if still processing previous request, [constant ERR_INVALID_PARAMETER] if given string is not a valid URL format, or [constant ERR_CANT_CONNECT] if not using thread and the [HTTPClient] cannot connect to host.
## [b]Note:[/b] When [param method] is [constant HTTPClient.METHOD_GET], the payload sent via [param request_data] might be ignored by the server or even cause the server to reject the request (check [url=https://datatracker.ietf.org/doc/html/rfc7231#section-4.3.1]RFC 7231 section 4.3.1[/url] for more details). As a workaround, you can send data as a query string in the URL (see [method String.uri_encode] for an example).
## [b]Note:[/b] It's recommended to use transport encryption (TLS) and to avoid sending sensitive information (such as login credentials) in HTTP GET URL parameters. Consider using HTTP POST requests or HTTP headers for such information instead.
func request(url: String, custom_headers: PackedStringArray = PackedStringArray(), method: int = 0, request_data: String = "") -> int:
	pass;

## Creates request on the underlying [HTTPClient] using a raw array of bytes for the request body. If there is no configuration errors, it tries to connect using [method HTTPClient.connect_to_host] and passes parameters onto [method HTTPClient.request].
## Returns [constant OK] if request is successfully created. (Does not imply that the server has responded), [constant ERR_UNCONFIGURED] if not in the tree, [constant ERR_BUSY] if still processing previous request, [constant ERR_INVALID_PARAMETER] if given string is not a valid URL format, or [constant ERR_CANT_CONNECT] if not using thread and the [HTTPClient] cannot connect to host.
func request_raw(url: String, custom_headers: PackedStringArray = PackedStringArray(), method: int = 0, request_data_raw: PackedByteArray = PackedByteArray()) -> int:
	pass;

## Sets the proxy server for HTTP requests.
## The proxy server is unset if [param host] is empty or [param port] is -1.
func set_http_proxy(host: String, port: int) -> void:
	pass;

## Sets the proxy server for HTTPS requests.
## The proxy server is unset if [param host] is empty or [param port] is -1.
func set_https_proxy(host: String, port: int) -> void:
	pass;

## Sets the [TLSOptions] to be used when connecting to an HTTPS server. See [method TLSOptions.client].
func set_tls_options(client_options: TLSOptions) -> void:
	pass;


func is_accepting_gzip() -> bool:
	return accept_gzip

func set_accept_gzip(value: bool) -> void:
	accept_gzip = value

func get_body_size_limit() -> int:
	return body_size_limit

func set_body_size_limit(value: int) -> void:
	body_size_limit = value

func get_download_chunk_size() -> int:
	return download_chunk_size

func set_download_chunk_size(value: int) -> void:
	download_chunk_size = value

func get_download_file() -> String:
	return download_file

func set_download_file(value: String) -> void:
	download_file = value

func get_max_redirects() -> int:
	return max_redirects

func set_max_redirects(value: int) -> void:
	max_redirects = value

func get_timeout() -> float:
	return timeout

func set_timeout(value: float) -> void:
	timeout = value

func is_using_threads() -> bool:
	return use_threads

func set_use_threads(value: bool) -> void:
	use_threads = value

