extends Resource
class_name GLTFDocument

## Class for importing and exporting glTF files in and out of Godot.
##
## GLTFDocument supports reading data from a glTF file, buffer, or Godot scene. This data can then be written to the filesystem, buffer, or used to create a Godot scene.
## All of the data in a GLTF scene is stored in the [GLTFState] class. GLTFDocument processes state objects, but does not contain any scene data itself. GLTFDocument has member variables to store export configuration settings such as the image format, but is otherwise stateless. Multiple scenes can be processed with the same settings using the same GLTFDocument object and different [GLTFState] objects.
## GLTFDocument can be extended with arbitrary functionality by extending the [GLTFDocumentExtension] class and registering it with GLTFDocument via [method register_gltf_document_extension]. This allows for custom data to be imported and exported.
##
## @tutorial(Runtime file loading and saving): https://docs.godotengine.org/en/stable/tutorials/io/runtime_file_loading_and_saving.html
## @tutorial(glTF 'What the duck?' guide): https://www.khronos.org/files/gltf20-reference-guide.pdf
## @tutorial(Khronos glTF specification): https://registry.khronos.org/glTF/


## Treat the Godot scene's root node as the root node of the glTF file, and mark it as the single root node via the [code]GODOT_single_root[/code] glTF extension. This will be parsed the same as [constant ROOT_NODE_MODE_KEEP_ROOT] if the implementation does not support [code]GODOT_single_root[/code].
## Treat the Godot scene's root node as the root node of the glTF file, but do not mark it as anything special. An extra root node will be generated when importing into Godot. This uses only vanilla glTF features. This is equivalent to the behavior in Godot 4.1 and earlier.
## Treat the Godot scene's root node as the name of the glTF scene, and add all of its children as root nodes of the glTF file. This uses only vanilla glTF features. This avoids an extra root node, but only the name of the Godot scene's root node will be preserved, as it will not be saved as a node.

#enum RootNodeMode
enum {
    ROOT_NODE_MODE_SINGLE_ROOT = 0,
    ROOT_NODE_MODE_KEEP_ROOT = 1,
    ROOT_NODE_MODE_MULTI_ROOT = 2,
}
## The user-friendly name of the export image format. This is used when exporting the GLTF file, including writing to a file and writing to a byte array.
## By default, Godot allows the following options: "None", "PNG", "JPEG", "Lossless WebP", and "Lossy WebP". Support for more image formats can be added in [GLTFDocumentExtension] classes.
var image_format: String:
	get = get_image_format, set = set_image_format

## If [member image_format] is a lossy image format, this determines the lossy quality of the image. On a range of [code]0.0[/code] to [code]1.0[/code], where [code]0.0[/code] is the lowest quality and [code]1.0[/code] is the highest quality. A lossy quality of [code]1.0[/code] is not the same as lossless.
var lossy_quality: float:
	get = get_lossy_quality, set = set_lossy_quality

## How to process the root node during export. See [enum RootNodeMode] for details. The default and recommended value is [constant ROOT_NODE_MODE_SINGLE_ROOT].
## [b]Note:[/b] Regardless of how the glTF file is exported, when importing, the root node type and name can be overridden in the scene import settings tab.
var root_node_mode: int:
	get = get_root_node_mode, set = set_root_node_mode



## Takes a [PackedByteArray] defining a GLTF and imports the data to the given [GLTFState] object through the [param state] parameter.
## [b]Note:[/b] The [param base_path] tells [method append_from_buffer] where to find dependencies and can be empty.
func append_from_buffer(bytes: PackedByteArray, base_path: String, state: GLTFState, flags: int = 0) -> int:
	pass;

## Takes a path to a GLTF file and imports the data at that file path to the given [GLTFState] object through the [param state] parameter.
## [b]Note:[/b] The [param base_path] tells [method append_from_file] where to find dependencies and can be empty.
func append_from_file(path: String, state: GLTFState, flags: int = 0, base_path: String = "") -> int:
	pass;

## Takes a Godot Engine scene node and exports it and its descendants to the given [GLTFState] object through the [param state] parameter.
func append_from_scene(node: Node, state: GLTFState, flags: int = 0) -> int:
	pass;

## Takes a [GLTFState] object through the [param state] parameter and returns a GLTF [PackedByteArray].
func generate_buffer(state: GLTFState) -> PackedByteArray:
	pass;

## Takes a [GLTFState] object through the [param state] parameter and returns a Godot Engine scene node.
## The [param bake_fps] parameter overrides the bake_fps in [param state].
func generate_scene(state: GLTFState, bake_fps: float = 30, trimming: bool = false, remove_immutable_tracks: bool = true) -> Node:
	pass;

## Registers the given [GLTFDocumentExtension] instance with GLTFDocument. If [param first_priority] is true, this extension will be run first. Otherwise, it will be run last.
## [b]Note:[/b] Like GLTFDocument itself, all GLTFDocumentExtension classes must be stateless in order to function properly. If you need to store data, use the [code]set_additional_data[/code] and [code]get_additional_data[/code] methods in [GLTFState] or [GLTFNode].
static func register_gltf_document_extension(extension: GLTFDocumentExtension, first_priority: bool = false) -> void:
	pass;

## Unregisters the given [GLTFDocumentExtension] instance.
static func unregister_gltf_document_extension(extension: GLTFDocumentExtension) -> void:
	pass;

## Takes a [GLTFState] object through the [param state] parameter and writes a glTF file to the filesystem.
## [b]Note:[/b] The extension of the glTF file determines if it is a .glb binary file or a .gltf text file.
func write_to_filesystem(state: GLTFState, path: String) -> int:
	pass;


func get_image_format() -> String:
	return image_format

func set_image_format(value: String) -> void:
	image_format = value

func get_lossy_quality() -> float:
	return lossy_quality

func set_lossy_quality(value: float) -> void:
	lossy_quality = value

func get_root_node_mode() -> int:
	return root_node_mode

func set_root_node_mode(value: int) -> void:
	root_node_mode = value

