extends Resource
class_name GLTFAccessor

## Represents a GLTF accessor.
##
## GLTFAccessor is a data structure representing GLTF a [code]accessor[/code] that would be found in the [code]"accessors"[/code] array. A buffer is a blob of binary data. A buffer view is a slice of a buffer. An accessor is a typed interpretation of the data in a buffer view.
## Most custom data stored in GLTF does not need accessors, only buffer views (see [GLTFBufferView]). Accessors are for more advanced use cases such as interleaved mesh data encoded for the GPU.
##
## @tutorial(Buffers, BufferViews, and Accessors in Khronos glTF specification): https://github.com/KhronosGroup/glTF-Tutorials/blob/master/gltfTutorial/gltfTutorial_005_BuffersBufferViewsAccessors.md
## @tutorial(Runtime file loading and saving): https://docs.godotengine.org/en/stable/tutorials/io/runtime_file_loading_and_saving.html


## Accessor type "SCALAR". For the glTF object model, this can be used to map to a single float, int, or bool value, or a float array.
## Accessor type "VEC2". For the glTF object model, this maps to "float2", represented in the glTF JSON as an array of two floats.
## Accessor type "VEC3". For the glTF object model, this maps to "float3", represented in the glTF JSON as an array of three floats.
## Accessor type "VEC4". For the glTF object model, this maps to "float4", represented in the glTF JSON as an array of four floats.
## Accessor type "MAT2". For the glTF object model, this maps to "float2x2", represented in the glTF JSON as an array of four floats.
## Accessor type "MAT3". For the glTF object model, this maps to "float3x3", represented in the glTF JSON as an array of nine floats.
## Accessor type "MAT4". For the glTF object model, this maps to "float4x4", represented in the glTF JSON as an array of sixteen floats.

#enum GLTFAccessorType
enum {
    TYPE_SCALAR = 0,
    TYPE_VEC2 = 1,
    TYPE_VEC3 = 2,
    TYPE_VEC4 = 3,
    TYPE_MAT2 = 4,
    TYPE_MAT3 = 5,
    TYPE_MAT4 = 6,
}
## The GLTF accessor type as an enum. Possible values are 0 for "SCALAR", 1 for "VEC2", 2 for "VEC3", 3 for "VEC4", 4 for "MAT2", 5 for "MAT3", and 6 for "MAT4".
var accessor_type: int:
	get = get_accessor_type, set = set_accessor_type

## The index of the buffer view this accessor is referencing. If [code]-1[/code], this accessor is not referencing any buffer view.
var buffer_view: int:
	get = get_buffer_view, set = set_buffer_view

## The offset relative to the start of the buffer view in bytes.
var byte_offset: int:
	get = get_byte_offset, set = set_byte_offset

## The GLTF component type as an enum. Possible values are 5120 for "BYTE", 5121 for "UNSIGNED_BYTE", 5122 for "SHORT", 5123 for "UNSIGNED_SHORT", 5125 for "UNSIGNED_INT", and 5126 for "FLOAT". A value of 5125 or "UNSIGNED_INT" must not be used for any accessor that is not referenced by mesh.primitive.indices.
var component_type: int:
	get = get_component_type, set = set_component_type

## The number of elements referenced by this accessor.
var count: int:
	get = get_count, set = set_count

## Maximum value of each component in this accessor.
var max: PackedFloat64Array:
	get = get_max, set = set_max

## Minimum value of each component in this accessor.
var min: PackedFloat64Array:
	get = get_min, set = set_min

## Specifies whether integer data values are normalized before usage.
var normalized: bool:
	get = get_normalized, set = set_normalized

## Number of deviating accessor values stored in the sparse array.
var sparse_count: int:
	get = get_sparse_count, set = set_sparse_count

## The index of the buffer view with sparse indices. The referenced buffer view MUST NOT have its target or byteStride properties defined. The buffer view and the optional byteOffset MUST be aligned to the componentType byte length.
var sparse_indices_buffer_view: int:
	get = get_sparse_indices_buffer_view, set = set_sparse_indices_buffer_view

## The offset relative to the start of the buffer view in bytes.
var sparse_indices_byte_offset: int:
	get = get_sparse_indices_byte_offset, set = set_sparse_indices_byte_offset

## The indices component data type as an enum. Possible values are 5121 for "UNSIGNED_BYTE", 5123 for "UNSIGNED_SHORT", and 5125 for "UNSIGNED_INT".
var sparse_indices_component_type: int:
	get = get_sparse_indices_component_type, set = set_sparse_indices_component_type

## The index of the bufferView with sparse values. The referenced buffer view MUST NOT have its target or byteStride properties defined.
var sparse_values_buffer_view: int:
	get = get_sparse_values_buffer_view, set = set_sparse_values_buffer_view

## The offset relative to the start of the bufferView in bytes.
var sparse_values_byte_offset: int:
	get = get_sparse_values_byte_offset, set = set_sparse_values_byte_offset

## The GLTF accessor type as an enum. Use [member accessor_type] instead.
var type: int:
	get = get_type, set = set_type




func get_accessor_type() -> int:
	return accessor_type

func set_accessor_type(value: int) -> void:
	accessor_type = value

func get_buffer_view() -> int:
	return buffer_view

func set_buffer_view(value: int) -> void:
	buffer_view = value

func get_byte_offset() -> int:
	return byte_offset

func set_byte_offset(value: int) -> void:
	byte_offset = value

func get_component_type() -> int:
	return component_type

func set_component_type(value: int) -> void:
	component_type = value

func get_count() -> int:
	return count

func set_count(value: int) -> void:
	count = value

func get_max() -> PackedFloat64Array:
	return max

func set_max(value: PackedFloat64Array) -> void:
	max = value

func get_min() -> PackedFloat64Array:
	return min

func set_min(value: PackedFloat64Array) -> void:
	min = value

func get_normalized() -> bool:
	return normalized

func set_normalized(value: bool) -> void:
	normalized = value

func get_sparse_count() -> int:
	return sparse_count

func set_sparse_count(value: int) -> void:
	sparse_count = value

func get_sparse_indices_buffer_view() -> int:
	return sparse_indices_buffer_view

func set_sparse_indices_buffer_view(value: int) -> void:
	sparse_indices_buffer_view = value

func get_sparse_indices_byte_offset() -> int:
	return sparse_indices_byte_offset

func set_sparse_indices_byte_offset(value: int) -> void:
	sparse_indices_byte_offset = value

func get_sparse_indices_component_type() -> int:
	return sparse_indices_component_type

func set_sparse_indices_component_type(value: int) -> void:
	sparse_indices_component_type = value

func get_sparse_values_buffer_view() -> int:
	return sparse_values_buffer_view

func set_sparse_values_buffer_view(value: int) -> void:
	sparse_values_buffer_view = value

func get_sparse_values_byte_offset() -> int:
	return sparse_values_byte_offset

func set_sparse_values_byte_offset(value: int) -> void:
	sparse_values_byte_offset = value

func get_type() -> int:
	return type

func set_type(value: int) -> void:
	type = value

