extends RefCounted
class_name DirAccess

## Provides methods for managing directories and their content.
##
## This class is used to manage directories and their content, even outside of the project folder.
## [DirAccess] can't be instantiated directly. Instead it is created with a static method that takes a path for which it will be opened.
## Most of the methods have a static alternative that can be used without creating a [DirAccess]. Static methods only support absolute paths (including [code]res://[/code] and [code]user://[/code]).
## [codeblock]
## # Standard
## var dir = DirAccess.open("user://levels")
## dir.make_dir("world1")
## # Static
## DirAccess.make_dir_absolute("user://levels/world1")
## [/codeblock]
## [b]Note:[/b] Many resources types are imported (e.g. textures or sound files), and their source asset will not be included in the exported game, as only the imported version is used. Use [ResourceLoader] to access imported resources.
## Here is an example on how to iterate through the files of a directory:
## [codeblocks]
## [gdscript]
## func dir_contents(path):
## var dir = DirAccess.open(path)
## if dir:
## dir.list_dir_begin()
## var file_name = dir.get_next()
## while file_name != "":
## if dir.current_is_dir():
## print("Found directory: " + file_name)
## else:
## print("Found file: " + file_name)
## file_name = dir.get_next()
## else:
## print("An error occurred when trying to access the path.")
## [/gdscript]
## [csharp]
## public void DirContents(string path)
## {
## using var dir = DirAccess.Open(path);
## if (dir != null)
## {
## dir.ListDirBegin();
## string fileName = dir.GetNext();
## while (fileName != "")
## {
## if (dir.CurrentIsDir())
## {
## GD.Print($"Found directory: {fileName}");
## }
## else
## {
## GD.Print($"Found file: {fileName}");
## }
## fileName = dir.GetNext();
## }
## }
## else
## {
## GD.Print("An error occurred when trying to access the path.");
## }
## }
## [/csharp]
## [/codeblocks]
##
## @tutorial(File system): https://docs.godotengine.org/en/stable/tutorials/scripting/filesystem.html



## If [code]true[/code], hidden files are included when navigating the directory.
## Affects [method list_dir_begin], [method get_directories] and [method get_files].
var include_hidden: bool:
	get = get_include_hidden, set = set_include_hidden

## If [code]true[/code], [code].[/code] and [code]..[/code] are included when navigating the directory.
## Affects [method list_dir_begin] and [method get_directories].
var include_navigational: bool:
	get = get_include_navigational, set = set_include_navigational



## Changes the currently opened directory to the one passed as an argument. The argument can be relative to the current directory (e.g. [code]newdir[/code] or [code]../newdir[/code]), or an absolute path (e.g. [code]/tmp/newdir[/code] or [code]res://somedir/newdir[/code]).
## Returns one of the [enum Error] code constants ([constant OK] on success).
## [b]Note:[/b] The new directory must be within the same scope, e.g. when you had opened a directory inside [code]res://[/code], you can't change it to [code]user://[/code] directory. If you need to open a directory in another access scope, use [method open] to create a new instance instead.
func change_dir(to_dir: String) -> int:
	pass;

## Copies the [param from] file to the [param to] destination. Both arguments should be paths to files, either relative or absolute. If the destination file exists and is not access-protected, it will be overwritten.
## If [param chmod_flags] is different than [code]-1[/code], the Unix permissions for the destination path will be set to the provided value, if available on the current operating system.
## Returns one of the [enum Error] code constants ([constant OK] on success).
func copy(from: String, to: String, chmod_flags: int = -1) -> int:
	pass;

## Static version of [method copy]. Supports only absolute paths.
static func copy_absolute(from: String, to: String, chmod_flags: int = -1) -> int:
	pass;

## Creates symbolic link between files or folders.
## [b]Note:[/b] On Windows, this method works only if the application is running with elevated privileges or Developer Mode is enabled.
## [b]Note:[/b] This method is implemented on macOS, Linux, and Windows.
func create_link(source: String, target: String) -> int:
	pass;

## Returns whether the current item processed with the last [method get_next] call is a directory ([code].[/code] and [code]..[/code] are considered directories).
func current_is_dir() -> bool:
	pass;

## Returns whether the target directory exists. The argument can be relative to the current directory, or an absolute path.
func dir_exists(path: String) -> bool:
	pass;

## Static version of [method dir_exists]. Supports only absolute paths.
static func dir_exists_absolute(path: String) -> bool:
	pass;

## Returns whether the target file exists. The argument can be relative to the current directory, or an absolute path.
## For a static equivalent, use [method FileAccess.file_exists].
func file_exists(path: String) -> bool:
	pass;

## Returns the absolute path to the currently opened directory (e.g. [code]res://folder[/code] or [code]C:\tmp\folder[/code]).
func get_current_dir(include_drive: bool = true) -> String:
	pass;

## Returns the currently opened directory's drive index. See [method get_drive_name] to convert returned index to the name of the drive.
func get_current_drive() -> int:
	pass;

## Returns a [PackedStringArray] containing filenames of the directory contents, excluding files. The array is sorted alphabetically.
## Affected by [member include_hidden] and [member include_navigational].
func get_directories() -> PackedStringArray:
	pass;

## Returns a [PackedStringArray] containing filenames of the directory contents, excluding files, at the given [param path]. The array is sorted alphabetically.
## Use [method get_directories] if you want more control of what gets included.
static func get_directories_at(path: String) -> PackedStringArray:
	pass;

## On Windows, returns the number of drives (partitions) mounted on the current filesystem.
## On macOS, returns the number of mounted volumes.
## On Linux, returns the number of mounted volumes and GTK 3 bookmarks.
## On other platforms, the method returns 0.
static func get_drive_count() -> int:
	pass;

## On Windows, returns the name of the drive (partition) passed as an argument (e.g. [code]C:[/code]).
## On macOS, returns the path to the mounted volume passed as an argument.
## On Linux, returns the path to the mounted volume or GTK 3 bookmark passed as an argument.
## On other platforms, or if the requested drive does not exist, the method returns an empty String.
static func get_drive_name(idx: int) -> String:
	pass;

## Returns a [PackedStringArray] containing filenames of the directory contents, excluding directories. The array is sorted alphabetically.
## Affected by [member include_hidden].
## [b]Note:[/b] When used on a [code]res://[/code] path in an exported project, only the files actually included in the PCK at the given folder level are returned. In practice, this means that since imported resources are stored in a top-level [code].godot/[/code] folder, only paths to [code]*.gd[/code] and [code]*.import[/code] files are returned (plus a few files such as [code]project.godot[/code] or [code]project.binary[/code] and the project icon). In an exported project, the list of returned files will also vary depending on whether [member ProjectSettings.editor/export/convert_text_resources_to_binary] is [code]true[/code].
func get_files() -> PackedStringArray:
	pass;

## Returns a [PackedStringArray] containing filenames of the directory contents, excluding directories, at the given [param path]. The array is sorted alphabetically.
## Use [method get_files] if you want more control of what gets included.
static func get_files_at(path: String) -> PackedStringArray:
	pass;

## Returns the next element (file or directory) in the current directory.
## The name of the file or directory is returned (and not its full path). Once the stream has been fully processed, the method returns an empty [String] and closes the stream automatically (i.e. [method list_dir_end] would not be mandatory in such a case).
func get_next() -> String:
	pass;

## Returns the result of the last [method open] call in the current thread.
static func get_open_error() -> int:
	pass;

## Returns the available space on the current directory's disk, in bytes. Returns [code]0[/code] if the platform-specific method to query the available space fails.
func get_space_left() -> int:
	pass;

## Returns [code]true[/code] if the file system or directory use case sensitive file names.
## [b]Note:[/b] This method is implemented on macOS, Linux (for EXT4 and F2FS filesystems only) and Windows. On other platforms, it always returns [code]true[/code].
func is_case_sensitive(path: String) -> bool:
	pass;

## Returns [code]true[/code] if the file or directory is a symbolic link, directory junction, or other reparse point.
## [b]Note:[/b] This method is implemented on macOS, Linux, and Windows.
func is_link(path: String) -> bool:
	pass;

## Initializes the stream used to list all files and directories using the [method get_next] function, closing the currently opened stream if needed. Once the stream has been processed, it should typically be closed with [method list_dir_end].
## Affected by [member include_hidden] and [member include_navigational].
## [b]Note:[/b] The order of files and directories returned by this method is not deterministic, and can vary between operating systems. If you want a list of all files or folders sorted alphabetically, use [method get_files] or [method get_directories].
func list_dir_begin() -> int:
	pass;

## Closes the current stream opened with [method list_dir_begin] (whether it has been fully processed with [method get_next] does not matter).
func list_dir_end() -> void:
	pass;

## Creates a directory. The argument can be relative to the current directory, or an absolute path. The target directory should be placed in an already existing directory (to create the full path recursively, see [method make_dir_recursive]).
## Returns one of the [enum Error] code constants ([constant OK] on success).
func make_dir(path: String) -> int:
	pass;

## Static version of [method make_dir]. Supports only absolute paths.
static func make_dir_absolute(path: String) -> int:
	pass;

## Creates a target directory and all necessary intermediate directories in its path, by calling [method make_dir] recursively. The argument can be relative to the current directory, or an absolute path.
## Returns one of the [enum Error] code constants ([constant OK] on success).
func make_dir_recursive(path: String) -> int:
	pass;

## Static version of [method make_dir_recursive]. Supports only absolute paths.
static func make_dir_recursive_absolute(path: String) -> int:
	pass;

## Creates a new [DirAccess] object and opens an existing directory of the filesystem. The [param path] argument can be within the project tree ([code]res://folder[/code]), the user directory ([code]user://folder[/code]) or an absolute path of the user filesystem (e.g. [code]/tmp/folder[/code] or [code]C:\tmp\folder[/code]).
## Returns [code]null[/code] if opening the directory failed. You can use [method get_open_error] to check the error that occurred.
static func open(path: String) -> DirAccess:
	pass;

## Returns target of the symbolic link.
## [b]Note:[/b] This method is implemented on macOS, Linux, and Windows.
func read_link(path: String) -> String:
	pass;

## Permanently deletes the target file or an empty directory. The argument can be relative to the current directory, or an absolute path. If the target directory is not empty, the operation will fail.
## If you don't want to delete the file/directory permanently, use [method OS.move_to_trash] instead.
## Returns one of the [enum Error] code constants ([constant OK] on success).
func remove(path: String) -> int:
	pass;

## Static version of [method remove]. Supports only absolute paths.
static func remove_absolute(path: String) -> int:
	pass;

## Renames (move) the [param from] file or directory to the [param to] destination. Both arguments should be paths to files or directories, either relative or absolute. If the destination file or directory exists and is not access-protected, it will be overwritten.
## Returns one of the [enum Error] code constants ([constant OK] on success).
func rename(from: String, to: String) -> int:
	pass;

## Static version of [method rename]. Supports only absolute paths.
static func rename_absolute(from: String, to: String) -> int:
	pass;


func get_include_hidden() -> bool:
	return include_hidden

func set_include_hidden(value: bool) -> void:
	include_hidden = value

func get_include_navigational() -> bool:
	return include_navigational

func set_include_navigational(value: bool) -> void:
	include_navigational = value

