extends Node
class_name CanvasItem

## Abstract base class for everything in 2D space.
##
## Abstract base class for everything in 2D space. Canvas items are laid out in a tree; children inherit and extend their parent's transform. [CanvasItem] is extended by [Control] for GUI-related nodes, and by [Node2D] for 2D game objects.
## Any [CanvasItem] can draw. For this, [method queue_redraw] is called by the engine, then [constant NOTIFICATION_DRAW] will be received on idle time to request a redraw. Because of this, canvas items don't need to be redrawn on every frame, improving the performance significantly. Several functions for drawing on the [CanvasItem] are provided (see [code]draw_*[/code] functions). However, they can only be used inside [method _draw], its corresponding [method Object._notification] or methods connected to the [signal draw] signal.
## Canvas items are drawn in tree order on their canvas layer. By default, children are on top of their parents, so a root [CanvasItem] will be drawn behind everything. This behavior can be changed on a per-item basis.
## A [CanvasItem] can be hidden, which will also hide its children. By adjusting various other properties of a [CanvasItem], you can also modulate its color (via [member modulate] or [member self_modulate]), change its Z-index, blend mode, and more.
## Note that properties like transform, modulation, and visibility are only propagated to [i]direct[/i] [CanvasItem] child nodes. If there is a non-[CanvasItem] node in between, like [Node] or [AnimationPlayer], the [CanvasItem] nodes below will have an independent position and [member modulate] chain. See also [member top_level].
##
## @tutorial(Viewport and canvas transforms): https://docs.godotengine.org/en/stable/tutorials/2d/2d_transforms.html
## @tutorial(Custom drawing in 2D): https://docs.godotengine.org/en/stable/tutorials/2d/custom_drawing_in_2d.html
## @tutorial(Audio Spectrum Visualizer Demo): https://godotengine.org/asset-library/asset/2762


## Emitted when the [CanvasItem] must redraw, [i]after[/i] the related [constant NOTIFICATION_DRAW] notification, and [i]before[/i] [method _draw] is called.
## [b]Note:[/b] Deferred connections do not allow drawing through the [code]draw_*[/code] methods.
signal draw
## Emitted when becoming hidden.
signal hidden
## Emitted when the item's [Rect2] boundaries (position or size) have changed, or when an action is taking place that may have impacted these boundaries (e.g. changing [member Sprite2D.texture]).
signal item_rect_changed
## Emitted when the visibility (hidden/visible) changes.
signal visibility_changed
## The [CanvasItem]'s global transform has changed. This notification is only received if enabled by [method set_notify_transform].
const NOTIFICATION_TRANSFORM_CHANGED = 2000;

## The [CanvasItem]'s local transform has changed. This notification is only received if enabled by [method set_notify_local_transform].
const NOTIFICATION_LOCAL_TRANSFORM_CHANGED = 35;

## The [CanvasItem] is requested to draw (see [method _draw]).
const NOTIFICATION_DRAW = 30;

## The [CanvasItem]'s visibility has changed.
const NOTIFICATION_VISIBILITY_CHANGED = 31;

## The [CanvasItem] has entered the canvas.
const NOTIFICATION_ENTER_CANVAS = 32;

## The [CanvasItem] has exited the canvas.
const NOTIFICATION_EXIT_CANVAS = 33;

## The [CanvasItem]'s active [World2D] changed.
const NOTIFICATION_WORLD_2D_CHANGED = 36;

## The [CanvasItem] will inherit the filter from its parent.
## The texture filter reads from the nearest pixel only. This makes the texture look pixelated from up close, and grainy from a distance (due to mipmaps not being sampled).
## The texture filter blends between the nearest 4 pixels. This makes the texture look smooth from up close, and grainy from a distance (due to mipmaps not being sampled).
## The texture filter reads from the nearest pixel and blends between the nearest 2 mipmaps (or uses the nearest mipmap if [member ProjectSettings.rendering/textures/default_filters/use_nearest_mipmap_filter] is [code]true[/code]). This makes the texture look pixelated from up close, and smooth from a distance.
## Use this for non-pixel art textures that may be viewed at a low scale (e.g. due to [Camera2D] zoom or sprite scaling), as mipmaps are important to smooth out pixels that are smaller than on-screen pixels.
## The texture filter blends between the nearest 4 pixels and between the nearest 2 mipmaps (or uses the nearest mipmap if [member ProjectSettings.rendering/textures/default_filters/use_nearest_mipmap_filter] is [code]true[/code]). This makes the texture look smooth from up close, and smooth from a distance.
## Use this for non-pixel art textures that may be viewed at a low scale (e.g. due to [Camera2D] zoom or sprite scaling), as mipmaps are important to smooth out pixels that are smaller than on-screen pixels.
## The texture filter reads from the nearest pixel and blends between 2 mipmaps (or uses the nearest mipmap if [member ProjectSettings.rendering/textures/default_filters/use_nearest_mipmap_filter] is [code]true[/code]) based on the angle between the surface and the camera view. This makes the texture look pixelated from up close, and smooth from a distance. Anisotropic filtering improves texture quality on surfaces that are almost in line with the camera, but is slightly slower. The anisotropic filtering level can be changed by adjusting [member ProjectSettings.rendering/textures/default_filters/anisotropic_filtering_level].
## [b]Note:[/b] This texture filter is rarely useful in 2D projects. [constant TEXTURE_FILTER_NEAREST_WITH_MIPMAPS] is usually more appropriate in this case.
## The texture filter blends between the nearest 4 pixels and blends between 2 mipmaps (or uses the nearest mipmap if [member ProjectSettings.rendering/textures/default_filters/use_nearest_mipmap_filter] is [code]true[/code]) based on the angle between the surface and the camera view. This makes the texture look smooth from up close, and smooth from a distance. Anisotropic filtering improves texture quality on surfaces that are almost in line with the camera, but is slightly slower. The anisotropic filtering level can be changed by adjusting [member ProjectSettings.rendering/textures/default_filters/anisotropic_filtering_level].
## [b]Note:[/b] This texture filter is rarely useful in 2D projects. [constant TEXTURE_FILTER_LINEAR_WITH_MIPMAPS] is usually more appropriate in this case.
## Represents the size of the [enum TextureFilter] enum.
## The [CanvasItem] will inherit the filter from its parent.
## Texture will not repeat.
## Texture will repeat normally.
## Texture will repeat in a 2×2 tiled mode, where elements at even positions are mirrored.
## Represents the size of the [enum TextureRepeat] enum.
## Child draws over parent and is not clipped.
## Parent is used for the purposes of clipping only. Child is clipped to the parent's visible area, parent is not drawn.
## Parent is used for clipping child, but parent is also drawn underneath child as normal before clipping child to its visible area.
## Represents the size of the [enum ClipChildrenMode] enum.

#enum TextureFilter
enum {
    TEXTURE_FILTER_PARENT_NODE = 0,
    TEXTURE_FILTER_NEAREST = 1,
    TEXTURE_FILTER_LINEAR = 2,
    TEXTURE_FILTER_NEAREST_WITH_MIPMAPS = 3,
    TEXTURE_FILTER_LINEAR_WITH_MIPMAPS = 4,
    TEXTURE_FILTER_NEAREST_WITH_MIPMAPS_ANISOTROPIC = 5,
    TEXTURE_FILTER_LINEAR_WITH_MIPMAPS_ANISOTROPIC = 6,
    TEXTURE_FILTER_MAX = 7,
}
#enum TextureRepeat
enum {
    TEXTURE_REPEAT_PARENT_NODE = 0,
    TEXTURE_REPEAT_DISABLED = 1,
    TEXTURE_REPEAT_ENABLED = 2,
    TEXTURE_REPEAT_MIRROR = 3,
    TEXTURE_REPEAT_MAX = 4,
}
#enum ClipChildrenMode
enum {
    CLIP_CHILDREN_DISABLED = 0,
    CLIP_CHILDREN_ONLY = 1,
    CLIP_CHILDREN_AND_DRAW = 2,
    CLIP_CHILDREN_MAX = 3,
}
## Allows the current node to clip child nodes, essentially acting as a mask.
var clip_children: int:
	get = get_clip_children_mode, set = set_clip_children_mode

## The rendering layers in which this [CanvasItem] responds to [Light2D] nodes.
var light_mask: int:
	get = get_light_mask, set = set_light_mask

## The material applied to this [CanvasItem].
var material: Material:
	get = get_material, set = set_material

## The color applied to this [CanvasItem]. This property does affect child [CanvasItem]s, unlike [member self_modulate] which only affects the node itself.
var modulate: Color:
	get = get_modulate, set = set_modulate

## The color applied to this [CanvasItem]. This property does [b]not[/b] affect child [CanvasItem]s, unlike [member modulate] which affects both the node itself and its children.
## [b]Note:[/b] Internal children (e.g. sliders in [ColorPicker] or tab bar in [TabContainer]) are also not affected by this property (see [code]include_internal[/code] parameter of [method Node.get_child] and other similar methods).
var self_modulate: Color:
	get = get_self_modulate, set = set_self_modulate

## If [code]true[/code], the object draws behind its parent.
var show_behind_parent: bool:
	get = is_draw_behind_parent_enabled, set = set_draw_behind_parent

## The texture filtering mode to use on this [CanvasItem].
var texture_filter: int:
	get = get_texture_filter, set = set_texture_filter

## The texture repeating mode to use on this [CanvasItem].
var texture_repeat: int:
	get = get_texture_repeat, set = set_texture_repeat

## If [code]true[/code], this [CanvasItem] will [i]not[/i] inherit its transform from parent [CanvasItem]s. Its draw order will also be changed to make it draw on top of other [CanvasItem]s that do not have [member top_level] set to [code]true[/code]. The [CanvasItem] will effectively act as if it was placed as a child of a bare [Node].
var top_level: bool:
	get = is_set_as_top_level, set = set_as_top_level

## If [code]true[/code], the parent [CanvasItem]'s [member material] property is used as this one's material.
var use_parent_material: bool:
	get = get_use_parent_material, set = set_use_parent_material

## The rendering layer in which this [CanvasItem] is rendered by [Viewport] nodes. A [Viewport] will render a [CanvasItem] if it and all its parents share a layer with the [Viewport]'s canvas cull mask.
var visibility_layer: int:
	get = get_visibility_layer, set = set_visibility_layer

## If [code]true[/code], this [CanvasItem] is drawn. The node is only visible if all of its ancestors are visible as well (in other words, [method is_visible_in_tree] must return [code]true[/code]).
## [b]Note:[/b] For controls that inherit [Popup], the correct way to make them visible is to call one of the multiple [code]popup*()[/code] functions instead.
var visible: bool:
	get = is_visible, set = set_visible

## If [code]true[/code], this and child [CanvasItem] nodes with a higher Y position are rendered in front of nodes with a lower Y position. If [code]false[/code], this and child [CanvasItem] nodes are rendered normally in scene tree order.
## With Y-sorting enabled on a parent node ('A') but disabled on a child node ('B'), the child node ('B') is sorted but its children ('C1', 'C2', etc) render together on the same Y position as the child node ('B'). This allows you to organize the render order of a scene without changing the scene tree.
## Nodes sort relative to each other only if they are on the same [member z_index].
var y_sort_enabled: bool:
	get = is_y_sort_enabled, set = set_y_sort_enabled

## If [code]true[/code], the node's Z index is relative to its parent's Z index. If this node's Z index is 2 and its parent's effective Z index is 3, then this node's effective Z index will be 2 + 3 = 5.
var z_as_relative: bool:
	get = is_z_relative, set = set_z_as_relative

## Controls the order in which the nodes render. A node with a higher Z index will display in front of others. Must be between [constant RenderingServer.CANVAS_ITEM_Z_MIN] and [constant RenderingServer.CANVAS_ITEM_Z_MAX] (inclusive).
## [b]Note:[/b] Changing the Z index of a [Control] only affects the drawing order, not the order in which input events are handled. This can be useful to implement certain UI animations, e.g. a menu where hovered items are scaled and should overlap others.
var z_index: int:
	get = get_z_index, set = set_z_index



## Called when [CanvasItem] has been requested to redraw (after [method queue_redraw] is called, either manually or by the engine).
## Corresponds to the [constant NOTIFICATION_DRAW] notification in [method Object._notification].
func _draw() -> void:
	pass;

## Subsequent drawing commands will be ignored unless they fall within the specified animation slice. This is a faster way to implement animations that loop on background rather than redrawing constantly.
func draw_animation_slice(animation_length: float, slice_begin: float, slice_end: float, offset: float = 0.0) -> void:
	pass;

## Draws an unfilled arc between the given angles with a uniform [param color] and [param width] and optional antialiasing (supported only for positive [param width]). The larger the value of [param point_count], the smoother the curve. See also [method draw_circle].
## If [param width] is negative, it will be ignored and the arc will be drawn using [constant RenderingServer.PRIMITIVE_LINE_STRIP]. This means that when the CanvasItem is scaled, the arc will remain thin. If this behavior is not desired, then pass a positive [param width] like [code]1.0[/code].
## The arc is drawn from [param start_angle] towards the value of [param end_angle] so in clockwise direction if [code]start_angle < end_angle[/code] and counter-clockwise otherwise. Passing the same angles but in reversed order will produce the same arc. If absolute difference of [param start_angle] and [param end_angle] is greater than [constant @GDScript.TAU] radians, then a full circle arc is drawn (i.e. arc will not overlap itself).
func draw_arc(center: Vector2, radius: float, start_angle: float, end_angle: float, point_count: int, color: Color, width: float = -1.0, antialiased: bool = false) -> void:
	pass;

## Draws a string first character using a custom font.
func draw_char(font: Font, pos: Vector2, char: String, font_size: int = 16, modulate: Color = Color(1, 1, 1, 1)) -> void:
	pass;

## Draws a string first character outline using a custom font.
func draw_char_outline(font: Font, pos: Vector2, char: String, font_size: int = 16, size: int = -1, modulate: Color = Color(1, 1, 1, 1)) -> void:
	pass;

## Draws a circle. See also [method draw_arc], [method draw_polyline], and [method draw_polygon].
## If [param filled] is [code]true[/code], the circle will be filled with the [param color] specified. If [param filled] is [code]false[/code], the circle will be drawn as a stroke with the [param color] and [param width] specified.
## If [param width] is negative, then two-point primitives will be drawn instead of a four-point ones. This means that when the CanvasItem is scaled, the lines will remain thin. If this behavior is not desired, then pass a positive [param width] like [code]1.0[/code].
## If [param antialiased] is [code]true[/code], half transparent "feathers" will be attached to the boundary, making outlines smooth.
## [b]Note:[/b] [param width] is only effective if [param filled] is [code]false[/code].
func draw_circle(position: Vector2, radius: float, color: Color, filled: bool = true, width: float = -1.0, antialiased: bool = false) -> void:
	pass;

## Draws a colored polygon of any number of points, convex or concave. Unlike [method draw_polygon], a single color must be specified for the whole polygon.
func draw_colored_polygon(points: PackedVector2Array, color: Color, uvs: PackedVector2Array = PackedVector2Array(), texture: Texture2D = null) -> void:
	pass;

## Draws a dashed line from a 2D point to another, with a given color and width. See also [method draw_multiline] and [method draw_polyline].
## If [param width] is negative, then a two-point primitives will be drawn instead of a four-point ones. This means that when the CanvasItem is scaled, the line parts will remain thin. If this behavior is not desired, then pass a positive [param width] like [code]1.0[/code].
## If [param antialiased] is [code]true[/code], half transparent "feathers" will be attached to the boundary, making outlines smooth.
## [b]Note:[/b] [param antialiased] is only effective if [param width] is greater than [code]0.0[/code].
func draw_dashed_line(from: Vector2, to: Vector2, color: Color, width: float = -1.0, dash: float = 2.0, aligned: bool = true, antialiased: bool = false) -> void:
	pass;

## After submitting all animations slices via [method draw_animation_slice], this function can be used to revert drawing to its default state (all subsequent drawing commands will be visible). If you don't care about this particular use case, usage of this function after submitting the slices is not required.
func draw_end_animation() -> void:
	pass;

## Draws a textured rectangle region of the font texture with LCD subpixel anti-aliasing at a given position, optionally modulated by a color.
## Texture is drawn using the following blend operation, blend mode of the [CanvasItemMaterial] is ignored:
## [codeblock]
## dst.r = texture.r * modulate.r * modulate.a + dst.r * (1.0 - texture.r * modulate.a);
## dst.g = texture.g * modulate.g * modulate.a + dst.g * (1.0 - texture.g * modulate.a);
## dst.b = texture.b * modulate.b * modulate.a + dst.b * (1.0 - texture.b * modulate.a);
## dst.a = modulate.a + dst.a * (1.0 - modulate.a);
## [/codeblock]
func draw_lcd_texture_rect_region(texture: Texture2D, rect: Rect2, src_rect: Rect2, modulate: Color = Color(1, 1, 1, 1)) -> void:
	pass;

## Draws a line from a 2D point to another, with a given color and width. It can be optionally antialiased. See also [method draw_multiline] and [method draw_polyline].
## If [param width] is negative, then a two-point primitive will be drawn instead of a four-point one. This means that when the CanvasItem is scaled, the line will remain thin. If this behavior is not desired, then pass a positive [param width] like [code]1.0[/code].
func draw_line(from: Vector2, to: Vector2, color: Color, width: float = -1.0, antialiased: bool = false) -> void:
	pass;

## Draws a [Mesh] in 2D, using the provided texture. See [MeshInstance2D] for related documentation.
func draw_mesh(mesh: Mesh, texture: Texture2D, transform: Transform2D = Transform2D(1, 0, 0, 1, 0, 0), modulate: Color = Color(1, 1, 1, 1)) -> void:
	pass;

## Draws a textured rectangle region of the multi-channel signed distance field texture at a given position, optionally modulated by a color. See [member FontFile.multichannel_signed_distance_field] for more information and caveats about MSDF font rendering.
## If [param outline] is positive, each alpha channel value of pixel in region is set to maximum value of true distance in the [param outline] radius.
## Value of the [param pixel_range] should the same that was used during distance field texture generation.
func draw_msdf_texture_rect_region(texture: Texture2D, rect: Rect2, src_rect: Rect2, modulate: Color = Color(1, 1, 1, 1), outline: float = 0.0, pixel_range: float = 4.0, scale: float = 1.0) -> void:
	pass;

## Draws multiple disconnected lines with a uniform [param width] and [param color]. Each line is defined by two consecutive points from [param points] array, i.e. i-th segment consists of [code]points[2 * i][/code], [code]points[2 * i + 1][/code] endpoints. When drawing large amounts of lines, this is faster than using individual [method draw_line] calls. To draw interconnected lines, use [method draw_polyline] instead.
## If [param width] is negative, then two-point primitives will be drawn instead of a four-point ones. This means that when the CanvasItem is scaled, the lines will remain thin. If this behavior is not desired, then pass a positive [param width] like [code]1.0[/code].
## [b]Note:[/b] [param antialiased] is only effective if [param width] is greater than [code]0.0[/code].
func draw_multiline(points: PackedVector2Array, color: Color, width: float = -1.0, antialiased: bool = false) -> void:
	pass;

## Draws multiple disconnected lines with a uniform [param width] and segment-by-segment coloring. Each segment is defined by two consecutive points from [param points] array and a corresponding color from [param colors] array, i.e. i-th segment consists of [code]points[2 * i][/code], [code]points[2 * i + 1][/code] endpoints and has [code]colors[i][/code] color. When drawing large amounts of lines, this is faster than using individual [method draw_line] calls. To draw interconnected lines, use [method draw_polyline_colors] instead.
## If [param width] is negative, then two-point primitives will be drawn instead of a four-point ones. This means that when the CanvasItem is scaled, the lines will remain thin. If this behavior is not desired, then pass a positive [param width] like [code]1.0[/code].
## [b]Note:[/b] [param antialiased] is only effective if [param width] is greater than [code]0.0[/code].
func draw_multiline_colors(points: PackedVector2Array, colors: PackedColorArray, width: float = -1.0, antialiased: bool = false) -> void:
	pass;

## Breaks [param text] into lines and draws it using the specified [param font] at the [param pos] (top-left corner). The text will have its color multiplied by [param modulate]. If [param width] is greater than or equal to 0, the text will be clipped if it exceeds the specified width.
func draw_multiline_string(font: Font, pos: Vector2, text: String, alignment: int = 0, width: float = -1, font_size: int = 16, max_lines: int = -1, modulate: Color = Color(1, 1, 1, 1), brk_flags: int = 3, justification_flags: int = 3, direction: int = 0, orientation: int = 0) -> void:
	pass;

## Breaks [param text] to the lines and draws text outline using the specified [param font] at the [param pos] (top-left corner). The text will have its color multiplied by [param modulate]. If [param width] is greater than or equal to 0, the text will be clipped if it exceeds the specified width.
func draw_multiline_string_outline(font: Font, pos: Vector2, text: String, alignment: int = 0, width: float = -1, font_size: int = 16, max_lines: int = -1, size: int = 1, modulate: Color = Color(1, 1, 1, 1), brk_flags: int = 3, justification_flags: int = 3, direction: int = 0, orientation: int = 0) -> void:
	pass;

## Draws a [MultiMesh] in 2D with the provided texture. See [MultiMeshInstance2D] for related documentation.
func draw_multimesh(multimesh: MultiMesh, texture: Texture2D) -> void:
	pass;

## Draws a solid polygon of any number of points, convex or concave. Unlike [method draw_colored_polygon], each point's color can be changed individually. See also [method draw_polyline] and [method draw_polyline_colors]. If you need more flexibility (such as being able to use bones), use [method RenderingServer.canvas_item_add_triangle_array] instead.
func draw_polygon(points: PackedVector2Array, colors: PackedColorArray, uvs: PackedVector2Array = PackedVector2Array(), texture: Texture2D = null) -> void:
	pass;

## Draws interconnected line segments with a uniform [param color] and [param width] and optional antialiasing (supported only for positive [param width]). When drawing large amounts of lines, this is faster than using individual [method draw_line] calls. To draw disconnected lines, use [method draw_multiline] instead. See also [method draw_polygon].
## If [param width] is negative, it will be ignored and the polyline will be drawn using [constant RenderingServer.PRIMITIVE_LINE_STRIP]. This means that when the CanvasItem is scaled, the polyline will remain thin. If this behavior is not desired, then pass a positive [param width] like [code]1.0[/code].
func draw_polyline(points: PackedVector2Array, color: Color, width: float = -1.0, antialiased: bool = false) -> void:
	pass;

## Draws interconnected line segments with a uniform [param width], point-by-point coloring, and optional antialiasing (supported only for positive [param width]). Colors assigned to line points match by index between [param points] and [param colors], i.e. each line segment is filled with a gradient between the colors of the endpoints. When drawing large amounts of lines, this is faster than using individual [method draw_line] calls. To draw disconnected lines, use [method draw_multiline_colors] instead. See also [method draw_polygon].
## If [param width] is negative, it will be ignored and the polyline will be drawn using [constant RenderingServer.PRIMITIVE_LINE_STRIP]. This means that when the CanvasItem is scaled, the polyline will remain thin. If this behavior is not desired, then pass a positive [param width] like [code]1.0[/code].
func draw_polyline_colors(points: PackedVector2Array, colors: PackedColorArray, width: float = -1.0, antialiased: bool = false) -> void:
	pass;

## Draws a custom primitive. 1 point for a point, 2 points for a line, 3 points for a triangle, and 4 points for a quad. If 0 points or more than 4 points are specified, nothing will be drawn and an error message will be printed. See also [method draw_line], [method draw_polyline], [method draw_polygon], and [method draw_rect].
func draw_primitive(points: PackedVector2Array, colors: PackedColorArray, uvs: PackedVector2Array, texture: Texture2D = null) -> void:
	pass;

## Draws a rectangle. If [param filled] is [code]true[/code], the rectangle will be filled with the [param color] specified. If [param filled] is [code]false[/code], the rectangle will be drawn as a stroke with the [param color] and [param width] specified. See also [method draw_texture_rect].
## If [param width] is negative, then two-point primitives will be drawn instead of a four-point ones. This means that when the CanvasItem is scaled, the lines will remain thin. If this behavior is not desired, then pass a positive [param width] like [code]1.0[/code].
## If [param antialiased] is [code]true[/code], half transparent "feathers" will be attached to the boundary, making outlines smooth.
## [b]Note:[/b] [param width] is only effective if [param filled] is [code]false[/code].
## [b]Note:[/b] Unfilled rectangles drawn with a negative [param width] may not display perfectly. For example, corners may be missing or brighter due to overlapping lines (for a translucent [param color]).
func draw_rect(rect: Rect2, color: Color, filled: bool = true, width: float = -1.0, antialiased: bool = false) -> void:
	pass;

## Sets a custom transform for drawing via components. Anything drawn afterwards will be transformed by this.
## [b]Note:[/b] [member FontFile.oversampling] does [i]not[/i] take [param scale] into account. This means that scaling up/down will cause bitmap fonts and rasterized (non-MSDF) dynamic fonts to appear blurry or pixelated. To ensure text remains crisp regardless of scale, you can enable MSDF font rendering by enabling [member ProjectSettings.gui/theme/default_font_multichannel_signed_distance_field] (applies to the default project font only), or enabling [b]Multichannel Signed Distance Field[/b] in the import options of a DynamicFont for custom fonts. On system fonts, [member SystemFont.multichannel_signed_distance_field] can be enabled in the inspector.
func draw_set_transform(position: Vector2, rotation: float = 0.0, scale: Vector2 = Vector2(1, 1)) -> void:
	pass;

## Sets a custom transform for drawing via matrix. Anything drawn afterwards will be transformed by this.
func draw_set_transform_matrix(xform: Transform2D) -> void:
	pass;

## Draws [param text] using the specified [param font] at the [param pos] (bottom-left corner using the baseline of the font). The text will have its color multiplied by [param modulate]. If [param width] is greater than or equal to 0, the text will be clipped if it exceeds the specified width.
## [b]Example using the default project font:[/b]
## [codeblocks]
## [gdscript]
## # If using this method in a script that redraws constantly, move the
## # `default_font` declaration to a member variable assigned in `_ready()`
## # so the Control is only created once.
## var default_font = ThemeDB.fallback_font
## var default_font_size = ThemeDB.fallback_font_size
## draw_string(default_font, Vector2(64, 64), "Hello world", HORIZONTAL_ALIGNMENT_LEFT, -1, default_font_size)
## [/gdscript]
## [csharp]
## // If using this method in a script that redraws constantly, move the
## // `default_font` declaration to a member variable assigned in `_Ready()`
## // so the Control is only created once.
## Font defaultFont = ThemeDB.FallbackFont;
## int defaultFontSize = ThemeDB.FallbackFontSize;
## DrawString(defaultFont, new Vector2(64, 64), "Hello world", HORIZONTAL_ALIGNMENT_LEFT, -1, defaultFontSize);
## [/csharp]
## [/codeblocks]
## See also [method Font.draw_string].
func draw_string(font: Font, pos: Vector2, text: String, alignment: int = 0, width: float = -1, font_size: int = 16, modulate: Color = Color(1, 1, 1, 1), justification_flags: int = 3, direction: int = 0, orientation: int = 0) -> void:
	pass;

## Draws [param text] outline using the specified [param font] at the [param pos] (bottom-left corner using the baseline of the font). The text will have its color multiplied by [param modulate]. If [param width] is greater than or equal to 0, the text will be clipped if it exceeds the specified width.
func draw_string_outline(font: Font, pos: Vector2, text: String, alignment: int = 0, width: float = -1, font_size: int = 16, size: int = 1, modulate: Color = Color(1, 1, 1, 1), justification_flags: int = 3, direction: int = 0, orientation: int = 0) -> void:
	pass;

## Draws a styled rectangle.
func draw_style_box(style_box: StyleBox, rect: Rect2) -> void:
	pass;

## Draws a texture at a given position.
func draw_texture(texture: Texture2D, position: Vector2, modulate: Color = Color(1, 1, 1, 1)) -> void:
	pass;

## Draws a textured rectangle at a given position, optionally modulated by a color. If [param transpose] is [code]true[/code], the texture will have its X and Y coordinates swapped. See also [method draw_rect] and [method draw_texture_rect_region].
func draw_texture_rect(texture: Texture2D, rect: Rect2, tile: bool, modulate: Color = Color(1, 1, 1, 1), transpose: bool = false) -> void:
	pass;

## Draws a textured rectangle from a texture's region (specified by [param src_rect]) at a given position, optionally modulated by a color. If [param transpose] is [code]true[/code], the texture will have its X and Y coordinates swapped. See also [method draw_texture_rect].
func draw_texture_rect_region(texture: Texture2D, rect: Rect2, src_rect: Rect2, modulate: Color = Color(1, 1, 1, 1), transpose: bool = false, clip_uv: bool = true) -> void:
	pass;

## Forces the transform to update. Transform changes in physics are not instant for performance reasons. Transforms are accumulated and then set. Use this if you need an up-to-date transform when doing physics operations.
func force_update_transform() -> void:
	pass;

## Returns the [RID] of the [World2D] canvas where this item is in.
func get_canvas() -> RID:
	pass;

## Returns the canvas item RID used by [RenderingServer] for this item.
func get_canvas_item() -> RID:
	pass;

## Returns the [CanvasLayer] that contains this node, or [code]null[/code] if the node is not in any [CanvasLayer].
func get_canvas_layer_node() -> CanvasLayer:
	pass;

## Returns the transform from the coordinate system of the canvas, this item is in, to the [Viewport]s coordinate system.
func get_canvas_transform() -> Transform2D:
	pass;

## Returns the mouse's position in the [CanvasLayer] that this [CanvasItem] is in using the coordinate system of the [CanvasLayer].
## [b]Note:[/b] For screen-space coordinates (e.g. when using a non-embedded [Popup]), you can use [method DisplayServer.mouse_get_position].
func get_global_mouse_position() -> Vector2:
	pass;

## Returns the global transform matrix of this item, i.e. the combined transform up to the topmost [CanvasItem] node. The topmost item is a [CanvasItem] that either has no parent, has non-[CanvasItem] parent or it has [member top_level] enabled.
func get_global_transform() -> Transform2D:
	pass;

## Returns the transform from the local coordinate system of this [CanvasItem] to the [Viewport]s coordinate system.
func get_global_transform_with_canvas() -> Transform2D:
	pass;

## Returns the mouse's position in this [CanvasItem] using the local coordinate system of this [CanvasItem].
func get_local_mouse_position() -> Vector2:
	pass;

## Returns the transform of this [CanvasItem] in global screen coordinates (i.e. taking window position into account). Mostly useful for editor plugins.
## Equals to [method get_global_transform] if the window is embedded (see [member Viewport.gui_embed_subwindows]).
func get_screen_transform() -> Transform2D:
	pass;

## Returns the transform matrix of this item.
func get_transform() -> Transform2D:
	pass;

## Returns the viewport's boundaries as a [Rect2].
func get_viewport_rect() -> Rect2:
	pass;

## Returns the transform from the coordinate system of the canvas, this item is in, to the [Viewport]s embedders coordinate system.
func get_viewport_transform() -> Transform2D:
	pass;

## Returns an individual bit on the rendering visibility layer.
func get_visibility_layer_bit(layer: int) -> bool:
	pass;

## Returns the [World2D] where this item is in.
func get_world_2d() -> World2D:
	pass;

## Hide the [CanvasItem] if it's currently visible. This is equivalent to setting [member visible] to [code]false[/code].
func hide() -> void:
	pass;

## Returns [code]true[/code] if local transform notifications are communicated to children.
func is_local_transform_notification_enabled() -> bool:
	pass;

## Returns [code]true[/code] if global transform notifications are communicated to children.
func is_transform_notification_enabled() -> bool:
	pass;

## Returns [code]true[/code] if the node is present in the [SceneTree], its [member visible] property is [code]true[/code] and all its ancestors are also visible. If any ancestor is hidden, this node will not be visible in the scene tree, and is therefore not drawn (see [method _draw]).
## Visibility is checked only in parent nodes that inherit from [CanvasItem], [CanvasLayer], and [Window]. If the parent is of any other type (such as [Node], [AnimationPlayer], or [Node3D]), it is assumed to be visible.
func is_visible_in_tree() -> bool:
	pass;

## Assigns [param screen_point] as this node's new local transform.
func make_canvas_position_local(screen_point: Vector2) -> Vector2:
	pass;

## Transformations issued by [param event]'s inputs are applied in local space instead of global space.
func make_input_local(event: InputEvent) -> InputEvent:
	pass;

## Moves this node to display on top of its siblings.
## Internally, the node is moved to the bottom of parent's child list. The method has no effect on nodes without a parent.
func move_to_front() -> void:
	pass;

## Queues the [CanvasItem] to redraw. During idle time, if [CanvasItem] is visible, [constant NOTIFICATION_DRAW] is sent and [method _draw] is called. This only occurs [b]once[/b] per frame, even if this method has been called multiple times.
func queue_redraw() -> void:
	pass;

## If [param enable] is [code]true[/code], this node will receive [constant NOTIFICATION_LOCAL_TRANSFORM_CHANGED] when its local transform changes.
func set_notify_local_transform(enable: bool) -> void:
	pass;

## If [param enable] is [code]true[/code], this node will receive [constant NOTIFICATION_TRANSFORM_CHANGED] when its global transform changes.
func set_notify_transform(enable: bool) -> void:
	pass;

## Set/clear individual bits on the rendering visibility layer. This simplifies editing this [CanvasItem]'s visibility layer.
func set_visibility_layer_bit(layer: int, enabled: bool) -> void:
	pass;

## Show the [CanvasItem] if it's currently hidden. This is equivalent to setting [member visible] to [code]true[/code]. For controls that inherit [Popup], the correct way to make them visible is to call one of the multiple [code]popup*()[/code] functions instead.
func show() -> void:
	pass;


func get_clip_children_mode() -> int:
	return clip_children

func set_clip_children_mode(value: int) -> void:
	clip_children = value

func get_light_mask() -> int:
	return light_mask

func set_light_mask(value: int) -> void:
	light_mask = value

func get_material() -> Material:
	return material

func set_material(value: Material) -> void:
	material = value

func get_modulate() -> Color:
	return modulate

func set_modulate(value: Color) -> void:
	modulate = value

func get_self_modulate() -> Color:
	return self_modulate

func set_self_modulate(value: Color) -> void:
	self_modulate = value

func is_draw_behind_parent_enabled() -> bool:
	return show_behind_parent

func set_draw_behind_parent(value: bool) -> void:
	show_behind_parent = value

func get_texture_filter() -> int:
	return texture_filter

func set_texture_filter(value: int) -> void:
	texture_filter = value

func get_texture_repeat() -> int:
	return texture_repeat

func set_texture_repeat(value: int) -> void:
	texture_repeat = value

func is_set_as_top_level() -> bool:
	return top_level

func set_as_top_level(value: bool) -> void:
	top_level = value

func get_use_parent_material() -> bool:
	return use_parent_material

func set_use_parent_material(value: bool) -> void:
	use_parent_material = value

func get_visibility_layer() -> int:
	return visibility_layer

func set_visibility_layer(value: int) -> void:
	visibility_layer = value

func is_visible() -> bool:
	return visible

func set_visible(value: bool) -> void:
	visible = value

func is_y_sort_enabled() -> bool:
	return y_sort_enabled

func set_y_sort_enabled(value: bool) -> void:
	y_sort_enabled = value

func is_z_relative() -> bool:
	return z_as_relative

func set_z_as_relative(value: bool) -> void:
	z_as_relative = value

func get_z_index() -> int:
	return z_index

func set_z_index(value: int) -> void:
	z_index = value

