class_name Callable

## A built-in type representing a method or a standalone function.
##
## [Callable] is a built-in [Variant] type that represents a function. It can either be a method within an [Object] instance, or a custom callable used for different purposes (see [method is_custom]). Like all [Variant] types, it can be stored in variables and passed to other functions. It is most commonly used for signal callbacks.
## [b]Example:[/b]
## [codeblocks]
## [gdscript]
## func print_args(arg1, arg2, arg3 = ""):
## prints(arg1, arg2, arg3)
## func test():
## var callable = Callable(self, "print_args")
## callable.call("hello", "world")  # Prints "hello world ".
## callable.call(Vector2.UP, 42, callable)  # Prints "(0, -1) 42 Node(node.gd)::print_args".
## callable.call("invalid")  # Invalid call, should have at least 2 arguments.
## [/gdscript]
## [csharp]
## // Default parameter values are not supported.
## public void PrintArgs(Variant arg1, Variant arg2, Variant arg3 = default)
## {
## GD.PrintS(arg1, arg2, arg3);
## }
## public void Test()
## {
## // Invalid calls fail silently.
## Callable callable = new Callable(this, MethodName.PrintArgs);
## callable.Call("hello", "world"); // Default parameter values are not supported, should have 3 arguments.
## callable.Call(Vector2.Up, 42, callable); // Prints "(0, -1) 42 Node(Node.cs)::PrintArgs".
## callable.Call("invalid"); // Invalid call, should have 3 arguments.
## }
## [/csharp]
## [/codeblocks]
## In GDScript, it's possible to create lambda functions within a method. Lambda functions are custom callables that are not associated with an [Object] instance. Optionally, lambda functions can also be named. The name will be displayed in the debugger, or when calling [method get_method].
## [codeblock]
## func _init():
## var my_lambda = func (message):
## print(message)
## # Prints Hello everyone!
## my_lambda.call("Hello everyone!")
## # Prints "Attack!", when the button_pressed signal is emitted.
## button_pressed.connect(func(): print("Attack!"))
## [/codeblock]
## In GDScript, you can access methods and global functions as [Callable]s:
## [codeblock]
## tween.tween_callback(node.queue_free)  # Object methods.
## tween.tween_callback(array.clear)  # Methods of built-in types.
## tween.tween_callback(print.bind("Test"))  # Global functions.
## [/codeblock]
## [b]Note:[/b] [Dictionary] does not support the above due to ambiguity with keys.
## [codeblock]
## var dictionary = {"hello": "world"}
## # This will not work, `clear` is treated as a key.
## tween.tween_callback(dictionary.clear)
## # This will work.
## tween.tween_callback(Callable.create(dictionary, "clear"))
## [/codeblock]




## Constructs an empty [Callable], with no object nor method bound.
func Callable() -> Callable:
	pass;

## Constructs a [Callable] as a copy of the given [Callable].
func Callable(from: Callable) -> Callable:
	pass;

## Creates a new [Callable] for the method named [param method] in the specified [param object].
## [b]Note:[/b] For methods of built-in [Variant] types, use [method create] instead.
func Callable(object: Object, method: StringName) -> Callable:
	pass;


## Returns a copy of this [Callable] with one or more arguments bound. When called, the bound arguments are passed [i]after[/i] the arguments supplied by [method call]. See also [method unbind].
## [b]Note:[/b] When this method is chained with other similar methods, the order in which the argument list is modified is read from right to left.
vararg func bind() -> Callable:
	pass;

## Returns a copy of this [Callable] with one or more arguments bound, reading them from an array. When called, the bound arguments are passed [i]after[/i] the arguments supplied by [method call]. See also [method unbind].
## [b]Note:[/b] When this method is chained with other similar methods, the order in which the argument list is modified is read from right to left.
func bindv(arguments: Array) -> Callable:
	pass;

## Calls the method represented by this [Callable]. Arguments can be passed and should match the method's signature.
vararg func call() -> Variant:
	pass;

## Calls the method represented by this [Callable] in deferred mode, i.e. at the end of the current frame. Arguments can be passed and should match the method's signature.
## [codeblocks]
## [gdscript]
## func _ready():
## grab_focus.call_deferred()
## [/gdscript]
## [csharp]
## public override void _Ready()
## {
## Callable.From(GrabFocus).CallDeferred();
## }
## [/csharp]
## [/codeblocks]
## [b]Note:[/b] Deferred calls are processed at idle time. Idle time happens mainly at the end of process and physics frames. In it, deferred calls will be run until there are none left, which means you can defer calls from other deferred calls and they'll still be run in the current idle time cycle. This means you should not call a method deferred from itself (or from a method called by it), as this causes infinite recursion the same way as if you had called the method directly.
## See also [method Object.call_deferred].
vararg func call_deferred() -> void:
	pass;

## Calls the method represented by this [Callable]. Unlike [method call], this method expects all arguments to be contained inside the [param arguments] [Array].
func callv(arguments: Array) -> Variant:
	pass;

## Creates a new [Callable] for the method named [param method] in the specified [param variant]. To represent a method of a built-in [Variant] type, a custom callable is used (see [method is_custom]). If [param variant] is [Object], then a standard callable will be created instead.
## [b]Note:[/b] This method is always necessary for the [Dictionary] type, as property syntax is used to access its entries. You may also use this method when [param variant]'s type is not known in advance (for polymorphism).
static func create(variant: Variant, method: StringName) -> Callable:
	pass;

## Returns the total number of arguments this [Callable] should take, including optional arguments. This means that any arguments bound with [method bind] are [i]subtracted[/i] from the result, and any arguments unbound with [method unbind] are [i]added[/i] to the result.
func get_argument_count() -> int:
	pass;

## Return the bound arguments (as long as [method get_bound_arguments_count] is greater than zero), or empty (if [method get_bound_arguments_count] is less than or equal to zero).
func get_bound_arguments() -> Array:
	pass;

## Returns the total amount of arguments bound (or unbound) via successive [method bind] or [method unbind] calls. If the amount of arguments unbound is greater than the ones bound, this function returns a value less than zero.
func get_bound_arguments_count() -> int:
	pass;

## Returns the name of the method represented by this [Callable]. If the callable is a GDScript lambda function, returns the function's name or [code]"<anonymous lambda>"[/code].
func get_method() -> StringName:
	pass;

## Returns the object on which this [Callable] is called.
func get_object() -> Object:
	pass;

## Returns the ID of this [Callable]'s object (see [method Object.get_instance_id]).
func get_object_id() -> int:
	pass;

## Returns the 32-bit hash value of this [Callable]'s object.
## [b]Note:[/b] [Callable]s with equal content will always produce identical hash values. However, the reverse is not true. Returning identical hash values does [i]not[/i] imply the callables are equal, because different callables can have identical hash values due to hash collisions. The engine uses a 32-bit hash algorithm for [method hash].
func hash() -> int:
	pass;

## Returns [code]true[/code] if this [Callable] is a custom callable. Custom callables are used:
## - for binding/unbinding arguments (see [method bind] and [method unbind]);
## - for representing methods of built-in [Variant] types (see [method create]);
## - for representing global, lambda, and RPC functions in GDScript;
## - for other purposes in the core, GDExtension, and C#.
func is_custom() -> bool:
	pass;

## Returns [code]true[/code] if this [Callable] has no target to call the method on.
func is_null() -> bool:
	pass;

## Returns [code]true[/code] if this [Callable] is a standard callable. This method is the opposite of [method is_custom]. Returns [code]false[/code] if this callable is a lambda function.
func is_standard() -> bool:
	pass;

## Returns [code]true[/code] if the callable's object exists and has a valid method name assigned, or is a custom callable.
func is_valid() -> bool:
	pass;

## Perform an RPC (Remote Procedure Call) on all connected peers. This is used for multiplayer and is normally not available, unless the function being called has been marked as [i]RPC[/i] (using [annotation @GDScript.@rpc] or [method Node.rpc_config]). Calling this method on unsupported functions will result in an error. See [method Node.rpc].
vararg func rpc() -> void:
	pass;

## Perform an RPC (Remote Procedure Call) on a specific peer ID (see multiplayer documentation for reference). This is used for multiplayer and is normally not available unless the function being called has been marked as [i]RPC[/i] (using [annotation @GDScript.@rpc] or [method Node.rpc_config]). Calling this method on unsupported functions will result in an error. See [method Node.rpc_id].
vararg func rpc_id(peer_id: int) -> void:
	pass;

## Returns a copy of this [Callable] with a number of arguments unbound. In other words, when the new callable is called the last few arguments supplied by the user are ignored, according to [param argcount]. The remaining arguments are passed to the callable. This allows to use the original callable in a context that attempts to pass more arguments than this callable can handle, e.g. a signal with a fixed number of arguments. See also [method bind].
## [b]Note:[/b] When this method is chained with other similar methods, the order in which the argument list is modified is read from right to left.
## [codeblock]
## func _ready():
## foo.unbind(1).call(1, 2) # Calls foo(1).
## foo.bind(3, 4).unbind(1).call(1, 2) # Calls foo(1, 3, 4), note that it does not change the arguments from bind.
## [/codeblock]
func unbind(argcount: int) -> Callable:
	pass;


