extends Mesh
class_name ArrayMesh

## [Mesh] type that provides utility for constructing a surface from arrays.
##
## The [ArrayMesh] is used to construct a [Mesh] by specifying the attributes as arrays.
## The most basic example is the creation of a single triangle:
## [codeblocks]
## [gdscript]
## var vertices = PackedVector3Array()
## vertices.push_back(Vector3(0, 1, 0))
## vertices.push_back(Vector3(1, 0, 0))
## vertices.push_back(Vector3(0, 0, 1))
## # Initialize the ArrayMesh.
## var arr_mesh = ArrayMesh.new()
## var arrays = []
## arrays.resize(Mesh.ARRAY_MAX)
## arrays[Mesh.ARRAY_VERTEX] = vertices
## # Create the Mesh.
## arr_mesh.add_surface_from_arrays(Mesh.PRIMITIVE_TRIANGLES, arrays)
## var m = MeshInstance3D.new()
## m.mesh = arr_mesh
## [/gdscript]
## [csharp]
## var vertices = new Vector3[]
## {
## new Vector3(0, 1, 0),
## new Vector3(1, 0, 0),
## new Vector3(0, 0, 1),
## };
## // Initialize the ArrayMesh.
## var arrMesh = new ArrayMesh();
## var arrays = new Godot.Collections.Array();
## arrays.Resize((int)Mesh.ArrayType.Max);
## arrays[(int)Mesh.ArrayType.Vertex] = vertices;
## // Create the Mesh.
## arrMesh.AddSurfaceFromArrays(Mesh.PrimitiveType.Triangles, arrays);
## var m = new MeshInstance3D();
## m.Mesh = arrMesh;
## [/csharp]
## [/codeblocks]
## The [MeshInstance3D] is ready to be added to the [SceneTree] to be shown.
## See also [ImmediateMesh], [MeshDataTool] and [SurfaceTool] for procedural geometry generation.
## [b]Note:[/b] Godot uses clockwise [url=https://learnopengl.com/Advanced-OpenGL/Face-culling]winding order[/url] for front faces of triangle primitive modes.
##
## @tutorial(Procedural geometry using the ArrayMesh): https://docs.godotengine.org/en/stable/tutorials/3d/procedural_geometry/arraymesh.html



## Sets the blend shape mode to one of [enum Mesh.BlendShapeMode].
var blend_shape_mode: int:
	get = get_blend_shape_mode, set = set_blend_shape_mode

## Overrides the [AABB] with one defined by user for use with frustum culling. Especially useful to avoid unexpected culling when using a shader to offset vertices.
var custom_aabb: AABB:
	get = get_custom_aabb, set = set_custom_aabb

## An optional mesh which can be used for rendering shadows and the depth prepass. Can be used to increase performance by supplying a mesh with fused vertices and only vertex position data (without normals, UVs, colors, etc.).
## [b]Note:[/b] This mesh must have exactly the same vertex positions as the source mesh (including the source mesh's LODs, if present). If vertex positions differ, then the mesh will not draw correctly.
var shadow_mesh: ArrayMesh:
	get = get_shadow_mesh, set = set_shadow_mesh



## Adds name for a blend shape that will be added with [method add_surface_from_arrays]. Must be called before surface is added.
func add_blend_shape(name: StringName) -> void:
	pass;

## Creates a new surface. [method Mesh.get_surface_count] will become the [code]surf_idx[/code] for this new surface.
## Surfaces are created to be rendered using a [param primitive], which may be any of the values defined in [enum Mesh.PrimitiveType].
## The [param arrays] argument is an array of arrays. Each of the [constant Mesh.ARRAY_MAX] elements contains an array with some of the mesh data for this surface as described by the corresponding member of [enum Mesh.ArrayType] or [code]null[/code] if it is not used by the surface. For example, [code]arrays[0][/code] is the array of vertices. That first vertex sub-array is always required; the others are optional. Adding an index array puts this surface into "index mode" where the vertex and other arrays become the sources of data and the index array defines the vertex order. All sub-arrays must have the same length as the vertex array (or be an exact multiple of the vertex array's length, when multiple elements of a sub-array correspond to a single vertex) or be empty, except for [constant Mesh.ARRAY_INDEX] if it is used.
## The [param blend_shapes] argument is an array of vertex data for each blend shape. Each element is an array of the same structure as [param arrays], but [constant Mesh.ARRAY_VERTEX], [constant Mesh.ARRAY_NORMAL], and [constant Mesh.ARRAY_TANGENT] are set if and only if they are set in [param arrays] and all other entries are [code]null[/code].
## The [param lods] argument is a dictionary with [float] keys and [PackedInt32Array] values. Each entry in the dictionary represents an LOD level of the surface, where the value is the [constant Mesh.ARRAY_INDEX] array to use for the LOD level and the key is roughly proportional to the distance at which the LOD stats being used. I.e., increasing the key of an LOD also increases the distance that the objects has to be from the camera before the LOD is used.
## The [param flags] argument is the bitwise or of, as required: One value of [enum Mesh.ArrayCustomFormat] left shifted by [code]ARRAY_FORMAT_CUSTOMn_SHIFT[/code] for each custom channel in use, [constant Mesh.ARRAY_FLAG_USE_DYNAMIC_UPDATE], [constant Mesh.ARRAY_FLAG_USE_8_BONE_WEIGHTS], or [constant Mesh.ARRAY_FLAG_USES_EMPTY_VERTEX_ARRAY].
## [b]Note:[/b] When using indices, it is recommended to only use points, lines, or triangles.
func add_surface_from_arrays(primitive: int, arrays: Array, blend_shapes: Array[Array] = [], lods: Dictionary = {}, flags: int = 0) -> void:
	pass;

## Removes all blend shapes from this [ArrayMesh].
func clear_blend_shapes() -> void:
	pass;

## Removes all surfaces from this [ArrayMesh].
func clear_surfaces() -> void:
	pass;

## Returns the number of blend shapes that the [ArrayMesh] holds.
func get_blend_shape_count() -> int:
	pass;

## Returns the name of the blend shape at this index.
func get_blend_shape_name(index: int) -> StringName:
	pass;

## Performs a UV unwrap on the [ArrayMesh] to prepare the mesh for lightmapping.
func lightmap_unwrap(transform: Transform3D, texel_size: float) -> int:
	pass;

## Regenerates tangents for each of the [ArrayMesh]'s surfaces.
func regen_normal_maps() -> void:
	pass;

## Sets the name of the blend shape at this index.
func set_blend_shape_name(index: int, name: StringName) -> void:
	pass;

## Returns the index of the first surface with this name held within this [ArrayMesh]. If none are found, -1 is returned.
func surface_find_by_name(name: String) -> int:
	pass;

## Returns the length in indices of the index array in the requested surface (see [method add_surface_from_arrays]).
func surface_get_array_index_len(surf_idx: int) -> int:
	pass;

## Returns the length in vertices of the vertex array in the requested surface (see [method add_surface_from_arrays]).
func surface_get_array_len(surf_idx: int) -> int:
	pass;

## Returns the format mask of the requested surface (see [method add_surface_from_arrays]).
func surface_get_format(surf_idx: int) -> int:
	pass;

## Gets the name assigned to this surface.
func surface_get_name(surf_idx: int) -> String:
	pass;

## Returns the primitive type of the requested surface (see [method add_surface_from_arrays]).
func surface_get_primitive_type(surf_idx: int) -> int:
	pass;

## Sets a name for a given surface.
func surface_set_name(surf_idx: int, name: String) -> void:
	pass;

func surface_update_attribute_region(surf_idx: int, offset: int, data: PackedByteArray) -> void:
	pass;

func surface_update_skin_region(surf_idx: int, offset: int, data: PackedByteArray) -> void:
	pass;

func surface_update_vertex_region(surf_idx: int, offset: int, data: PackedByteArray) -> void:
	pass;


func get_blend_shape_mode() -> int:
	return blend_shape_mode

func set_blend_shape_mode(value: int) -> void:
	blend_shape_mode = value

func get_custom_aabb() -> AABB:
	return custom_aabb

func set_custom_aabb(value: AABB) -> void:
	custom_aabb = value

func get_shadow_mesh() -> ArrayMesh:
	return shadow_mesh

func set_shadow_mesh(value: ArrayMesh) -> void:
	shadow_mesh = value

