extends Resource
class_name AnimationNode

## Base class for [AnimationTree] nodes. Not related to scene nodes.
##
## Base resource for [AnimationTree] nodes. In general, it's not used directly, but you can create custom ones with custom blending formulas.
## Inherit this when creating animation nodes mainly for use in [AnimationNodeBlendTree], otherwise [AnimationRootNode] should be used instead.
## You can access the time information as read-only parameter which is processed and stored in the previous frame for all nodes except [AnimationNodeOutput].
## [b]Note:[/b] If multiple inputs exist in the [AnimationNode], which time information takes precedence depends on the type of [AnimationNode].
## [codeblock]
## var current_length = $AnimationTree[parameters/AnimationNodeName/current_length]
## var current_position = $AnimationTree[parameters/AnimationNodeName/current_position]
## var current_delta = $AnimationTree[parameters/AnimationNodeName/current_delta]
## [/codeblock]
##
## @tutorial(Using AnimationTree): https://docs.godotengine.org/en/stable/tutorials/animation/animation_tree.html


## Emitted by nodes that inherit from this class and that have an internal tree when one of their animation nodes removes. The animation nodes that emit this signal are [AnimationNodeBlendSpace1D], [AnimationNodeBlendSpace2D], [AnimationNodeStateMachine], and [AnimationNodeBlendTree].
signal animation_node_removed(object_id: int, name: String)
## Emitted by nodes that inherit from this class and that have an internal tree when one of their animation node names changes. The animation nodes that emit this signal are [AnimationNodeBlendSpace1D], [AnimationNodeBlendSpace2D], [AnimationNodeStateMachine], and [AnimationNodeBlendTree].
signal animation_node_renamed(object_id: int, old_name: String, new_name: String)
## Emitted by nodes that inherit from this class and that have an internal tree when one of their animation nodes changes. The animation nodes that emit this signal are [AnimationNodeBlendSpace1D], [AnimationNodeBlendSpace2D], [AnimationNodeStateMachine], [AnimationNodeBlendTree] and [AnimationNodeTransition].
signal tree_changed
## Do not use filtering.
## Paths matching the filter will be allowed to pass.
## Paths matching the filter will be discarded.
## Paths matching the filter will be blended (by the blend value).

#enum FilterAction
enum {
    FILTER_IGNORE = 0,
    FILTER_PASS = 1,
    FILTER_STOP = 2,
    FILTER_BLEND = 3,
}
## If [code]true[/code], filtering is enabled.
var filter_enabled: bool:
	get = is_filter_enabled, set = set_filter_enabled



## When inheriting from [AnimationRootNode], implement this virtual method to override the text caption for this animation node.
func _get_caption() -> String:
	pass;

## When inheriting from [AnimationRootNode], implement this virtual method to return a child animation node by its [param name].
func _get_child_by_name(name: StringName) -> AnimationNode:
	pass;

## When inheriting from [AnimationRootNode], implement this virtual method to return all child animation nodes in order as a [code]name: node[/code] dictionary.
func _get_child_nodes() -> Dictionary:
	pass;

## When inheriting from [AnimationRootNode], implement this virtual method to return the default value of a [param parameter]. Parameters are custom local memory used for your animation nodes, given a resource can be reused in multiple trees.
func _get_parameter_default_value(parameter: StringName) -> Variant:
	pass;

## When inheriting from [AnimationRootNode], implement this virtual method to return a list of the properties on this animation node. Parameters are custom local memory used for your animation nodes, given a resource can be reused in multiple trees. Format is similar to [method Object.get_property_list].
func _get_parameter_list() -> Array:
	pass;

## When inheriting from [AnimationRootNode], implement this virtual method to return whether the blend tree editor should display filter editing on this animation node.
func _has_filter() -> bool:
	pass;

## When inheriting from [AnimationRootNode], implement this virtual method to return whether the [param parameter] is read-only. Parameters are custom local memory used for your animation nodes, given a resource can be reused in multiple trees.
func _is_parameter_read_only(parameter: StringName) -> bool:
	pass;

## When inheriting from [AnimationRootNode], implement this virtual method to run some code when this animation node is processed. The [param time] parameter is a relative delta, unless [param seek] is [code]true[/code], in which case it is absolute.
## Here, call the [method blend_input], [method blend_node] or [method blend_animation] functions. You can also use [method get_parameter] and [method set_parameter] to modify local memory.
## This function should return the delta.
func _process(time: float, seek: bool, is_external_seeking: bool, test_only: bool) -> float:
	pass;

## Adds an input to the animation node. This is only useful for animation nodes created for use in an [AnimationNodeBlendTree]. If the addition fails, returns [code]false[/code].
func add_input(name: String) -> bool:
	pass;

## Blend an animation by [param blend] amount (name must be valid in the linked [AnimationPlayer]). A [param time] and [param delta] may be passed, as well as whether [param seeked] happened.
## A [param looped_flag] is used by internal processing immediately after the loop. See also [enum Animation.LoopedFlag].
func blend_animation(animation: StringName, time: float, delta: float, seeked: bool, is_external_seeking: bool, blend: float, looped_flag: int = 0) -> void:
	pass;

## Blend an input. This is only useful for animation nodes created for an [AnimationNodeBlendTree]. The [param time] parameter is a relative delta, unless [param seek] is [code]true[/code], in which case it is absolute. A filter mode may be optionally passed (see [enum FilterAction] for options).
func blend_input(input_index: int, time: float, seek: bool, is_external_seeking: bool, blend: float, filter: int = 0, sync: bool = true, test_only: bool = false) -> float:
	pass;

## Blend another animation node (in case this animation node contains child animation nodes). This function is only useful if you inherit from [AnimationRootNode] instead, otherwise editors will not display your animation node for addition.
func blend_node(name: StringName, node: AnimationNode, time: float, seek: bool, is_external_seeking: bool, blend: float, filter: int = 0, sync: bool = true, test_only: bool = false) -> float:
	pass;

## Returns the input index which corresponds to [param name]. If not found, returns [code]-1[/code].
func find_input(name: String) -> int:
	pass;

## Amount of inputs in this animation node, only useful for animation nodes that go into [AnimationNodeBlendTree].
func get_input_count() -> int:
	pass;

## Gets the name of an input by index.
func get_input_name(input: int) -> String:
	pass;

## Gets the value of a parameter. Parameters are custom local memory used for your animation nodes, given a resource can be reused in multiple trees.
func get_parameter(name: StringName) -> Variant:
	pass;

## Returns whether the given path is filtered.
func is_path_filtered(path: NodePath) -> bool:
	pass;

## Removes an input, call this only when inactive.
func remove_input(index: int) -> void:
	pass;

## Adds or removes a path for the filter.
func set_filter_path(path: NodePath, enable: bool) -> void:
	pass;

## Sets the name of the input at the given [param input] index. If the setting fails, returns [code]false[/code].
func set_input_name(input: int, name: String) -> bool:
	pass;

## Sets a custom parameter. These are used as local memory, because resources can be reused across the tree or scenes.
func set_parameter(name: StringName, value: Variant) -> void:
	pass;


func is_filter_enabled() -> bool:
	return filter_enabled

func set_filter_enabled(value: bool) -> void:
	filter_enabled = value

