extends RefCounted
class_name AStarGrid2D

## An implementation of A* for finding the shortest path between two points on a partial 2D grid.
##
## [AStarGrid2D] is a variant of [AStar2D] that is specialized for partial 2D grids. It is simpler to use because it doesn't require you to manually create points and connect them together. This class also supports multiple types of heuristics, modes for diagonal movement, and a jumping mode to speed up calculations.
## To use [AStarGrid2D], you only need to set the [member region] of the grid, optionally set the [member cell_size], and then call the [method update] method:
## [codeblocks]
## [gdscript]
## var astar_grid = AStarGrid2D.new()
## astar_grid.region = Rect2i(0, 0, 32, 32)
## astar_grid.cell_size = Vector2(16, 16)
## astar_grid.update()
## print(astar_grid.get_id_path(Vector2i(0, 0), Vector2i(3, 4))) # prints (0, 0), (1, 1), (2, 2), (3, 3), (3, 4)
## print(astar_grid.get_point_path(Vector2i(0, 0), Vector2i(3, 4))) # prints (0, 0), (16, 16), (32, 32), (48, 48), (48, 64)
## [/gdscript]
## [csharp]
## AStarGrid2D astarGrid = new AStarGrid2D();
## astarGrid.Region = new Rect2I(0, 0, 32, 32);
## astarGrid.CellSize = new Vector2I(16, 16);
## astarGrid.Update();
## GD.Print(astarGrid.GetIdPath(Vector2I.Zero, new Vector2I(3, 4))); // prints (0, 0), (1, 1), (2, 2), (3, 3), (3, 4)
## GD.Print(astarGrid.GetPointPath(Vector2I.Zero, new Vector2I(3, 4))); // prints (0, 0), (16, 16), (32, 32), (48, 48), (48, 64)
## [/csharp]
## [/codeblocks]
## To remove a point from the pathfinding grid, it must be set as "solid" with [method set_point_solid].


## The [url=https://en.wikipedia.org/wiki/Euclidean_distance]Euclidean heuristic[/url] to be used for the pathfinding using the following formula:
## [codeblock]
## dx = abs(to_id.x - from_id.x)
## dy = abs(to_id.y - from_id.y)
## result = sqrt(dx * dx + dy * dy)
## [/codeblock]
## [b]Note:[/b] This is also the internal heuristic used in [AStar3D] and [AStar2D] by default (with the inclusion of possible z-axis coordinate).
## The [url=https://en.wikipedia.org/wiki/Taxicab_geometry]Manhattan heuristic[/url] to be used for the pathfinding using the following formula:
## [codeblock]
## dx = abs(to_id.x - from_id.x)
## dy = abs(to_id.y - from_id.y)
## result = dx + dy
## [/codeblock]
## [b]Note:[/b] This heuristic is intended to be used with 4-side orthogonal movements, provided by setting the [member diagonal_mode] to [constant DIAGONAL_MODE_NEVER].
## The Octile heuristic to be used for the pathfinding using the following formula:
## [codeblock]
## dx = abs(to_id.x - from_id.x)
## dy = abs(to_id.y - from_id.y)
## f = sqrt(2) - 1
## result = (dx < dy) ? f * dx + dy : f * dy + dx;
## [/codeblock]
## The [url=https://en.wikipedia.org/wiki/Chebyshev_distance]Chebyshev heuristic[/url] to be used for the pathfinding using the following formula:
## [codeblock]
## dx = abs(to_id.x - from_id.x)
## dy = abs(to_id.y - from_id.y)
## result = max(dx, dy)
## [/codeblock]
## Represents the size of the [enum Heuristic] enum.
## The pathfinding algorithm will ignore solid neighbors around the target cell and allow passing using diagonals.
## The pathfinding algorithm will ignore all diagonals and the way will be always orthogonal.
## The pathfinding algorithm will avoid using diagonals if at least two obstacles have been placed around the neighboring cells of the specific path segment.
## The pathfinding algorithm will avoid using diagonals if any obstacle has been placed around the neighboring cells of the specific path segment.
## Represents the size of the [enum DiagonalMode] enum.
## Rectangular cell shape.
## Diamond cell shape (for isometric look). Cell coordinates layout where the horizontal axis goes up-right, and the vertical one goes down-right.
## Diamond cell shape (for isometric look). Cell coordinates layout where the horizontal axis goes down-right, and the vertical one goes down-left.
## Represents the size of the [enum CellShape] enum.

#enum Heuristic
enum {
    HEURISTIC_EUCLIDEAN = 0,
    HEURISTIC_MANHATTAN = 1,
    HEURISTIC_OCTILE = 2,
    HEURISTIC_CHEBYSHEV = 3,
    HEURISTIC_MAX = 4,
}
#enum DiagonalMode
enum {
    DIAGONAL_MODE_ALWAYS = 0,
    DIAGONAL_MODE_NEVER = 1,
    DIAGONAL_MODE_AT_LEAST_ONE_WALKABLE = 2,
    DIAGONAL_MODE_ONLY_IF_NO_OBSTACLES = 3,
    DIAGONAL_MODE_MAX = 4,
}
#enum CellShape
enum {
    CELL_SHAPE_SQUARE = 0,
    CELL_SHAPE_ISOMETRIC_RIGHT = 1,
    CELL_SHAPE_ISOMETRIC_DOWN = 2,
    CELL_SHAPE_MAX = 3,
}
## The cell shape. Affects how the positions are placed in the grid. If changed, [method update] needs to be called before finding the next path.
var cell_shape: int:
	get = get_cell_shape, set = set_cell_shape

## The size of the point cell which will be applied to calculate the resulting point position returned by [method get_point_path]. If changed, [method update] needs to be called before finding the next path.
var cell_size: Vector2:
	get = get_cell_size, set = set_cell_size

## The default [enum Heuristic] which will be used to calculate the cost between two points if [method _compute_cost] was not overridden.
var default_compute_heuristic: int:
	get = get_default_compute_heuristic, set = set_default_compute_heuristic

## The default [enum Heuristic] which will be used to calculate the cost between the point and the end point if [method _estimate_cost] was not overridden.
var default_estimate_heuristic: int:
	get = get_default_estimate_heuristic, set = set_default_estimate_heuristic

## A specific [enum DiagonalMode] mode which will force the path to avoid or accept the specified diagonals.
var diagonal_mode: int:
	get = get_diagonal_mode, set = set_diagonal_mode

## Enables or disables jumping to skip up the intermediate points and speeds up the searching algorithm.
## [b]Note:[/b] Currently, toggling it on disables the consideration of weight scaling in pathfinding.
var jumping_enabled: bool:
	get = is_jumping_enabled, set = set_jumping_enabled

## The offset of the grid which will be applied to calculate the resulting point position returned by [method get_point_path]. If changed, [method update] needs to be called before finding the next path.
var offset: Vector2:
	get = get_offset, set = set_offset

## The region of grid cells available for pathfinding. If changed, [method update] needs to be called before finding the next path.
var region: Rect2i:
	get = get_region, set = set_region

## The size of the grid (number of cells of size [member cell_size] on each axis). If changed, [method update] needs to be called before finding the next path.
var size: Vector2i:
	get = get_size, set = set_size



## Called when computing the cost between two connected points.
## Note that this function is hidden in the default [AStarGrid2D] class.
func _compute_cost(from_id: Vector2i, to_id: Vector2i) -> float:
	pass;

## Called when estimating the cost between a point and the path's ending point.
## Note that this function is hidden in the default [AStarGrid2D] class.
func _estimate_cost(from_id: Vector2i, to_id: Vector2i) -> float:
	pass;

## Clears the grid and sets the [member region] to [code]Rect2i(0, 0, 0, 0)[/code].
func clear() -> void:
	pass;

## Fills the given [param region] on the grid with the specified value for the solid flag.
## [b]Note:[/b] Calling [method update] is not needed after the call of this function.
func fill_solid_region(region: Rect2i, solid: bool = true) -> void:
	pass;

## Fills the given [param region] on the grid with the specified value for the weight scale.
## [b]Note:[/b] Calling [method update] is not needed after the call of this function.
func fill_weight_scale_region(region: Rect2i, weight_scale: float) -> void:
	pass;

## Returns an array with the IDs of the points that form the path found by AStar2D between the given points. The array is ordered from the starting point to the ending point of the path.
## If there is no valid path to the target, and [param allow_partial_path] is [code]true[/code], returns a path to the point closest to the target that can be reached.
func get_id_path(from_id: Vector2i, to_id: Vector2i, allow_partial_path: bool = false) -> Array[Vector2i]:
	pass;

## Returns an array with the points that are in the path found by [AStarGrid2D] between the given points. The array is ordered from the starting point to the ending point of the path.
## If there is no valid path to the target, and [param allow_partial_path] is [code]true[/code], returns a path to the point closest to the target that can be reached.
## [b]Note:[/b] This method is not thread-safe. If called from a [Thread], it will return an empty array and will print an error message.
func get_point_path(from_id: Vector2i, to_id: Vector2i, allow_partial_path: bool = false) -> PackedVector2Array:
	pass;

## Returns the position of the point associated with the given [param id].
func get_point_position(id: Vector2i) -> Vector2:
	pass;

## Returns the weight scale of the point associated with the given [param id].
func get_point_weight_scale(id: Vector2i) -> float:
	pass;

## Indicates that the grid parameters were changed and [method update] needs to be called.
func is_dirty() -> bool:
	pass;

## Returns [code]true[/code] if the [param x] and [param y] is a valid grid coordinate (id), i.e. if it is inside [member region]. Equivalent to [code]region.has_point(Vector2i(x, y))[/code].
func is_in_bounds(x: int, y: int) -> bool:
	pass;

## Returns [code]true[/code] if the [param id] vector is a valid grid coordinate, i.e. if it is inside [member region]. Equivalent to [code]region.has_point(id)[/code].
func is_in_boundsv(id: Vector2i) -> bool:
	pass;

## Returns [code]true[/code] if a point is disabled for pathfinding. By default, all points are enabled.
func is_point_solid(id: Vector2i) -> bool:
	pass;

## Disables or enables the specified point for pathfinding. Useful for making an obstacle. By default, all points are enabled.
## [b]Note:[/b] Calling [method update] is not needed after the call of this function.
func set_point_solid(id: Vector2i, solid: bool = true) -> void:
	pass;

## Sets the [param weight_scale] for the point with the given [param id]. The [param weight_scale] is multiplied by the result of [method _compute_cost] when determining the overall cost of traveling across a segment from a neighboring point to this point.
## [b]Note:[/b] Calling [method update] is not needed after the call of this function.
func set_point_weight_scale(id: Vector2i, weight_scale: float) -> void:
	pass;

## Updates the internal state of the grid according to the parameters to prepare it to search the path. Needs to be called if parameters like [member region], [member cell_size] or [member offset] are changed. [method is_dirty] will return [code]true[/code] if this is the case and this needs to be called.
## [b]Note:[/b] All point data (solidity and weight scale) will be cleared.
func update() -> void:
	pass;


func get_cell_shape() -> int:
	return cell_shape

func set_cell_shape(value: int) -> void:
	cell_shape = value

func get_cell_size() -> Vector2:
	return cell_size

func set_cell_size(value: Vector2) -> void:
	cell_size = value

func get_default_compute_heuristic() -> int:
	return default_compute_heuristic

func set_default_compute_heuristic(value: int) -> void:
	default_compute_heuristic = value

func get_default_estimate_heuristic() -> int:
	return default_estimate_heuristic

func set_default_estimate_heuristic(value: int) -> void:
	default_estimate_heuristic = value

func get_diagonal_mode() -> int:
	return diagonal_mode

func set_diagonal_mode(value: int) -> void:
	diagonal_mode = value

func is_jumping_enabled() -> bool:
	return jumping_enabled

func set_jumping_enabled(value: bool) -> void:
	jumping_enabled = value

func get_offset() -> Vector2:
	return offset

func set_offset(value: Vector2) -> void:
	offset = value

func get_region() -> Rect2i:
	return region

func set_region(value: Rect2i) -> void:
	region = value

func get_size() -> Vector2i:
	return size

func set_size(value: Vector2i) -> void:
	size = value

