class_name AABB

## A 3D axis-aligned bounding box.
##
## The [AABB] built-in [Variant] type represents an axis-aligned bounding box in a 3D space. It is defined by its [member position] and [member size], which are [Vector3]. It is frequently used for fast overlap tests (see [method intersects]). Although [AABB] itself is axis-aligned, it can be combined with [Transform3D] to represent a rotated or skewed bounding box.
## It uses floating-point coordinates. The 2D counterpart to [AABB] is [Rect2]. There is no version of [AABB] that uses integer coordinates.
## [b]Note:[/b] Negative values for [member size] are not supported. With negative size, most [AABB] methods do not work correctly. Use [method abs] to get an equivalent [AABB] with a non-negative size.
## [b]Note:[/b] In a boolean context, a [AABB] evaluates to [code]false[/code] if both [member position] and [member size] are zero (equal to [constant Vector3.ZERO]). Otherwise, it always evaluates to [code]true[/code].
##
## @tutorial(Math documentation index): https://docs.godotengine.org/en/stable/tutorials/math/index.html
## @tutorial(Vector math): https://docs.godotengine.org/en/stable/tutorials/math/vector_math.html
## @tutorial(Advanced vector math): https://docs.godotengine.org/en/stable/tutorials/math/vectors_advanced.html



## The ending point. This is usually the corner on the top-right and forward of the bounding box, and is equivalent to [code]position + size[/code]. Setting this point affects the [member size].
var end: Vector3

## The origin point. This is usually the corner on the bottom-left and back of the bounding box.
var position: Vector3

## The bounding box's width, height, and depth starting from [member position]. Setting this value also affects the [member end] point.
## [b]Note:[/b] It's recommended setting the width, height, and depth to non-negative values. This is because most methods in Godot assume that the [member position] is the bottom-left-back corner, and the [member end] is the top-right-forward corner. To get an equivalent bounding box with non-negative size, use [method abs].
var size: Vector3


## Constructs an [AABB] with its [member position] and [member size] set to [constant Vector3.ZERO].
func AABB() -> AABB:
	pass;

## Constructs an [AABB] as a copy of the given [AABB].
func AABB(from: AABB) -> AABB:
	pass;

## Constructs an [AABB] by [param position] and [param size].
func AABB(position: Vector3, size: Vector3) -> AABB:
	pass;


## Returns an [AABB] equivalent to this bounding box, with its width, height, and depth modified to be non-negative values.
## [codeblocks]
## [gdscript]
## var box = AABB(Vector3(5, 0, 5), Vector3(-20, -10, -5))
## var absolute = box.abs()
## print(absolute.position) # Prints (-15, -10, 0)
## print(absolute.size)     # Prints (20, 10, 5)
## [/gdscript]
## [csharp]
## var box = new Aabb(new Vector3(5, 0, 5), new Vector3(-20, -10, -5));
## var absolute = box.Abs();
## GD.Print(absolute.Position); // Prints (-15, -10, 0)
## GD.Print(absolute.Size);     // Prints (20, 10, 5)
## [/csharp]
## [/codeblocks]
## [b]Note:[/b] It's recommended to use this method when [member size] is negative, as most other methods in Godot assume that the [member size]'s components are greater than [code]0[/code].
func abs() -> AABB:
	pass;

## Returns [code]true[/code] if this bounding box [i]completely[/i] encloses the [param with] box. The edges of both boxes are included.
## [codeblocks]
## [gdscript]
## var a = AABB(Vector3(0, 0, 0), Vector3(4, 4, 4))
## var b = AABB(Vector3(1, 1, 1), Vector3(3, 3, 3))
## var c = AABB(Vector3(2, 2, 2), Vector3(8, 8, 8))
## print(a.encloses(a)) # Prints true
## print(a.encloses(b)) # Prints true
## print(a.encloses(c)) # Prints false
## [/gdscript]
## [csharp]
## var a = new Aabb(new Vector3(0, 0, 0), new Vector3(4, 4, 4));
## var b = new Aabb(new Vector3(1, 1, 1), new Vector3(3, 3, 3));
## var c = new Aabb(new Vector3(2, 2, 2), new Vector3(8, 8, 8));
## GD.Print(a.Encloses(a)); // Prints True
## GD.Print(a.Encloses(b)); // Prints True
## GD.Print(a.Encloses(c)); // Prints False
## [/csharp]
## [/codeblocks]
func encloses(with: AABB) -> bool:
	pass;

## Returns a copy of this bounding box expanded to align the edges with the given [param to_point], if necessary.
## [codeblocks]
## [gdscript]
## var box = AABB(Vector3(0, 0, 0), Vector3(5, 2, 5))
## box = box.expand(Vector3(10, 0, 0))
## print(box.position) # Prints (0, 0, 0)
## print(box.size)     # Prints (10, 2, 5)
## box = box.expand(Vector3(-5, 0, 5))
## print(box.position) # Prints (-5, 0, 0)
## print(box.size)     # Prints (15, 2, 5)
## [/gdscript]
## [csharp]
## var box = new Aabb(new Vector3(0, 0, 0), new Vector3(5, 2, 5));
## box = box.Expand(new Vector3(10, 0, 0));
## GD.Print(box.Position); // Prints (0, 0, 0)
## GD.Print(box.Size);     // Prints (10, 2, 5)
## box = box.Expand(new Vector3(-5, 0, 5));
## GD.Print(box.Position); // Prints (-5, 0, 0)
## GD.Print(box.Size);     // Prints (15, 2, 5)
## [/csharp]
## [/codeblocks]
func expand(to_point: Vector3) -> AABB:
	pass;

## Returns the center point of the bounding box. This is the same as [code]position + (size / 2.0)[/code].
func get_center() -> Vector3:
	pass;

## Returns the position of one of the 8 vertices that compose this bounding box. With a [param idx] of [code]0[/code] this is the same as [member position], and a [param idx] of [code]7[/code] is the same as [member end].
func get_endpoint(idx: int) -> Vector3:
	pass;

## Returns the longest normalized axis of this bounding box's [member size], as a [Vector3] ([constant Vector3.RIGHT], [constant Vector3.UP], or [constant Vector3.BACK]).
## [codeblocks]
## [gdscript]
## var box = AABB(Vector3(0, 0, 0), Vector3(2, 4, 8))
## print(box.get_longest_axis())       # Prints (0, 0, 1)
## print(box.get_longest_axis_index()) # Prints 2
## print(box.get_longest_axis_size())  # Prints 8
## [/gdscript]
## [csharp]
## var box = new Aabb(new Vector3(0, 0, 0), new Vector3(2, 4, 8));
## GD.Print(box.GetLongestAxis());      // Prints (0, 0, 1)
## GD.Print(box.GetLongestAxisIndex()); // Prints 2
## GD.Print(box.GetLongestAxisSize());  // Prints 8
## [/csharp]
## [/codeblocks]
## See also [method get_longest_axis_index] and [method get_longest_axis_size].
func get_longest_axis() -> Vector3:
	pass;

## Returns the index to the longest axis of this bounding box's [member size] (see [constant Vector3.AXIS_X], [constant Vector3.AXIS_Y], and [constant Vector3.AXIS_Z]).
## For an example, see [method get_longest_axis].
func get_longest_axis_index() -> int:
	pass;

## Returns the longest dimension of this bounding box's [member size].
## For an example, see [method get_longest_axis].
func get_longest_axis_size() -> float:
	pass;

## Returns the shortest normalized axis of this bounding box's [member size], as a [Vector3] ([constant Vector3.RIGHT], [constant Vector3.UP], or [constant Vector3.BACK]).
## [codeblocks]
## [gdscript]
## var box = AABB(Vector3(0, 0, 0), Vector3(2, 4, 8))
## print(box.get_shortest_axis())       # Prints (1, 0, 0)
## print(box.get_shortest_axis_index()) # Prints 0
## print(box.get_shortest_axis_size())  # Prints 2
## [/gdscript]
## [csharp]
## var box = new Aabb(new Vector3(0, 0, 0), new Vector3(2, 4, 8));
## GD.Print(box.GetShortestAxis());      // Prints (1, 0, 0)
## GD.Print(box.GetShortestAxisIndex()); // Prints 0
## GD.Print(box.GetShortestAxisSize());  // Prints 2
## [/csharp]
## [/codeblocks]
## See also [method get_shortest_axis_index] and [method get_shortest_axis_size].
func get_shortest_axis() -> Vector3:
	pass;

## Returns the index to the shortest axis of this bounding box's [member size] (see [constant Vector3.AXIS_X], [constant Vector3.AXIS_Y], and [constant Vector3.AXIS_Z]).
## For an example, see [method get_shortest_axis].
func get_shortest_axis_index() -> int:
	pass;

## Returns the shortest dimension of this bounding box's [member size].
## For an example, see [method get_shortest_axis].
func get_shortest_axis_size() -> float:
	pass;

## Returns the vertex's position of this bounding box that's the farthest in the given direction. This point is commonly known as the support point in collision detection algorithms.
func get_support(dir: Vector3) -> Vector3:
	pass;

## Returns the bounding box's volume. This is equivalent to [code]size.x * size.y * size.z[/code]. See also [method has_volume].
func get_volume() -> float:
	pass;

## Returns a copy of this bounding box extended on all sides by the given amount [param by]. A negative amount shrinks the box instead.
## [codeblocks]
## [gdscript]
## var a = AABB(Vector3(4, 4, 4), Vector3(8, 8, 8)).grow(4)
## print(a.position) # Prints (0, 0, 0)
## print(a.size)     # Prints (16, 16, 16)
## var b = AABB(Vector3(0, 0, 0), Vector3(8, 4, 2)).grow(2)
## print(b.position) # Prints (-2, -2, -2)
## print(b.size)     # Prints (12, 8, 6)
## [/gdscript]
## [csharp]
## var a = new Aabb(new Vector3(4, 4, 4), new Vector3(8, 8, 8)).Grow(4);
## GD.Print(a.Position); // Prints (0, 0, 0)
## GD.Print(a.Size);     // Prints (16, 16, 16)
## var b = new Aabb(new Vector3(0, 0, 0), new Vector3(8, 4, 2)).Grow(2);
## GD.Print(b.Position); // Prints (-2, -2, -2)
## GD.Print(b.Size);     // Prints (12, 8, 6)
## [/csharp]
## [/codeblocks]
func grow(by: float) -> AABB:
	pass;

## Returns [code]true[/code] if the bounding box contains the given [param point]. By convention, points exactly on the right, top, and front sides are [b]not[/b] included.
## [b]Note:[/b] This method is not reliable for [AABB] with a [i]negative[/i] [member size]. Use [method abs] first to get a valid bounding box.
func has_point(point: Vector3) -> bool:
	pass;

## Returns [code]true[/code] if this bounding box has a surface or a length, that is, at least one component of [member size] is greater than [code]0[/code]. Otherwise, returns [code]false[/code].
func has_surface() -> bool:
	pass;

## Returns [code]true[/code] if this bounding box's width, height, and depth are all positive. See also [method get_volume].
func has_volume() -> bool:
	pass;

## Returns the intersection between this bounding box and [param with]. If the boxes do not intersect, returns an empty [AABB]. If the boxes intersect at the edge, returns a flat [AABB] with no volume (see [method has_surface] and [method has_volume]).
## [codeblocks]
## [gdscript]
## var box1 = AABB(Vector3(0, 0, 0), Vector3(5, 2, 8))
## var box2 = AABB(Vector3(2, 0, 2), Vector3(8, 4, 4))
## var intersection = box1.intersection(box2)
## print(intersection.position) # Prints (2, 0, 2)
## print(intersection.size)     # Prints (3, 2, 4)
## [/gdscript]
## [csharp]
## var box1 = new Aabb(new Vector3(0, 0, 0), new Vector3(5, 2, 8));
## var box2 = new Aabb(new Vector3(2, 0, 2), new Vector3(8, 4, 4));
## var intersection = box1.Intersection(box2);
## GD.Print(intersection.Position); // Prints (2, 0, 2)
## GD.Print(intersection.Size);     // Prints (3, 2, 4)
## [/csharp]
## [/codeblocks]
## [b]Note:[/b] If you only need to know whether two bounding boxes are intersecting, use [method intersects], instead.
func intersection(with: AABB) -> AABB:
	pass;

## Returns [code]true[/code] if this bounding box overlaps with the box [param with]. The edges of both boxes are [i]always[/i] excluded.
func intersects(with: AABB) -> bool:
	pass;

## Returns [code]true[/code] if this bounding box is on both sides of the given [param plane].
func intersects_plane(plane: Plane) -> bool:
	pass;

## Returns the first point where this bounding box and the given ray intersect, as a [Vector3]. If no intersection occurs, returns [code]null[/code].
## The ray begin at [param from], faces [param dir] and extends towards infinity.
func intersects_ray(from: Vector3, dir: Vector3) -> Variant:
	pass;

## Returns the first point where this bounding box and the given segment intersect, as a [Vector3]. If no intersection occurs, returns [code]null[/code].
## The segment begins at [param from] and ends at [param to].
func intersects_segment(from: Vector3, to: Vector3) -> Variant:
	pass;

## Returns [code]true[/code] if this bounding box and [param aabb] are approximately equal, by calling [method Vector2.is_equal_approx] on the [member position] and the [member size].
func is_equal_approx(aabb: AABB) -> bool:
	pass;

## Returns [code]true[/code] if this bounding box's values are finite, by calling [method Vector2.is_finite] on the [member position] and the [member size].
func is_finite() -> bool:
	pass;

## Returns an [AABB] that encloses both this bounding box and [param with] around the edges. See also [method encloses].
func merge(with: AABB) -> AABB:
	pass;


