extends Control
class_name Tree

## A control used to show a set of internal [TreeItem]s in a hierarchical structure.
##
## A control used to show a set of internal [TreeItem]s in a hierarchical structure. The tree items can be selected, expanded and collapsed. The tree can have multiple columns with custom controls like [LineEdit]s, buttons and popups. It can be useful for structured displays and interactions.
## Trees are built via code, using [TreeItem] objects to create the structure. They have a single root, but multiple roots can be simulated with [member hide_root]:
## [codeblocks]
## [gdscript]
## func _ready():
## var tree = Tree.new()
## var root = tree.create_item()
## tree.hide_root = true
## var child1 = tree.create_item(root)
## var child2 = tree.create_item(root)
## var subchild1 = tree.create_item(child1)
## subchild1.set_text(0, "Subchild1")
## [/gdscript]
## [csharp]
## public override void _Ready()
## {
## var tree = new Tree();
## TreeItem root = tree.CreateItem();
## tree.HideRoot = true;
## TreeItem child1 = tree.CreateItem(root);
## TreeItem child2 = tree.CreateItem(root);
## TreeItem subchild1 = tree.CreateItem(child1);
## subchild1.SetText(0, "Subchild1");
## }
## [/csharp]
## [/codeblocks]
## To iterate over all the [TreeItem] objects in a [Tree] object, use [method TreeItem.get_next] and [method TreeItem.get_first_child] after getting the root through [method get_root]. You can use [method Object.free] on a [TreeItem] to remove it from the [Tree].
## [b]Incremental search:[/b] Like [ItemList] and [PopupMenu], [Tree] supports searching within the list while the control is focused. Press a key that matches the first letter of an item's name to select the first item starting with the given letter. After that point, there are two ways to perform incremental search: 1) Press the same key again before the timeout duration to select the next item starting with the same letter. 2) Press letter keys that match the rest of the word before the timeout duration to match to select the item in question directly. Both of these actions will be reset to the beginning of the list if the timeout duration has passed since the last keystroke was registered. You can adjust the timeout duration by changing [member ProjectSettings.gui/timers/incremental_search_max_interval_msec].


## Emitted when a button on the tree was pressed (see [method TreeItem.add_button]).
signal button_clicked(item: TreeItem, column: int, id: int, mouse_button_index: int)
## Emitted when a cell is selected.
signal cell_selected
## Emitted when [method TreeItem.propagate_check] is called. Connect to this signal to process the items that are affected when [method TreeItem.propagate_check] is invoked. The order that the items affected will be processed is as follows: the item that invoked the method, children of that item, and finally parents of that item.
signal check_propagated_to_item(item: TreeItem, column: int)
## Emitted when a column's title is clicked with either [constant MOUSE_BUTTON_LEFT] or [constant MOUSE_BUTTON_RIGHT].
signal column_title_clicked(column: int, mouse_button_index: int)
## Emitted when an item with [constant TreeItem.CELL_MODE_CUSTOM] is clicked with a mouse button.
signal custom_item_clicked(mouse_button_index: int)
## Emitted when a cell with the [constant TreeItem.CELL_MODE_CUSTOM] is clicked to be edited.
signal custom_popup_edited(arrow_clicked: bool)
## Emitted when a mouse button is clicked in the empty space of the tree.
signal empty_clicked(position: Vector2, mouse_button_index: int)
## Emitted when an item is double-clicked, or selected with a [code]ui_accept[/code] input event (e.g. using [kbd]Enter[/kbd] or [kbd]Space[/kbd] on the keyboard).
signal item_activated
## Emitted when an item is collapsed by a click on the folding arrow.
signal item_collapsed(item: TreeItem)
## Emitted when an item is edited.
signal item_edited
## Emitted when an item's icon is double-clicked. For a signal that emits when any part of the item is double-clicked, see [signal item_activated].
signal item_icon_double_clicked
## Emitted when an item is selected with a mouse button.
signal item_mouse_selected(position: Vector2, mouse_button_index: int)
## Emitted when an item is selected.
signal item_selected
## Emitted instead of [signal item_selected] if [member select_mode] is set to [constant SELECT_MULTI].
signal multi_selected(item: TreeItem, column: int, selected: bool)
## Emitted when a left mouse button click does not select any item.
signal nothing_selected
## Allows selection of a single cell at a time. From the perspective of items, only a single item is allowed to be selected. And there is only one column selected in the selected item.
## The focus cursor is always hidden in this mode, but it is positioned at the current selection, making the currently selected item the currently focused item.
## Allows selection of a single row at a time. From the perspective of items, only a single items is allowed to be selected. And all the columns are selected in the selected item.
## The focus cursor is always hidden in this mode, but it is positioned at the first column of the current selection, making the currently selected item the currently focused item.
## Allows selection of multiple cells at the same time. From the perspective of items, multiple items are allowed to be selected. And there can be multiple columns selected in each selected item.
## The focus cursor is visible in this mode, the item or column under the cursor is not necessarily selected.
## Disables all drop sections, but still allows to detect the "on item" drop section by [method get_drop_section_at_position].
## [b]Note:[/b] This is the default flag, it has no effect when combined with other flags.
## Enables the "on item" drop section. This drop section covers the entire item.
## When combined with [constant DROP_MODE_INBETWEEN], this drop section halves the height and stays centered vertically.
## Enables "above item" and "below item" drop sections. The "above item" drop section covers the top half of the item, and the "below item" drop section covers the bottom half.
## When combined with [constant DROP_MODE_ON_ITEM], these drop sections halves the height and stays on top / bottom accordingly.

#enum SelectMode
enum {
    SELECT_SINGLE = 0,
    SELECT_ROW = 1,
    SELECT_MULTI = 2,
}
#enum DropModeFlags
enum {
    DROP_MODE_DISABLED = 0,
    DROP_MODE_ON_ITEM = 1,
    DROP_MODE_INBETWEEN = 2,
}
## If [code]true[/code], the currently selected cell may be selected again.
var allow_reselect: bool:
	get = get_allow_reselect, set = set_allow_reselect

## If [code]true[/code], a right mouse button click can select items.
var allow_rmb_select: bool:
	get = get_allow_rmb_select, set = set_allow_rmb_select

## If [code]true[/code], allows navigating the [Tree] with letter keys through incremental search.
var allow_search: bool:
	get = get_allow_search, set = set_allow_search

var clip_contents: bool:
	get = is_clipping_contents, set = set_clip_contents

## If [code]true[/code], column titles are visible.
var column_titles_visible: bool:
	get = are_column_titles_visible, set = set_column_titles_visible

## The number of columns.
var columns: int:
	get = get_columns, set = set_columns

## The drop mode as an OR combination of flags. See [enum DropModeFlags] constants. Once dropping is done, reverts to [constant DROP_MODE_DISABLED]. Setting this during [method Control._can_drop_data] is recommended.
## This controls the drop sections, i.e. the decision and drawing of possible drop locations based on the mouse position.
var drop_mode_flags: int:
	get = get_drop_mode_flags, set = set_drop_mode_flags

## If [code]true[/code], recursive folding is enabled for this [Tree]. Holding down Shift while clicking the fold arrow collapses or uncollapses the [TreeItem] and all its descendants.
var enable_recursive_folding: bool:
	get = is_recursive_folding_enabled, set = set_enable_recursive_folding

var focus_mode: int:
	get = get_focus_mode, set = set_focus_mode

## If [code]true[/code], the folding arrow is hidden.
var hide_folding: bool:
	get = is_folding_hidden, set = set_hide_folding

## If [code]true[/code], the tree's root is hidden.
var hide_root: bool:
	get = is_root_hidden, set = set_hide_root

## If [code]true[/code], enables horizontal scrolling.
var scroll_horizontal_enabled: bool:
	get = is_h_scroll_enabled, set = set_h_scroll_enabled

## If [code]true[/code], enables vertical scrolling.
var scroll_vertical_enabled: bool:
	get = is_v_scroll_enabled, set = set_v_scroll_enabled

## Allows single or multiple selection. See the [enum SelectMode] constants.
var select_mode: int:
	get = get_select_mode, set = set_select_mode



## Clears the tree. This removes all items.
func clear() -> void:
	pass;

## Creates an item in the tree and adds it as a child of [param parent], which can be either a valid [TreeItem] or [code]null[/code].
## If [param parent] is [code]null[/code], the root item will be the parent, or the new item will be the root itself if the tree is empty.
## The new item will be the [param index]-th child of parent, or it will be the last child if there are not enough siblings.
func create_item(parent: TreeItem = null, index: int = -1) -> TreeItem:
	pass;

## Deselects all tree items (rows and columns). In [constant SELECT_MULTI] mode also removes selection cursor.
func deselect_all() -> void:
	pass;

## Edits the selected tree item as if it was clicked.
## Either the item must be set editable with [method TreeItem.set_editable] or [param force_edit] must be [code]true[/code].
## Returns [code]true[/code] if the item could be edited. Fails if no item is selected.
func edit_selected(force_edit: bool = false) -> bool:
	pass;

## Makes the currently focused cell visible.
## This will scroll the tree if necessary. In [constant SELECT_ROW] mode, this will not do horizontal scrolling, as all the cells in the selected row is focused logically.
## [b]Note:[/b] Despite the name of this method, the focus cursor itself is only visible in [constant SELECT_MULTI] mode.
func ensure_cursor_is_visible() -> void:
	pass;

## Returns the button ID at [param position], or -1 if no button is there.
func get_button_id_at_position(position: Vector2) -> int:
	pass;

## Returns the column index at [param position], or -1 if no item is there.
func get_column_at_position(position: Vector2) -> int:
	pass;

## Returns the expand ratio assigned to the column.
func get_column_expand_ratio(column: int) -> int:
	pass;

## Returns the column's title.
func get_column_title(column: int) -> String:
	pass;

## Returns the column title alignment.
func get_column_title_alignment(column: int) -> int:
	pass;

## Returns column title base writing direction.
func get_column_title_direction(column: int) -> int:
	pass;

## Returns column title language code.
func get_column_title_language(column: int) -> String:
	pass;

## Returns the column's width in pixels.
func get_column_width(column: int) -> int:
	pass;

## Returns the rectangle for custom popups. Helper to create custom cell controls that display a popup. See [method TreeItem.set_cell_mode].
func get_custom_popup_rect() -> Rect2:
	pass;

## Returns the drop section at [param position], or -100 if no item is there.
## Values -1, 0, or 1 will be returned for the "above item", "on item", and "below item" drop sections, respectively. See [enum DropModeFlags] for a description of each drop section.
## To get the item which the returned drop section is relative to, use [method get_item_at_position].
func get_drop_section_at_position(position: Vector2) -> int:
	pass;

## Returns the currently edited item. Can be used with [signal item_edited] to get the item that was modified.
## [codeblocks]
## [gdscript]
## func _ready():
## $Tree.item_edited.connect(on_Tree_item_edited)
## func on_Tree_item_edited():
## print($Tree.get_edited()) # This item just got edited (e.g. checked).
## [/gdscript]
## [csharp]
## public override void _Ready()
## {
## GetNode<Tree>("Tree").ItemEdited += OnTreeItemEdited;
## }
## public void OnTreeItemEdited()
## {
## GD.Print(GetNode<Tree>("Tree").GetEdited()); // This item just got edited (e.g. checked).
## }
## [/csharp]
## [/codeblocks]
func get_edited() -> TreeItem:
	pass;

## Returns the column for the currently edited item.
func get_edited_column() -> int:
	pass;

## Returns the rectangle area for the specified [TreeItem]. If [param column] is specified, only get the position and size of that column, otherwise get the rectangle containing all columns. If a button index is specified, the rectangle of that button will be returned.
func get_item_area_rect(item: TreeItem, column: int = -1, button_index: int = -1) -> Rect2:
	pass;

## Returns the tree item at the specified position (relative to the tree origin position).
func get_item_at_position(position: Vector2) -> TreeItem:
	pass;

## Returns the next selected [TreeItem] after the given one, or [code]null[/code] if the end is reached.
## If [param from] is [code]null[/code], this returns the first selected item.
func get_next_selected(from: TreeItem) -> TreeItem:
	pass;

## Returns the last pressed button's index.
func get_pressed_button() -> int:
	pass;

## Returns the tree's root item, or [code]null[/code] if the tree is empty.
func get_root() -> TreeItem:
	pass;

## Returns the current scrolling position.
func get_scroll() -> Vector2:
	pass;

## Returns the currently focused item, or [code]null[/code] if no item is focused.
## In [constant SELECT_ROW] and [constant SELECT_SINGLE] modes, the focused item is same as the selected item. In [constant SELECT_MULTI] mode, the focused item is the item under the focus cursor, not necessarily selected.
## To get the currently selected item(s), use [method get_next_selected].
func get_selected() -> TreeItem:
	pass;

## Returns the currently focused column, or -1 if no column is focused.
## In [constant SELECT_SINGLE] mode, the focused column is the selected column. In [constant SELECT_ROW] mode, the focused column is always 0 if any item is selected. In [constant SELECT_MULTI] mode, the focused column is the column under the focus cursor, and there are not necessarily any column selected.
## To tell whether a column of an item is selected, use [method TreeItem.is_selected].
func get_selected_column() -> int:
	pass;

## Returns [code]true[/code] if the column has enabled clipping (see [method set_column_clip_content]).
func is_column_clipping_content(column: int) -> bool:
	pass;

## Returns [code]true[/code] if the column has enabled expanding (see [method set_column_expand]).
func is_column_expanding(column: int) -> bool:
	pass;

## Causes the [Tree] to jump to the specified [TreeItem].
func scroll_to_item(item: TreeItem, center_on_item: bool = false) -> void:
	pass;

## Allows to enable clipping for column's content, making the content size ignored.
func set_column_clip_content(column: int, enable: bool) -> void:
	pass;

## Overrides the calculated minimum width of a column. It can be set to [code]0[/code] to restore the default behavior. Columns that have the "Expand" flag will use their "min_width" in a similar fashion to [member Control.size_flags_stretch_ratio].
func set_column_custom_minimum_width(column: int, min_width: int) -> void:
	pass;

## If [code]true[/code], the column will have the "Expand" flag of [Control]. Columns that have the "Expand" flag will use their expand ratio in a similar fashion to [member Control.size_flags_stretch_ratio] (see [method set_column_expand_ratio]).
func set_column_expand(column: int, expand: bool) -> void:
	pass;

## Sets the relative expand ratio for a column. See [method set_column_expand].
func set_column_expand_ratio(column: int, ratio: int) -> void:
	pass;

## Sets the title of a column.
func set_column_title(column: int, title: String) -> void:
	pass;

## Sets the column title alignment. Note that [constant @GlobalScope.HORIZONTAL_ALIGNMENT_FILL] is not supported for column titles.
func set_column_title_alignment(column: int, title_alignment: int) -> void:
	pass;

## Sets column title base writing direction.
func set_column_title_direction(column: int, direction: int) -> void:
	pass;

## Sets language code of column title used for line-breaking and text shaping algorithms, if left empty current locale is used instead.
func set_column_title_language(column: int, language: String) -> void:
	pass;

## Selects the specified [TreeItem] and column.
func set_selected(item: TreeItem, column: int) -> void:
	pass;


func get_allow_reselect() -> bool:
	return allow_reselect

func set_allow_reselect(value: bool) -> void:
	allow_reselect = value

func get_allow_rmb_select() -> bool:
	return allow_rmb_select

func set_allow_rmb_select(value: bool) -> void:
	allow_rmb_select = value

func get_allow_search() -> bool:
	return allow_search

func set_allow_search(value: bool) -> void:
	allow_search = value

func is_clipping_contents() -> bool:
	return clip_contents

func set_clip_contents(value: bool) -> void:
	clip_contents = value

func are_column_titles_visible() -> bool:
	return column_titles_visible

func set_column_titles_visible(value: bool) -> void:
	column_titles_visible = value

func get_columns() -> int:
	return columns

func set_columns(value: int) -> void:
	columns = value

func get_drop_mode_flags() -> int:
	return drop_mode_flags

func set_drop_mode_flags(value: int) -> void:
	drop_mode_flags = value

func is_recursive_folding_enabled() -> bool:
	return enable_recursive_folding

func set_enable_recursive_folding(value: bool) -> void:
	enable_recursive_folding = value

func get_focus_mode() -> int:
	return focus_mode

func set_focus_mode(value: int) -> void:
	focus_mode = value

func is_folding_hidden() -> bool:
	return hide_folding

func set_hide_folding(value: bool) -> void:
	hide_folding = value

func is_root_hidden() -> bool:
	return hide_root

func set_hide_root(value: bool) -> void:
	hide_root = value

func is_h_scroll_enabled() -> bool:
	return scroll_horizontal_enabled

func set_h_scroll_enabled(value: bool) -> void:
	scroll_horizontal_enabled = value

func is_v_scroll_enabled() -> bool:
	return scroll_vertical_enabled

func set_v_scroll_enabled(value: bool) -> void:
	scroll_vertical_enabled = value

func get_select_mode() -> int:
	return select_mode

func set_select_mode(value: int) -> void:
	select_mode = value

