extends Container
class_name ScrollContainer

## A container used to provide scrollbars to a child control when needed.
##
## A container used to provide a child control with scrollbars when needed. Scrollbars will automatically be drawn at the right (for vertical) or bottom (for horizontal) and will enable dragging to move the viewable Control (and its children) within the ScrollContainer. Scrollbars will also automatically resize the grabber based on the [member Control.custom_minimum_size] of the Control relative to the ScrollContainer.
##
## @tutorial(Using Containers): https://docs.godotengine.org/en/stable/tutorials/ui/gui_containers.html


## Emitted when scrolling stops when dragging the scrollable area [i]with a touch event[/i]. This signal is [i]not[/i] emitted when scrolling by dragging the scrollbar, scrolling with the mouse wheel or scrolling with keyboard/gamepad events.
## [b]Note:[/b] This signal is only emitted on Android or iOS, or on desktop/web platforms when [member ProjectSettings.input_devices/pointing/emulate_touch_from_mouse] is enabled.
signal scroll_ended
## Emitted when scrolling starts when dragging the scrollable area w[i]ith a touch event[/i]. This signal is [i]not[/i] emitted when scrolling by dragging the scrollbar, scrolling with the mouse wheel or scrolling with keyboard/gamepad events.
## [b]Note:[/b] This signal is only emitted on Android or iOS, or on desktop/web platforms when [member ProjectSettings.input_devices/pointing/emulate_touch_from_mouse] is enabled.
signal scroll_started
## Scrolling disabled, scrollbar will be invisible.
## Scrolling enabled, scrollbar will be visible only if necessary, i.e. container's content is bigger than the container.
## Scrolling enabled, scrollbar will be always visible.
## Scrolling enabled, scrollbar will be hidden.

#enum ScrollMode
enum {
    SCROLL_MODE_DISABLED = 0,
    SCROLL_MODE_AUTO = 1,
    SCROLL_MODE_SHOW_ALWAYS = 2,
    SCROLL_MODE_SHOW_NEVER = 3,
}
var clip_contents: bool:
	get = is_clipping_contents, set = set_clip_contents

## If [code]true[/code], the ScrollContainer will automatically scroll to focused children (including indirect children) to make sure they are fully visible.
var follow_focus: bool:
	get = is_following_focus, set = set_follow_focus

## Controls whether horizontal scrollbar can be used and when it should be visible. See [enum ScrollMode] for options.
var horizontal_scroll_mode: int:
	get = get_horizontal_scroll_mode, set = set_horizontal_scroll_mode

## Deadzone for touch scrolling. Lower deadzone makes the scrolling more sensitive.
var scroll_deadzone: int:
	get = get_deadzone, set = set_deadzone

## The current horizontal scroll value.
## [b]Note:[/b] If you are setting this value in the [method Node._ready] function or earlier, it needs to be wrapped with [method Object.set_deferred], since scroll bar's [member Range.max_value] is not initialized yet.
## [codeblock]
## func _ready():
## set_deferred("scroll_horizontal", 600)
## [/codeblock]
var scroll_horizontal: int:
	get = get_h_scroll, set = set_h_scroll

## Overrides the [member ScrollBar.custom_step] used when clicking the internal scroll bar's horizontal increment and decrement buttons or when using arrow keys when the [ScrollBar] is focused.
var scroll_horizontal_custom_step: float:
	get = get_horizontal_custom_step, set = set_horizontal_custom_step

## The current vertical scroll value.
## [b]Note:[/b] Setting it early needs to be deferred, just like in [member scroll_horizontal].
## [codeblock]
## func _ready():
## set_deferred("scroll_vertical", 600)
## [/codeblock]
var scroll_vertical: int:
	get = get_v_scroll, set = set_v_scroll

## Overrides the [member ScrollBar.custom_step] used when clicking the internal scroll bar's vertical increment and decrement buttons or when using arrow keys when the [ScrollBar] is focused.
var scroll_vertical_custom_step: float:
	get = get_vertical_custom_step, set = set_vertical_custom_step

## Controls whether vertical scrollbar can be used and when it should be visible. See [enum ScrollMode] for options.
var vertical_scroll_mode: int:
	get = get_vertical_scroll_mode, set = set_vertical_scroll_mode



## Ensures the given [param control] is visible (must be a direct or indirect child of the ScrollContainer). Used by [member follow_focus].
## [b]Note:[/b] This will not work on a node that was just added during the same frame. If you want to scroll to a newly added child, you must wait until the next frame using [signal SceneTree.process_frame]:
## [codeblock]
## add_child(child_node)
## await get_tree().process_frame
## ensure_control_visible(child_node)
## [/codeblock]
func ensure_control_visible(control: Control) -> void:
	pass;

## Returns the horizontal scrollbar [HScrollBar] of this [ScrollContainer].
## [b]Warning:[/b] This is a required internal node, removing and freeing it may cause a crash. If you wish to disable or hide a scrollbar, you can use [member horizontal_scroll_mode].
func get_h_scroll_bar() -> HScrollBar:
	pass;

## Returns the vertical scrollbar [VScrollBar] of this [ScrollContainer].
## [b]Warning:[/b] This is a required internal node, removing and freeing it may cause a crash. If you wish to disable or hide a scrollbar, you can use [member vertical_scroll_mode].
func get_v_scroll_bar() -> VScrollBar:
	pass;


func is_clipping_contents() -> bool:
	return clip_contents

func set_clip_contents(value: bool) -> void:
	clip_contents = value

func is_following_focus() -> bool:
	return follow_focus

func set_follow_focus(value: bool) -> void:
	follow_focus = value

func get_horizontal_scroll_mode() -> int:
	return horizontal_scroll_mode

func set_horizontal_scroll_mode(value: int) -> void:
	horizontal_scroll_mode = value

func get_deadzone() -> int:
	return scroll_deadzone

func set_deadzone(value: int) -> void:
	scroll_deadzone = value

func get_h_scroll() -> int:
	return scroll_horizontal

func set_h_scroll(value: int) -> void:
	scroll_horizontal = value

func get_horizontal_custom_step() -> float:
	return scroll_horizontal_custom_step

func set_horizontal_custom_step(value: float) -> void:
	scroll_horizontal_custom_step = value

func get_v_scroll() -> int:
	return scroll_vertical

func set_v_scroll(value: int) -> void:
	scroll_vertical = value

func get_vertical_custom_step() -> float:
	return scroll_vertical_custom_step

func set_vertical_custom_step(value: float) -> void:
	scroll_vertical_custom_step = value

func get_vertical_scroll_mode() -> int:
	return vertical_scroll_mode

func set_vertical_scroll_mode(value: int) -> void:
	vertical_scroll_mode = value

