extends AnimationMixer
class_name AnimationPlayer

## A node used for animation playback.
##
## An animation player is used for general-purpose playback of animations. It contains a dictionary of [AnimationLibrary] resources and custom blend times between animation transitions.
## Some methods and properties use a single key to reference an animation directly. These keys are formatted as the key for the library, followed by a forward slash, then the key for the animation within the library, for example [code]"movement/run"[/code]. If the library's key is an empty string (known as the default library), the forward slash is omitted, being the same key used by the library.
## [AnimationPlayer] is better-suited than [Tween] for more complex animations, for example ones with non-trivial timings. It can also be used over [Tween] if the animation track editor is more convenient than doing it in code.
## Updating the target properties of animations occurs at the process frame.
##
## @tutorial(2D Sprite animation): https://docs.godotengine.org/en/stable/tutorials/2d/2d_sprite_animation.html
## @tutorial(Animation documentation index): https://docs.godotengine.org/en/stable/tutorials/animation/index.html
## @tutorial(Third Person Shooter Demo): https://godotengine.org/asset-library/asset/678


## Emitted when a queued animation plays after the previous animation finished. See also [method AnimationPlayer.queue].
## [b]Note:[/b] The signal is not emitted when the animation is changed via [method AnimationPlayer.play] or by an [AnimationTree].
signal animation_changed(old_name: StringName, new_name: StringName)
## Emitted when [member current_animation] changes.
signal current_animation_changed(name: String)
## For backward compatibility. See [constant AnimationMixer.ANIMATION_CALLBACK_MODE_PROCESS_PHYSICS].
## @deprecated
## For backward compatibility. See [constant AnimationMixer.ANIMATION_CALLBACK_MODE_PROCESS_IDLE].
## @deprecated
## For backward compatibility. See [constant AnimationMixer.ANIMATION_CALLBACK_MODE_PROCESS_MANUAL].
## @deprecated
## For backward compatibility. See [constant AnimationMixer.ANIMATION_CALLBACK_MODE_METHOD_DEFERRED].
## @deprecated
## For backward compatibility. See [constant AnimationMixer.ANIMATION_CALLBACK_MODE_METHOD_IMMEDIATE].
## @deprecated

#enum AnimationProcessCallback
enum {
    ANIMATION_PROCESS_PHYSICS = 0,
    ANIMATION_PROCESS_IDLE = 1,
    ANIMATION_PROCESS_MANUAL = 2,
}
#enum AnimationMethodCallMode
enum {
    ANIMATION_METHOD_CALL_DEFERRED = 0,
    ANIMATION_METHOD_CALL_IMMEDIATE = 1,
}
## If playing, the current animation's key, otherwise, the animation last played. When set, this changes the animation, but will not play it unless already playing. See also [member current_animation].
var assigned_animation: String:
	get = get_assigned_animation, set = set_assigned_animation

## The key of the animation to play when the scene loads.
var autoplay: String:
	get = get_autoplay, set = set_autoplay

## The key of the currently playing animation. If no animation is playing, the property's value is an empty string. Changing this value does not restart the animation. See [method play] for more information on playing animations.
## [b]Note:[/b] While this property appears in the Inspector, it's not meant to be edited, and it's not saved in the scene. This property is mainly used to get the currently playing animation, and internally for animation playback tracks. For more information, see [Animation].
var current_animation: String:
	get = get_current_animation, set = set_current_animation

## The length (in seconds) of the currently playing animation.
var current_animation_length: float:
	get = get_current_animation_length

## The position (in seconds) of the currently playing animation.
var current_animation_position: float:
	get = get_current_animation_position

## If [code]true[/code] and the engine is running in Movie Maker mode (see [MovieWriter]), exits the engine with [method SceneTree.quit] as soon as an animation is done playing in this [AnimationPlayer]. A message is printed when the engine quits for this reason.
## [b]Note:[/b] This obeys the same logic as the [signal AnimationMixer.animation_finished] signal, so it will not quit the engine if the animation is set to be looping.
var movie_quit_on_finish: bool:
	get = is_movie_quit_on_finish_enabled, set = set_movie_quit_on_finish_enabled

## The default time in which to blend animations. Ranges from 0 to 4096 with 0.01 precision.
var playback_default_blend_time: float:
	get = get_default_blend_time, set = set_default_blend_time

## The speed scaling ratio. For example, if this value is [code]1[/code], then the animation plays at normal speed. If it's [code]0.5[/code], then it plays at half speed. If it's [code]2[/code], then it plays at double speed.
## If set to a negative value, the animation is played in reverse. If set to [code]0[/code], the animation will not advance.
var speed_scale: float:
	get = get_speed_scale, set = set_speed_scale



## Returns the key of the animation which is queued to play after the [param animation_from] animation.
func animation_get_next(animation_from: StringName) -> StringName:
	pass;

## Triggers the [param animation_to] animation when the [param animation_from] animation completes.
func animation_set_next(animation_from: StringName, animation_to: StringName) -> void:
	pass;

## Clears all queued, unplayed animations.
func clear_queue() -> void:
	pass;

## Returns the blend time (in seconds) between two animations, referenced by their keys.
func get_blend_time(animation_from: StringName, animation_to: StringName) -> float:
	pass;

## For backward compatibility. See [enum AnimationMixer.AnimationCallbackModeMethod].
## @deprecated
func get_method_call_mode() -> int:
	pass;

## Returns the actual playing speed of current animation or [code]0[/code] if not playing. This speed is the [member speed_scale] property multiplied by [code]custom_speed[/code] argument specified when calling the [method play] method.
## Returns a negative value if the current animation is playing backwards.
func get_playing_speed() -> float:
	pass;

## For backward compatibility. See [enum AnimationMixer.AnimationCallbackModeProcess].
## @deprecated
func get_process_callback() -> int:
	pass;

## Returns a list of the animation keys that are currently queued to play.
func get_queue() -> PackedStringArray:
	pass;

## For backward compatibility. See [member AnimationMixer.root_node].
## @deprecated
func get_root() -> NodePath:
	pass;

## Returns [code]true[/code] if an animation is currently playing (even if [member speed_scale] and/or [code]custom_speed[/code] are [code]0[/code]).
func is_playing() -> bool:
	pass;

## Pauses the currently playing animation. The [member current_animation_position] will be kept and calling [method play] or [method play_backwards] without arguments or with the same animation name as [member assigned_animation] will resume the animation.
## See also [method stop].
func pause() -> void:
	pass;

## Plays the animation with key [param name]. Custom blend times and speed can be set.
## The [param from_end] option only affects when switching to a new animation track, or if the same track but at the start or end. It does not affect resuming playback that was paused in the middle of an animation. If [param custom_speed] is negative and [param from_end] is [code]true[/code], the animation will play backwards (which is equivalent to calling [method play_backwards]).
## The [AnimationPlayer] keeps track of its current or last played animation with [member assigned_animation]. If this method is called with that same animation [param name], or with no [param name] parameter, the assigned animation will resume playing if it was paused.
## [b]Note:[/b] The animation will be updated the next time the [AnimationPlayer] is processed. If other variables are updated at the same time this is called, they may be updated too early. To perform the update immediately, call [code]advance(0)[/code].
func play(name: StringName = "", custom_blend: float = -1, custom_speed: float = 1.0, from_end: bool = false) -> void:
	pass;

## Plays the animation with key [param name] in reverse.
## This method is a shorthand for [method play] with [code]custom_speed = -1.0[/code] and [code]from_end = true[/code], so see its description for more information.
func play_backwards(name: StringName = "", custom_blend: float = -1) -> void:
	pass;

## Queues an animation for playback once the current one is done.
## [b]Note:[/b] If a looped animation is currently playing, the queued animation will never play unless the looped animation is stopped somehow.
func queue(name: StringName) -> void:
	pass;

## Seeks the animation to the [param seconds] point in time (in seconds). If [param update] is [code]true[/code], the animation updates too, otherwise it updates at process time. Events between the current frame and [param seconds] are skipped.
## If [param update_only] is true, the method / audio / animation playback tracks will not be processed.
## [b]Note:[/b] Seeking to the end of the animation doesn't emit [signal AnimationMixer.animation_finished]. If you want to skip animation and emit the signal, use [method AnimationMixer.advance].
func seek(seconds: float, update: bool = false, update_only: bool = false) -> void:
	pass;

## Specifies a blend time (in seconds) between two animations, referenced by their keys.
func set_blend_time(animation_from: StringName, animation_to: StringName, sec: float) -> void:
	pass;

## For backward compatibility. See [enum AnimationMixer.AnimationCallbackModeMethod].
## @deprecated
func set_method_call_mode(mode: int) -> void:
	pass;

## For backward compatibility. See [enum AnimationMixer.AnimationCallbackModeProcess].
## @deprecated
func set_process_callback(mode: int) -> void:
	pass;

## For backward compatibility. See [member AnimationMixer.root_node].
## @deprecated
func set_root(path: NodePath) -> void:
	pass;

## Stops the currently playing animation. The animation position is reset to [code]0[/code] and the [code]custom_speed[/code] is reset to [code]1.0[/code]. See also [method pause].
## If [param keep_state] is [code]true[/code], the animation state is not updated visually.
## [b]Note:[/b] The method / audio / animation playback tracks will not be processed by this method.
func stop(keep_state: bool = false) -> void:
	pass;


func get_assigned_animation() -> String:
	return assigned_animation

func set_assigned_animation(value: String) -> void:
	assigned_animation = value

func get_autoplay() -> String:
	return autoplay

func set_autoplay(value: String) -> void:
	autoplay = value

func get_current_animation() -> String:
	return current_animation

func set_current_animation(value: String) -> void:
	current_animation = value

func get_current_animation_length() -> float:
	return current_animation_length

func get_current_animation_position() -> float:
	return current_animation_position

func is_movie_quit_on_finish_enabled() -> bool:
	return movie_quit_on_finish

func set_movie_quit_on_finish_enabled(value: bool) -> void:
	movie_quit_on_finish = value

func get_default_blend_time() -> float:
	return playback_default_blend_time

func set_default_blend_time(value: float) -> void:
	playback_default_blend_time = value

func get_speed_scale() -> float:
	return speed_scale

func set_speed_scale(value: float) -> void:
	speed_scale = value

