extends RefCounted
class_name Resource

## Base class for serializable objects.
##
## Resource is the base class for all Godot-specific resource types, serving primarily as data containers. Since they inherit from [RefCounted], resources are reference-counted and freed when no longer in use. They can also be nested within other resources, and saved on disk. Once loaded from disk, further attempts to load a resource by [member resource_path] returns the same reference. [PackedScene], one of the most common [Object]s in a Godot project, is also a resource, uniquely capable of storing and instantiating the [Node]s it contains as many times as desired.
## In GDScript, resources can loaded from disk by their [member resource_path] using [method @GDScript.load] or [method @GDScript.preload].
## [b]Note:[/b] In C#, resources will not be freed instantly after they are no longer in use. Instead, garbage collection will run periodically and will free resources that are no longer in use. This means that unused resources will linger on for a while before being removed.
##
## @tutorial(Resources): https://docs.godotengine.org/en/stable/tutorials/scripting/resources.html
## @tutorial(When and how to avoid using nodes for everything): https://docs.godotengine.org/en/stable/tutorials/best_practices/node_alternatives.html


## Emitted when the resource changes, usually when one of its properties is modified. See also [method emit_changed].
## [b]Note:[/b] This signal is not emitted automatically for properties of custom resources. If necessary, a setter needs to be created to emit the signal.
signal changed
## Emitted by a newly duplicated resource with [member resource_local_to_scene] set to [code]true[/code].
## [i]Deprecated.[/i] This signal is only emitted when the resource is created. Override [method _setup_local_to_scene] instead.
## @deprecated
signal setup_local_to_scene_requested

## If [code]true[/code], the resource is duplicated for each instance of all scenes using it. At run-time, the resource can be modified in one scene without affecting other instances (see [method PackedScene.instantiate]).
## [b]Note:[/b] Changing this property at run-time has no effect on already created duplicate resources.
var resource_local_to_scene: bool:
	get = is_local_to_scene, set = set_local_to_scene

## An optional name for this resource. When defined, its value is displayed to represent the resource in the Inspector dock. For built-in scripts, the name is displayed as part of the tab name in the script editor.
## [b]Note:[/b] Some resource formats do not support resource names. You can still set the name in the editor or via code, but it will be lost when the resource is reloaded. For example, only built-in scripts can have a resource name, while scripts stored in separate files cannot.
var resource_name: String:
	get = get_name, set = set_name

## The unique path to this resource. If it has been saved to disk, the value will be its filepath. If the resource is exclusively contained within a scene, the value will be the [PackedScene]'s filepath, followed by a unique identifier.
## [b]Note:[/b] Setting this property manually may fail if a resource with the same path has already been previously loaded. If necessary, use [method take_over_path].
var resource_path: String:
	get = get_path, set = set_path



## Override this method to return a custom [RID] when [method get_rid] is called.
func _get_rid() -> RID:
	pass;

## Override this method to customize the newly duplicated resource created from [method PackedScene.instantiate], if the original's [member resource_local_to_scene] is set to [code]true[/code].
## [b]Example:[/b] Set a random [code]damage[/code] value to every local resource from an instantiated scene.
## [codeblock]
## extends Resource
## var damage = 0
## func _setup_local_to_scene():
## damage = randi_range(10, 40)
## [/codeblock]
func _setup_local_to_scene() -> void:
	pass;

## Duplicates this resource, returning a new resource with its [code]export[/code]ed or [constant PROPERTY_USAGE_STORAGE] properties copied from the original.
## If [param subresources] is [code]false[/code], a shallow copy is returned; nested resources within subresources are not duplicated and are shared from the original resource. If [param subresources] is [code]true[/code], a deep copy is returned; nested subresources will be duplicated and are not shared.
## Subresource properties with the [constant PROPERTY_USAGE_ALWAYS_DUPLICATE] flag are always duplicated even with [param subresources] set to [code]false[/code], and properties with the [constant PROPERTY_USAGE_NEVER_DUPLICATE] flag are never duplicated even with [param subresources] set to [code]true[/code].
## [b]Note:[/b] For custom resources, this method will fail if [method Object._init] has been defined with required parameters.
func duplicate(subresources: bool = false) -> Resource:
	pass;

## Emits the [signal changed] signal. This method is called automatically for some built-in resources.
## [b]Note:[/b] For custom resources, it's recommended to call this method whenever a meaningful change occurs, such as a modified property. This ensures that custom [Object]s depending on the resource are properly updated.
## [codeblock]
## var damage:
## set(new_value):
## if damage != new_value:
## damage = new_value
## emit_changed()
## [/codeblock]
func emit_changed() -> void:
	pass;

## If [member resource_local_to_scene] is set to [code]true[/code] and the resource has been loaded from a [PackedScene] instantiation, returns the root [Node] of the scene where this resource is used. Otherwise, returns [code]null[/code].
func get_local_scene() -> Node:
	pass;

## Returns the [RID] of this resource (or an empty RID). Many resources (such as [Texture2D], [Mesh], and so on) are high-level abstractions of resources stored in a specialized server ([DisplayServer], [RenderingServer], etc.), so this function will return the original [RID].
func get_rid() -> RID:
	pass;

## Calls [method _setup_local_to_scene]. If [member resource_local_to_scene] is set to [code]true[/code], this method is automatically called from [method PackedScene.instantiate] by the newly duplicated resource within the scene instance.
## [i]Deprecated.[/i] This method should only be called internally. Override [method _setup_local_to_scene] instead.
## @deprecated
func setup_local_to_scene() -> void:
	pass;

## Sets the [member resource_path] to [param path], potentially overriding an existing cache entry for this path. Further attempts to load an overridden resource by path will instead return this resource.
func take_over_path(path: String) -> void:
	pass;


func is_local_to_scene() -> bool:
	return resource_local_to_scene

func set_local_to_scene(value: bool) -> void:
	resource_local_to_scene = value

func get_name() -> String:
	return resource_name

func set_name(value: String) -> void:
	resource_name = value

func get_path() -> String:
	return resource_path

func set_path(value: String) -> void:
	resource_path = value

