extends Resource
class_name Material

## Virtual base class for applying visual properties to an object, such as color and roughness.
##
## [Material] is a base resource used for coloring and shading geometry. All materials inherit from it and almost all [VisualInstance3D] derived nodes carry a [Material]. A few flags and parameters are shared between all material types and are configured here.
## Importantly, you can inherit from [Material] to create your own custom material type in script or in GDExtension.
##
## @tutorial(3D Material Testers Demo): https://godotengine.org/asset-library/asset/123
## @tutorial(Third Person Shooter Demo): https://godotengine.org/asset-library/asset/678


## Maximum value for the [member render_priority] parameter.
const RENDER_PRIORITY_MAX = 127;

## Minimum value for the [member render_priority] parameter.
const RENDER_PRIORITY_MIN = -128;


## Sets the [Material] to be used for the next pass. This renders the object again using a different material.
## [b]Note:[/b] [member next_pass] materials are not necessarily drawn immediately after the source [Material]. Draw order is determined by material properties, [member render_priority], and distance to camera.
## [b]Note:[/b] This only applies to [StandardMaterial3D]s and [ShaderMaterial]s with type "Spatial".
var next_pass: Material:
	get = get_next_pass, set = set_next_pass

## Sets the render priority for objects in 3D scenes. Higher priority objects will be sorted in front of lower priority objects. In other words, all objects with [member render_priority] [code]1[/code] will render before all objects with [member render_priority] [code]0[/code]).
## [b]Note:[/b] This only applies to [StandardMaterial3D]s and [ShaderMaterial]s with type "Spatial".
## [b]Note:[/b] This will not impact how transparent objects are sorted relative to opaque objects or how dynamic meshes will be sorted relative to other opaque meshes. This is because all transparent objects are drawn after all opaque objects and all dynamic opaque meshes are drawn before other opaque meshes.
var render_priority: int:
	get = get_render_priority, set = set_render_priority



## Only exposed for the purpose of overriding. You cannot call this function directly. Used internally to determine if [member next_pass] should be shown in the editor or not.
func _can_do_next_pass() -> bool:
	pass;

## Only exposed for the purpose of overriding. You cannot call this function directly. Used internally to determine if [member render_priority] should be shown in the editor or not.
func _can_use_render_priority() -> bool:
	pass;

## Only exposed for the purpose of overriding. You cannot call this function directly. Used internally by various editor tools.
func _get_shader_mode() -> int:
	pass;

## Only exposed for the purpose of overriding. You cannot call this function directly. Used internally by various editor tools. Used to access the RID of the [Material]'s [Shader].
func _get_shader_rid() -> RID:
	pass;

## Creates a placeholder version of this resource ([PlaceholderMaterial]).
func create_placeholder() -> Resource:
	pass;

## Only available when running in the editor. Opens a popup that visualizes the generated shader code, including all variants and internal shader code.
func inspect_native_shader_code() -> void:
	pass;


func get_next_pass() -> Material:
	return next_pass

func set_next_pass(value: Material) -> void:
	next_pass = value

func get_render_priority() -> int:
	return render_priority

func set_render_priority(value: int) -> void:
	render_priority = value

