extends Object
class_name MainLoop

## Abstract base class for the game's main loop.
##
## [MainLoop] is the abstract base class for a Godot project's game loop. It is inherited by [SceneTree], which is the default game loop implementation used in Godot projects, though it is also possible to write and use one's own [MainLoop] subclass instead of the scene tree.
## Upon the application start, a [MainLoop] implementation must be provided to the OS; otherwise, the application will exit. This happens automatically (and a [SceneTree] is created) unless a [MainLoop] [Script] is provided from the command line (with e.g. [code]godot -s my_loop.gd[/code]) or the "Main Loop Type" project setting is overwritten.
## Here is an example script implementing a simple [MainLoop]:
## [codeblocks]
## [gdscript]
## class_name CustomMainLoop
## extends MainLoop
## var time_elapsed = 0
## func _initialize():
## print("Initialized:")
## print("  Starting time: %s" % str(time_elapsed))
## func _process(delta):
## time_elapsed += delta
## # Return true to end the main loop.
## return Input.get_mouse_button_mask() != 0 || Input.is_key_pressed(KEY_ESCAPE)
## func _finalize():
## print("Finalized:")
## print("  End time: %s" % str(time_elapsed))
## [/gdscript]
## [csharp]
## using Godot;
## [GlobalClass]
## public partial class CustomMainLoop : MainLoop
## {
## private double _timeElapsed = 0;
## public override void _Initialize()
## {
## GD.Print("Initialized:");
## GD.Print($"  Starting Time: {_timeElapsed}");
## }
## public override bool _Process(double delta)
## {
## _timeElapsed += delta;
## // Return true to end the main loop.
## return Input.GetMouseButtonMask() != 0 || Input.IsKeyPressed(Key.Escape);
## }
## private void _Finalize()
## {
## GD.Print("Finalized:");
## GD.Print($"  End Time: {_timeElapsed}");
## }
## }
## [/csharp]
## [/codeblocks]


## Emitted when a user responds to a permission request.
signal on_request_permissions_result(permission: String, granted: bool)
## Notification received from the OS when the application is exceeding its allocated memory.
## Specific to the iOS platform.
const NOTIFICATION_OS_MEMORY_WARNING = 2009;

## Notification received when translations may have changed. Can be triggered by the user changing the locale. Can be used to respond to language changes, for example to change the UI strings on the fly. Useful when working with the built-in translation support, like [method Object.tr].
const NOTIFICATION_TRANSLATION_CHANGED = 2010;

## Notification received from the OS when a request for "About" information is sent.
## Specific to the macOS platform.
const NOTIFICATION_WM_ABOUT = 2011;

## Notification received from Godot's crash handler when the engine is about to crash.
## Implemented on desktop platforms if the crash handler is enabled.
const NOTIFICATION_CRASH = 2012;

## Notification received from the OS when an update of the Input Method Engine occurs (e.g. change of IME cursor position or composition string).
## Specific to the macOS platform.
const NOTIFICATION_OS_IME_UPDATE = 2013;

## Notification received from the OS when the application is resumed.
## Specific to the Android platform.
const NOTIFICATION_APPLICATION_RESUMED = 2014;

## Notification received from the OS when the application is paused.
## Specific to the Android platform.
const NOTIFICATION_APPLICATION_PAUSED = 2015;

## Notification received from the OS when the application is focused, i.e. when changing the focus from the OS desktop or a thirdparty application to any open window of the Godot instance.
## Implemented on desktop platforms.
const NOTIFICATION_APPLICATION_FOCUS_IN = 2016;

## Notification received from the OS when the application is defocused, i.e. when changing the focus from any open window of the Godot instance to the OS desktop or a thirdparty application.
## Implemented on desktop platforms.
const NOTIFICATION_APPLICATION_FOCUS_OUT = 2017;

## Notification received when text server is changed.
const NOTIFICATION_TEXT_SERVER_CHANGED = 2018;




## Called before the program exits.
func _finalize() -> void:
	pass;

## Called once during initialization.
func _initialize() -> void:
	pass;

## Called each physics frame with the time since the last physics frame as argument ([param delta], in seconds). Equivalent to [method Node._physics_process].
## If implemented, the method must return a boolean value. [code]true[/code] ends the main loop, while [code]false[/code] lets it proceed to the next frame.
func _physics_process(delta: float) -> bool:
	pass;

## Called each process (idle) frame with the time since the last process frame as argument (in seconds). Equivalent to [method Node._process].
## If implemented, the method must return a boolean value. [code]true[/code] ends the main loop, while [code]false[/code] lets it proceed to the next frame.
func _process(delta: float) -> bool:
	pass;


