extends GeometryInstance3D
class_name GPUParticles3D

## A 3D particle emitter.
##
## 3D particle node used to create a variety of particle systems and effects. [GPUParticles3D] features an emitter that generates some number of particles at a given rate.
## Use [member process_material] to add a [ParticleProcessMaterial] to configure particle appearance and behavior. Alternatively, you can add a [ShaderMaterial] which will be applied to all particles.
##
## @tutorial(Particle systems (3D)): https://docs.godotengine.org/en/stable/tutorials/3d/particles/index.html
## @tutorial(Controlling thousands of fish with Particles): https://docs.godotengine.org/en/stable/tutorials/performance/vertex_animation/controlling_thousands_of_fish.html
## @tutorial(Third Person Shooter Demo): https://godotengine.org/asset-library/asset/678


## Emitted when all active particles have finished processing. When [member one_shot] is disabled, particles will process continuously, so this is never emitted.
## [b]Note:[/b] Due to the particles being computed on the GPU there might be a delay before the signal gets emitted.
signal finished
## Particles are drawn in the order emitted.
## Particles are drawn in order of remaining lifetime. In other words, the particle with the highest lifetime is drawn at the front.
## Particles are drawn in reverse order of remaining lifetime. In other words, the particle with the lowest lifetime is drawn at the front.
## Particles are drawn in order of depth.
## Particle starts at the specified position.
## Particle starts with specified rotation and scale.
## Particle starts with the specified velocity vector, which defines the emission direction and speed.
## Particle starts with specified color.
## Particle starts with specified [code]CUSTOM[/code] data.
## Maximum number of draw passes supported.
const MAX_DRAW_PASSES = 4;


#enum DrawOrder
enum {
    DRAW_ORDER_INDEX = 0,
    DRAW_ORDER_LIFETIME = 1,
    DRAW_ORDER_REVERSE_LIFETIME = 2,
    DRAW_ORDER_VIEW_DEPTH = 3,
}
#enum EmitFlags
enum {
    EMIT_FLAG_POSITION = 1,
    EMIT_FLAG_ROTATION_SCALE = 2,
    EMIT_FLAG_VELOCITY = 4,
    EMIT_FLAG_COLOR = 8,
    EMIT_FLAG_CUSTOM = 16,
}
#enum TransformAlign
enum {
    TRANSFORM_ALIGN_DISABLED = 0,
    TRANSFORM_ALIGN_Z_BILLBOARD = 1,
    TRANSFORM_ALIGN_Y_TO_VELOCITY = 2,
    TRANSFORM_ALIGN_Z_BILLBOARD_Y_TO_VELOCITY = 3,
}
## The number of particles to emit in one emission cycle. The effective emission rate is [code](amount * amount_ratio) / lifetime[/code] particles per second. Higher values will increase GPU requirements, even if not all particles are visible at a given time or if [member amount_ratio] is decreased.
## [b]Note:[/b] Changing this value will cause the particle system to restart. To avoid this, change [member amount_ratio] instead.
var amount: int:
	get = get_amount, set = set_amount

## The ratio of particles that should actually be emitted. If set to a value lower than [code]1.0[/code], this will set the amount of emitted particles throughout the lifetime to [code]amount * amount_ratio[/code]. Unlike changing [member amount], changing [member amount_ratio] while emitting does not affect already-emitted particles and doesn't cause the particle system to restart. [member amount_ratio] can be used to create effects that make the number of emitted particles vary over time.
## [b]Note:[/b] Reducing the [member amount_ratio] has no performance benefit, since resources need to be allocated and processed for the total [member amount] of particles regardless of the [member amount_ratio]. If you don't intend to change the number of particles emitted while the particles are emitting, make sure [member amount_ratio] is set to [code]1[/code] and change [member amount] to your liking instead.
var amount_ratio: float:
	get = get_amount_ratio, set = set_amount_ratio

## The base diameter for particle collision in meters. If particles appear to sink into the ground when colliding, increase this value. If particles appear to float when colliding, decrease this value. Only effective if [member ParticleProcessMaterial.collision_mode] is [constant ParticleProcessMaterial.COLLISION_RIGID] or [constant ParticleProcessMaterial.COLLISION_HIDE_ON_CONTACT].
## [b]Note:[/b] Particles always have a spherical collision shape.
var collision_base_size: float:
	get = get_collision_base_size, set = set_collision_base_size

## Particle draw order. Uses [enum DrawOrder] values.
## [b]Note:[/b] [constant DRAW_ORDER_INDEX] is the only option that supports motion vectors for effects like TAA. It is suggested to use this draw order if the particles are opaque to fix ghosting artifacts.
var draw_order: int:
	get = get_draw_order, set = set_draw_order

## [Mesh] that is drawn for the first draw pass.
var draw_pass_1: Mesh:
	get = get_draw_pass_mesh, set = set_draw_pass_mesh

## [Mesh] that is drawn for the second draw pass.
var draw_pass_2: Mesh:
	get = get_draw_pass_mesh, set = set_draw_pass_mesh

## [Mesh] that is drawn for the third draw pass.
var draw_pass_3: Mesh:
	get = get_draw_pass_mesh, set = set_draw_pass_mesh

## [Mesh] that is drawn for the fourth draw pass.
var draw_pass_4: Mesh:
	get = get_draw_pass_mesh, set = set_draw_pass_mesh

## The number of draw passes when rendering particles.
var draw_passes: int:
	get = get_draw_passes, set = set_draw_passes

var draw_skin: Skin:
	get = get_skin, set = set_skin

## If [code]true[/code], particles are being emitted. [member emitting] can be used to start and stop particles from emitting. However, if [member one_shot] is [code]true[/code] setting [member emitting] to [code]true[/code] will not restart the emission cycle until after all active particles finish processing. You can use the [signal finished] signal to be notified once all active particles finish processing.
var emitting: bool:
	get = is_emitting, set = set_emitting

## Time ratio between each emission. If [code]0[/code], particles are emitted continuously. If [code]1[/code], all particles are emitted simultaneously.
var explosiveness: float:
	get = get_explosiveness_ratio, set = set_explosiveness_ratio

## The particle system's frame rate is fixed to a value. For example, changing the value to 2 will make the particles render at 2 frames per second. Note this does not slow down the simulation of the particle system itself.
var fixed_fps: int:
	get = get_fixed_fps, set = set_fixed_fps

## If [code]true[/code], results in fractional delta calculation which has a smoother particles display effect.
var fract_delta: bool:
	get = get_fractional_delta, set = set_fractional_delta

## Causes all the particles in this node to interpolate towards the end of their lifetime.
## [b]Note:[/b] This only works when used with a [ParticleProcessMaterial]. It needs to be manually implemented for custom process shaders.
var interp_to_end: float:
	get = get_interp_to_end, set = set_interp_to_end

## Enables particle interpolation, which makes the particle movement smoother when their [member fixed_fps] is lower than the screen refresh rate.
var interpolate: bool:
	get = get_interpolate, set = set_interpolate

## The amount of time each particle will exist (in seconds). The effective emission rate is [code](amount * amount_ratio) / lifetime[/code] particles per second.
var lifetime: float:
	get = get_lifetime, set = set_lifetime

## If [code]true[/code], particles use the parent node's coordinate space (known as local coordinates). This will cause particles to move and rotate along the [GPUParticles3D] node (and its parents) when it is moved or rotated. If [code]false[/code], particles use global coordinates; they will not move or rotate along the [GPUParticles3D] node (and its parents) when it is moved or rotated.
var local_coords: bool:
	get = get_use_local_coordinates, set = set_use_local_coordinates

## If [code]true[/code], only the number of particles equal to [member amount] will be emitted.
var one_shot: bool:
	get = get_one_shot, set = set_one_shot

## Amount of time to preprocess the particles before animation starts. Lets you start the animation some time after particles have started emitting.
var preprocess: float:
	get = get_pre_process_time, set = set_pre_process_time

## [Material] for processing particles. Can be a [ParticleProcessMaterial] or a [ShaderMaterial].
var process_material: Material:
	get = get_process_material, set = set_process_material

## Emission randomness ratio.
var randomness: float:
	get = get_randomness_ratio, set = set_randomness_ratio

## Speed scaling ratio. A value of [code]0[/code] can be used to pause the particles.
var speed_scale: float:
	get = get_speed_scale, set = set_speed_scale

## Path to another [GPUParticles3D] node that will be used as a subemitter (see [member ParticleProcessMaterial.sub_emitter_mode]). Subemitters can be used to achieve effects such as fireworks, sparks on collision, bubbles popping into water drops, and more.
## [b]Note:[/b] When [member sub_emitter] is set, the target [GPUParticles3D] node will no longer emit particles on its own.
var sub_emitter: NodePath:
	get = get_sub_emitter, set = set_sub_emitter

## If [code]true[/code], enables particle trails using a mesh skinning system. Designed to work with [RibbonTrailMesh] and [TubeTrailMesh].
## [b]Note:[/b] [member BaseMaterial3D.use_particle_trails] must also be enabled on the particle mesh's material. Otherwise, setting [member trail_enabled] to [code]true[/code] will have no effect.
## [b]Note:[/b] Unlike [GPUParticles2D], the number of trail sections and subdivisions is set in the [RibbonTrailMesh] or the [TubeTrailMesh]'s properties.
var trail_enabled: bool:
	get = is_trail_enabled, set = set_trail_enabled

## The amount of time the particle's trail should represent (in seconds). Only effective if [member trail_enabled] is [code]true[/code].
var trail_lifetime: float:
	get = get_trail_lifetime, set = set_trail_lifetime

var transform_align: int:
	get = get_transform_align, set = set_transform_align

## The [AABB] that determines the node's region which needs to be visible on screen for the particle system to be active. [member GeometryInstance3D.extra_cull_margin] is added on each of the AABB's axes. Particle collisions and attraction will only occur within this area.
## Grow the box if particles suddenly appear/disappear when the node enters/exits the screen. The [AABB] can be grown via code or with the [b]Particles → Generate AABB[/b] editor tool.
## [b]Note:[/b] [member visibility_aabb] is overridden by [member GeometryInstance3D.custom_aabb] if that property is set to a non-default value.
var visibility_aabb: AABB:
	get = get_visibility_aabb, set = set_visibility_aabb



## Returns the axis-aligned bounding box that contains all the particles that are active in the current frame.
func capture_aabb() -> AABB:
	pass;

## Sets this node's properties to match a given [CPUParticles3D] node.
func convert_from_particles(particles: Node) -> void:
	pass;

## Emits a single particle. Whether [param xform], [param velocity], [param color] and [param custom] are applied depends on the value of [param flags]. See [enum EmitFlags].
func emit_particle(xform: Transform3D, velocity: Vector3, color: Color, custom: Color, flags: int) -> void:
	pass;

## Returns the [Mesh] that is drawn at index [param pass].
func get_draw_pass_mesh(pass: int) -> Mesh:
	pass;

## Restarts the particle emission, clearing existing particles.
func restart() -> void:
	pass;

## Sets the [Mesh] that is drawn at index [param pass].
func set_draw_pass_mesh(pass: int, mesh: Mesh) -> void:
	pass;


func get_amount() -> int:
	return amount

func set_amount(value: int) -> void:
	amount = value

func get_amount_ratio() -> float:
	return amount_ratio

func set_amount_ratio(value: float) -> void:
	amount_ratio = value

func get_collision_base_size() -> float:
	return collision_base_size

func set_collision_base_size(value: float) -> void:
	collision_base_size = value

func get_draw_order() -> int:
	return draw_order

func set_draw_order(value: int) -> void:
	draw_order = value

func get_draw_pass_mesh() -> Mesh:
	return draw_pass_1

func set_draw_pass_mesh(value: Mesh) -> void:
	draw_pass_1 = value

func get_draw_pass_mesh() -> Mesh:
	return draw_pass_2

func set_draw_pass_mesh(value: Mesh) -> void:
	draw_pass_2 = value

func get_draw_pass_mesh() -> Mesh:
	return draw_pass_3

func set_draw_pass_mesh(value: Mesh) -> void:
	draw_pass_3 = value

func get_draw_pass_mesh() -> Mesh:
	return draw_pass_4

func set_draw_pass_mesh(value: Mesh) -> void:
	draw_pass_4 = value

func get_draw_passes() -> int:
	return draw_passes

func set_draw_passes(value: int) -> void:
	draw_passes = value

func get_skin() -> Skin:
	return draw_skin

func set_skin(value: Skin) -> void:
	draw_skin = value

func is_emitting() -> bool:
	return emitting

func set_emitting(value: bool) -> void:
	emitting = value

func get_explosiveness_ratio() -> float:
	return explosiveness

func set_explosiveness_ratio(value: float) -> void:
	explosiveness = value

func get_fixed_fps() -> int:
	return fixed_fps

func set_fixed_fps(value: int) -> void:
	fixed_fps = value

func get_fractional_delta() -> bool:
	return fract_delta

func set_fractional_delta(value: bool) -> void:
	fract_delta = value

func get_interp_to_end() -> float:
	return interp_to_end

func set_interp_to_end(value: float) -> void:
	interp_to_end = value

func get_interpolate() -> bool:
	return interpolate

func set_interpolate(value: bool) -> void:
	interpolate = value

func get_lifetime() -> float:
	return lifetime

func set_lifetime(value: float) -> void:
	lifetime = value

func get_use_local_coordinates() -> bool:
	return local_coords

func set_use_local_coordinates(value: bool) -> void:
	local_coords = value

func get_one_shot() -> bool:
	return one_shot

func set_one_shot(value: bool) -> void:
	one_shot = value

func get_pre_process_time() -> float:
	return preprocess

func set_pre_process_time(value: float) -> void:
	preprocess = value

func get_process_material() -> Material:
	return process_material

func set_process_material(value: Material) -> void:
	process_material = value

func get_randomness_ratio() -> float:
	return randomness

func set_randomness_ratio(value: float) -> void:
	randomness = value

func get_speed_scale() -> float:
	return speed_scale

func set_speed_scale(value: float) -> void:
	speed_scale = value

func get_sub_emitter() -> NodePath:
	return sub_emitter

func set_sub_emitter(value: NodePath) -> void:
	sub_emitter = value

func is_trail_enabled() -> bool:
	return trail_enabled

func set_trail_enabled(value: bool) -> void:
	trail_enabled = value

func get_trail_lifetime() -> float:
	return trail_lifetime

func set_trail_lifetime(value: float) -> void:
	trail_lifetime = value

func get_transform_align() -> int:
	return transform_align

func set_transform_align(value: int) -> void:
	transform_align = value

func get_visibility_aabb() -> AABB:
	return visibility_aabb

func set_visibility_aabb(value: AABB) -> void:
	visibility_aabb = value

