extends Viewport
class_name Window

## Base class for all windows, dialogs, and popups.
##
## A node that creates a window. The window can either be a native system window or embedded inside another [Window] (see [member Viewport.gui_embed_subwindows]).
## At runtime, [Window]s will not close automatically when requested. You need to handle it manually using the [signal close_requested] signal (this applies both to pressing the close button and clicking outside of a popup).


## Emitted right after [method popup] call, before the [Window] appears or does anything.
signal about_to_popup
## Emitted when the [Window]'s close button is pressed or when [member popup_window] is enabled and user clicks outside the window.
## This signal can be used to handle window closing, e.g. by connecting it to [method hide].
signal close_requested
## Emitted when the [Window]'s DPI changes as a result of OS-level changes (e.g. moving the window from a Retina display to a lower resolution one).
## [b]Note:[/b] Only implemented on macOS.
signal dpi_changed
## Emitted when files are dragged from the OS file manager and dropped in the game window. The argument is a list of file paths.
## Note that this method only works with native windows, i.e. the main window and [Window]-derived nodes when [member Viewport.gui_embed_subwindows] is disabled in the main viewport.
## Example usage:
## [codeblock]
## func _ready():
## get_viewport().files_dropped.connect(on_files_dropped)
## func on_files_dropped(files):
## print(files)
## [/codeblock]
signal files_dropped(files: PackedStringArray)
## Emitted when the [Window] gains focus.
signal focus_entered
## Emitted when the [Window] loses its focus.
signal focus_exited
## Emitted when a go back request is sent (e.g. pressing the "Back" button on Android), right after [constant Node.NOTIFICATION_WM_GO_BACK_REQUEST].
signal go_back_requested
## Emitted when the mouse cursor enters the [Window]'s visible area, that is not occluded behind other [Control]s or windows, provided its [member Viewport.gui_disable_input] is [code]false[/code] and regardless if it's currently focused or not.
signal mouse_entered
## Emitted when the mouse cursor leaves the [Window]'s visible area, that is not occluded behind other [Control]s or windows, provided its [member Viewport.gui_disable_input] is [code]false[/code] and regardless if it's currently focused or not.
signal mouse_exited
## Emitted when the [constant NOTIFICATION_THEME_CHANGED] notification is sent.
signal theme_changed
## Emitted when window title bar decorations are changed, e.g. macOS window enter/exit full screen mode, or extend-to-title flag is changed.
signal titlebar_changed
## Emitted when [Window] is made visible or disappears.
signal visibility_changed
## Emitted when the [Window] is currently focused and receives any input, passing the received event as an argument. The event's position, if present, is in the embedder's coordinate system.
signal window_input(event: InputEvent)
## Emitted when [Window]'s visibility changes, right before [signal visibility_changed].
const NOTIFICATION_VISIBILITY_CHANGED = 30;

## Sent when the node needs to refresh its theme items. This happens in one of the following cases:
## - The [member theme] property is changed on this node or any of its ancestors.
## - The [member theme_type_variation] property is changed on this node.
## - The node enters the scene tree.
## [b]Note:[/b] As an optimization, this notification won't be sent from changes that occur while this node is outside of the scene tree. Instead, all of the theme item updates can be applied at once when the node enters the scene tree.
const NOTIFICATION_THEME_CHANGED = 32;

## Windowed mode, i.e. [Window] doesn't occupy the whole screen (unless set to the size of the screen).
## Minimized window mode, i.e. [Window] is not visible and available on window manager's window list. Normally happens when the minimize button is pressed.
## Maximized window mode, i.e. [Window] will occupy whole screen area except task bar and still display its borders. Normally happens when the maximize button is pressed.
## Full screen mode with full multi-window support.
## Full screen window covers the entire display area of a screen and has no decorations. The display's video mode is not changed.
## [b]On Windows:[/b] Multi-window full-screen mode has a 1px border of the [member ProjectSettings.rendering/environment/defaults/default_clear_color] color.
## [b]On macOS:[/b] A new desktop is used to display the running project.
## [b]Note:[/b] Regardless of the platform, enabling full screen will change the window size to match the monitor's size. Therefore, make sure your project supports [url=$DOCS_URL/tutorials/rendering/multiple_resolutions.html]multiple resolutions[/url] when enabling full screen mode.
## A single window full screen mode. This mode has less overhead, but only one window can be open on a given screen at a time (opening a child window or application switching will trigger a full screen transition).
## Full screen window covers the entire display area of a screen and has no border or decorations. The display's video mode is not changed.
## [b]On Windows:[/b] Depending on video driver, full screen transition might cause screens to go black for a moment.
## [b]On macOS:[/b] A new desktop is used to display the running project. Exclusive full screen mode prevents Dock and Menu from showing up when the mouse pointer is hovering the edge of the screen.
## [b]On Linux (X11):[/b] Exclusive full screen mode bypasses compositor.
## [b]Note:[/b] Regardless of the platform, enabling full screen will change the window size to match the monitor's size. Therefore, make sure your project supports [url=$DOCS_URL/tutorials/rendering/multiple_resolutions.html]multiple resolutions[/url] when enabling full screen mode.
## The window can't be resized by dragging its resize grip. It's still possible to resize the window using [member size]. This flag is ignored for full screen windows. Set with [member unresizable].
## The window do not have native title bar and other decorations. This flag is ignored for full-screen windows. Set with [member borderless].
## The window is floating on top of all other windows. This flag is ignored for full-screen windows. Set with [member always_on_top].
## The window background can be transparent. Set with [member transparent].
## [b]Note:[/b] This flag has no effect if either [member ProjectSettings.display/window/per_pixel_transparency/allowed], or the window's [member Viewport.transparent_bg] is set to [code]false[/code].
## The window can't be focused. No-focus window will ignore all input, except mouse clicks. Set with [member unfocusable].
## Window is part of menu or [OptionButton] dropdown. This flag can't be changed when the window is visible. An active popup window will exclusively receive all input, without stealing focus from its parent. Popup windows are automatically closed when uses click outside it, or when an application is switched. Popup window must have transient parent set (see [member transient]).
## [b]Note:[/b] This flag has no effect in embedded windows (unless said window is a [Popup]).
## Window content is expanded to the full size of the window. Unlike borderless window, the frame is left intact and can be used to resize the window, title bar is transparent, but have minimize/maximize/close buttons. Set with [member extend_to_title].
## [b]Note:[/b] This flag is implemented only on macOS.
## [b]Note:[/b] This flag has no effect in embedded windows.
## All mouse events are passed to the underlying window of the same application.
## [b]Note:[/b] This flag has no effect in embedded windows.
## Max value of the [enum Flags].
## The content will not be scaled to match the [Window]'s size.
## The content will be rendered at the target size. This is more performance-expensive than [constant CONTENT_SCALE_MODE_VIEWPORT], but provides better results.
## The content will be rendered at the base size and then scaled to the target size. More performant than [constant CONTENT_SCALE_MODE_CANVAS_ITEMS], but results in pixelated image.
## The aspect will be ignored. Scaling will simply stretch the content to fit the target size.
## The content's aspect will be preserved. If the target size has different aspect from the base one, the image will be centered and black bars will appear on left and right sides.
## The content can be expanded vertically. Scaling horizontally will result in keeping the width ratio and then black bars on left and right sides.
## The content can be expanded horizontally. Scaling vertically will result in keeping the height ratio and then black bars on top and bottom sides.
## The content's aspect will be preserved. If the target size has different aspect from the base one, the content will stay in the top-left corner and add an extra visible area in the stretched space.
## The content will be stretched according to a fractional factor. This fills all the space available in the window, but allows "pixel wobble" to occur due to uneven pixel scaling.
## The content will be stretched only according to an integer factor, preserving sharp pixels. This may leave a black background visible on the window's edges depending on the window size.
## Automatic layout direction, determined from the parent window layout direction.
## Automatic layout direction, determined from the current locale.
## Left-to-right layout direction.
## Right-to-left layout direction.
## Initial window position is determined by [member position].
## Initial window position is the center of the primary screen.
## Initial window position is the center of the main window screen.
## Initial window position is the center of [member current_screen] screen.
## Initial window position is the center of the screen containing the mouse pointer.
## Initial window position is the center of the screen containing the window with the keyboard focus.

#enum Mode
enum {
    MODE_WINDOWED = 0,
    MODE_MINIMIZED = 1,
    MODE_MAXIMIZED = 2,
    MODE_FULLSCREEN = 3,
    MODE_EXCLUSIVE_FULLSCREEN = 4,
}
#enum Flags
enum {
    FLAG_RESIZE_DISABLED = 0,
    FLAG_BORDERLESS = 1,
    FLAG_ALWAYS_ON_TOP = 2,
    FLAG_TRANSPARENT = 3,
    FLAG_NO_FOCUS = 4,
    FLAG_POPUP = 5,
    FLAG_EXTEND_TO_TITLE = 6,
    FLAG_MOUSE_PASSTHROUGH = 7,
    FLAG_MAX = 8,
}
#enum ContentScaleMode
enum {
    CONTENT_SCALE_MODE_DISABLED = 0,
    CONTENT_SCALE_MODE_CANVAS_ITEMS = 1,
    CONTENT_SCALE_MODE_VIEWPORT = 2,
}
#enum ContentScaleAspect
enum {
    CONTENT_SCALE_ASPECT_IGNORE = 0,
    CONTENT_SCALE_ASPECT_KEEP = 1,
    CONTENT_SCALE_ASPECT_KEEP_WIDTH = 2,
    CONTENT_SCALE_ASPECT_KEEP_HEIGHT = 3,
    CONTENT_SCALE_ASPECT_EXPAND = 4,
}
#enum ContentScaleStretch
enum {
    CONTENT_SCALE_STRETCH_FRACTIONAL = 0,
    CONTENT_SCALE_STRETCH_INTEGER = 1,
}
#enum LayoutDirection
enum {
    LAYOUT_DIRECTION_INHERITED = 0,
    LAYOUT_DIRECTION_LOCALE = 1,
    LAYOUT_DIRECTION_LTR = 2,
    LAYOUT_DIRECTION_RTL = 3,
}
#enum WindowInitialPosition
enum {
    WINDOW_INITIAL_POSITION_ABSOLUTE = 0,
    WINDOW_INITIAL_POSITION_CENTER_PRIMARY_SCREEN = 1,
    WINDOW_INITIAL_POSITION_CENTER_MAIN_WINDOW_SCREEN = 2,
    WINDOW_INITIAL_POSITION_CENTER_OTHER_SCREEN = 3,
    WINDOW_INITIAL_POSITION_CENTER_SCREEN_WITH_MOUSE_FOCUS = 4,
    WINDOW_INITIAL_POSITION_CENTER_SCREEN_WITH_KEYBOARD_FOCUS = 5,
}
## If [code]true[/code], the window will be on top of all other windows. Does not work if [member transient] is enabled.
var always_on_top: bool:
	get = get_flag, set = set_flag

## Toggles if any text should automatically change to its translated version depending on the current locale.
var auto_translate: bool:
	get = is_auto_translating, set = set_auto_translate

## If [code]true[/code], the window will have no borders.
var borderless: bool:
	get = get_flag, set = set_flag

## Specifies how the content's aspect behaves when the [Window] is resized. The base aspect is determined by [member content_scale_size].
var content_scale_aspect: int:
	get = get_content_scale_aspect, set = set_content_scale_aspect

## Specifies the base scale of [Window]'s content when its [member size] is equal to [member content_scale_size].
var content_scale_factor: float:
	get = get_content_scale_factor, set = set_content_scale_factor

## Specifies how the content is scaled when the [Window] is resized.
var content_scale_mode: int:
	get = get_content_scale_mode, set = set_content_scale_mode

## Base size of the content (i.e. nodes that are drawn inside the window). If non-zero, [Window]'s content will be scaled when the window is resized to a different size.
var content_scale_size: Vector2i:
	get = get_content_scale_size, set = set_content_scale_size

## The policy to use to determine the final scale factor for 2D elements. This affects how [member content_scale_factor] is applied, in addition to the automatic scale factor determined by [member content_scale_size].
var content_scale_stretch: int:
	get = get_content_scale_stretch, set = set_content_scale_stretch

## The screen the window is currently on.
var current_screen: int:
	get = get_current_screen, set = set_current_screen

## If [code]true[/code], the [Window] will be in exclusive mode. Exclusive windows are always on top of their parent and will block all input going to the parent [Window].
## Needs [member transient] enabled to work.
var exclusive: bool:
	get = is_exclusive, set = set_exclusive

## If [code]true[/code], the [Window] contents is expanded to the full size of the window, window title bar is transparent.
## [b]Note:[/b] This property is implemented only on macOS.
## [b]Note:[/b] This property only works with native windows.
var extend_to_title: bool:
	get = get_flag, set = set_flag

## Specifies the initial type of position for the [Window]. See [enum WindowInitialPosition] constants.
var initial_position: int:
	get = get_initial_position, set = set_initial_position

## If [code]true[/code], the [Window] width is expanded to keep the title bar text fully visible.
var keep_title_visible: bool:
	get = get_keep_title_visible, set = set_keep_title_visible

## If non-zero, the [Window] can't be resized to be bigger than this size.
## [b]Note:[/b] This property will be ignored if the value is lower than [member min_size].
var max_size: Vector2i:
	get = get_max_size, set = set_max_size

## If non-zero, the [Window] can't be resized to be smaller than this size.
## [b]Note:[/b] This property will be ignored in favor of [method get_contents_minimum_size] if [member wrap_controls] is enabled and if its size is bigger.
var min_size: Vector2i:
	get = get_min_size, set = set_min_size

## Set's the window's current mode.
## [b]Note:[/b] Fullscreen mode is not exclusive full screen on Windows and Linux.
## [b]Note:[/b] This method only works with native windows, i.e. the main window and [Window]-derived nodes when [member Viewport.gui_embed_subwindows] is disabled in the main viewport.
var mode: int:
	get = get_mode, set = set_mode

## If [code]true[/code], all mouse events will be passed to the underlying window of the same application. See also [member mouse_passthrough_polygon].
## [b]Note:[/b] This property is implemented on Linux (X11), macOS and Windows.
## [b]Note:[/b] This property only works with native windows.
var mouse_passthrough: bool:
	get = get_flag, set = set_flag

## Sets a polygonal region of the window which accepts mouse events. Mouse events outside the region will be passed through.
## Passing an empty array will disable passthrough support (all mouse events will be intercepted by the window, which is the default behavior).
## [codeblocks]
## [gdscript]
## # Set region, using Path2D node.
## $Window.mouse_passthrough_polygon = $Path2D.curve.get_baked_points()
## # Set region, using Polygon2D node.
## $Window.mouse_passthrough_polygon = $Polygon2D.polygon
## # Reset region to default.
## $Window.mouse_passthrough_polygon = []
## [/gdscript]
## [csharp]
## // Set region, using Path2D node.
## GetNode<Window>("Window").MousePassthrough = GetNode<Path2D>("Path2D").Curve.GetBakedPoints();
## // Set region, using Polygon2D node.
## GetNode<Window>("Window").MousePassthrough = GetNode<Polygon2D>("Polygon2D").Polygon;
## // Reset region to default.
## GetNode<Window>("Window").MousePassthrough = new Vector2[] {};
## [/csharp]
## [/codeblocks]
## [b]Note:[/b] This property is ignored if [member mouse_passthrough] is set to [code]true[/code].
## [b]Note:[/b] On Windows, the portion of a window that lies outside the region is not drawn, while on Linux (X11) and macOS it is.
## [b]Note:[/b] This property is implemented on Linux (X11), macOS and Windows.
var mouse_passthrough_polygon: PackedVector2Array:
	get = get_mouse_passthrough_polygon, set = set_mouse_passthrough_polygon

## If [code]true[/code], the [Window] will be considered a popup. Popups are sub-windows that don't show as separate windows in system's window manager's window list and will send close request when anything is clicked outside of them (unless [member exclusive] is enabled).
var popup_window: bool:
	get = get_flag, set = set_flag

## The window's position in pixels.
## If [member ProjectSettings.display/window/subwindows/embed_subwindows] is [code]false[/code], the position is in absolute screen coordinates. This typically applies to editor plugins. If the setting is [code]true[/code], the window's position is in the coordinates of its parent [Viewport].
## [b]Note:[/b] This property only works if [member initial_position] is set to [constant WINDOW_INITIAL_POSITION_ABSOLUTE].
var position: Vector2i:
	get = get_position, set = set_position

## The window's size in pixels.
var size: Vector2i:
	get = get_size, set = set_size

## The [Theme] resource this node and all its [Control] and [Window] children use. If a child node has its own [Theme] resource set, theme items are merged with child's definitions having higher priority.
## [b]Note:[/b] [Window] styles will have no effect unless the window is embedded.
var theme: Theme:
	get = get_theme, set = set_theme

## The name of a theme type variation used by this [Window] to look up its own theme items. See [member Control.theme_type_variation] for more details.
var theme_type_variation: StringName:
	get = get_theme_type_variation, set = set_theme_type_variation

## The window's title. If the [Window] is native, title styles set in [Theme] will have no effect.
var title: String:
	get = get_title, set = set_title

## If [code]true[/code], the [Window] is transient, i.e. it's considered a child of another [Window]. The transient window will be destroyed with its transient parent and will return focus to their parent when closed. The transient window is displayed on top of a non-exclusive full-screen parent window. Transient windows can't enter full-screen mode.
## Note that behavior might be different depending on the platform.
var transient: bool:
	get = is_transient, set = set_transient

## If [code]true[/code], the [Window]'s background can be transparent. This is best used with embedded windows.
## [b]Note:[/b] Transparency support is implemented on Linux, macOS and Windows, but availability might vary depending on GPU driver, display manager, and compositor capabilities.
## [b]Note:[/b] This property has no effect if either [member ProjectSettings.display/window/per_pixel_transparency/allowed], or the window's [member Viewport.transparent_bg] is set to [code]false[/code].
var transparent: bool:
	get = get_flag, set = set_flag

## If [code]true[/code], the [Window] can't be focused nor interacted with. It can still be visible.
var unfocusable: bool:
	get = get_flag, set = set_flag

## If [code]true[/code], the window can't be resized. Minimize and maximize buttons are disabled.
var unresizable: bool:
	get = get_flag, set = set_flag

## If [code]true[/code], the window is visible.
var visible: bool:
	get = is_visible, set = set_visible

## If [code]true[/code], the window's size will automatically update when a child node is added or removed, ignoring [member min_size] if the new size is bigger.
## If [code]false[/code], you need to call [method child_controls_changed] manually.
var wrap_controls: bool:
	get = is_wrapping_controls, set = set_wrap_controls



## Virtual method to be implemented by the user. Overrides the value returned by [method get_contents_minimum_size].
func _get_contents_minimum_size() -> Vector2:
	pass;

## Creates a local override for a theme [Color] with the specified [param name]. Local overrides always take precedence when fetching theme items for the control. An override can be removed with [method remove_theme_color_override].
## See also [method get_theme_color] and [method Control.add_theme_color_override] for more details.
func add_theme_color_override(name: StringName, color: Color) -> void:
	pass;

## Creates a local override for a theme constant with the specified [param name]. Local overrides always take precedence when fetching theme items for the control. An override can be removed with [method remove_theme_constant_override].
## See also [method get_theme_constant].
func add_theme_constant_override(name: StringName, constant: int) -> void:
	pass;

## Creates a local override for a theme [Font] with the specified [param name]. Local overrides always take precedence when fetching theme items for the control. An override can be removed with [method remove_theme_font_override].
## See also [method get_theme_font].
func add_theme_font_override(name: StringName, font: Font) -> void:
	pass;

## Creates a local override for a theme font size with the specified [param name]. Local overrides always take precedence when fetching theme items for the control. An override can be removed with [method remove_theme_font_size_override].
## See also [method get_theme_font_size].
func add_theme_font_size_override(name: StringName, font_size: int) -> void:
	pass;

## Creates a local override for a theme icon with the specified [param name]. Local overrides always take precedence when fetching theme items for the control. An override can be removed with [method remove_theme_icon_override].
## See also [method get_theme_icon].
func add_theme_icon_override(name: StringName, texture: Texture2D) -> void:
	pass;

## Creates a local override for a theme [StyleBox] with the specified [param name]. Local overrides always take precedence when fetching theme items for the control. An override can be removed with [method remove_theme_stylebox_override].
## See also [method get_theme_stylebox] and [method Control.add_theme_stylebox_override] for more details.
func add_theme_stylebox_override(name: StringName, stylebox: StyleBox) -> void:
	pass;

## Prevents [code]*_theme_*_override[/code] methods from emitting [constant NOTIFICATION_THEME_CHANGED] until [method end_bulk_theme_override] is called.
func begin_bulk_theme_override() -> void:
	pass;

## Returns whether the window is being drawn to the screen.
func can_draw() -> bool:
	pass;

## Requests an update of the [Window] size to fit underlying [Control] nodes.
func child_controls_changed() -> void:
	pass;

## Ends a bulk theme override update. See [method begin_bulk_theme_override].
func end_bulk_theme_override() -> void:
	pass;

## Returns the combined minimum size from the child [Control] nodes of the window. Use [method child_controls_changed] to update it when children nodes have changed.
## The value returned by this method can be overridden with [method _get_contents_minimum_size].
func get_contents_minimum_size() -> Vector2:
	pass;

## Returns [code]true[/code] if the [param flag] is set.
func get_flag(flag: int) -> bool:
	pass;

## Returns layout direction and text writing direction.
func get_layout_direction() -> int:
	pass;

## Returns the window's position including its border.
func get_position_with_decorations() -> Vector2i:
	pass;

## Returns the window's size including its border.
func get_size_with_decorations() -> Vector2i:
	pass;

## Returns a [Color] from the first matching [Theme] in the tree if that [Theme] has a color item with the specified [param name] and [param theme_type].
## See [method Control.get_theme_color] for more details.
func get_theme_color(name: StringName, theme_type: StringName = "") -> Color:
	pass;

## Returns a constant from the first matching [Theme] in the tree if that [Theme] has a constant item with the specified [param name] and [param theme_type].
## See [method Control.get_theme_color] for more details.
func get_theme_constant(name: StringName, theme_type: StringName = "") -> int:
	pass;

## Returns the default base scale value from the first matching [Theme] in the tree if that [Theme] has a valid [member Theme.default_base_scale] value.
## See [method Control.get_theme_color] for details.
func get_theme_default_base_scale() -> float:
	pass;

## Returns the default font from the first matching [Theme] in the tree if that [Theme] has a valid [member Theme.default_font] value.
## See [method Control.get_theme_color] for details.
func get_theme_default_font() -> Font:
	pass;

## Returns the default font size value from the first matching [Theme] in the tree if that [Theme] has a valid [member Theme.default_font_size] value.
## See [method Control.get_theme_color] for details.
func get_theme_default_font_size() -> int:
	pass;

## Returns a [Font] from the first matching [Theme] in the tree if that [Theme] has a font item with the specified [param name] and [param theme_type].
## See [method Control.get_theme_color] for details.
func get_theme_font(name: StringName, theme_type: StringName = "") -> Font:
	pass;

## Returns a font size from the first matching [Theme] in the tree if that [Theme] has a font size item with the specified [param name] and [param theme_type].
## See [method Control.get_theme_color] for details.
func get_theme_font_size(name: StringName, theme_type: StringName = "") -> int:
	pass;

## Returns an icon from the first matching [Theme] in the tree if that [Theme] has an icon item with the specified [param name] and [param theme_type].
## See [method Control.get_theme_color] for details.
func get_theme_icon(name: StringName, theme_type: StringName = "") -> Texture2D:
	pass;

## Returns a [StyleBox] from the first matching [Theme] in the tree if that [Theme] has a stylebox item with the specified [param name] and [param theme_type].
## See [method Control.get_theme_color] for details.
func get_theme_stylebox(name: StringName, theme_type: StringName = "") -> StyleBox:
	pass;

## Returns the ID of the window.
func get_window_id() -> int:
	pass;

## Causes the window to grab focus, allowing it to receive user input.
func grab_focus() -> void:
	pass;

## Returns [code]true[/code] if the window is focused.
func has_focus() -> bool:
	pass;

## Returns [code]true[/code] if there is a matching [Theme] in the tree that has a color item with the specified [param name] and [param theme_type].
## See [method Control.get_theme_color] for details.
func has_theme_color(name: StringName, theme_type: StringName = "") -> bool:
	pass;

## Returns [code]true[/code] if there is a local override for a theme [Color] with the specified [param name] in this [Control] node.
## See [method add_theme_color_override].
func has_theme_color_override(name: StringName) -> bool:
	pass;

## Returns [code]true[/code] if there is a matching [Theme] in the tree that has a constant item with the specified [param name] and [param theme_type].
## See [method Control.get_theme_color] for details.
func has_theme_constant(name: StringName, theme_type: StringName = "") -> bool:
	pass;

## Returns [code]true[/code] if there is a local override for a theme constant with the specified [param name] in this [Control] node.
## See [method add_theme_constant_override].
func has_theme_constant_override(name: StringName) -> bool:
	pass;

## Returns [code]true[/code] if there is a matching [Theme] in the tree that has a font item with the specified [param name] and [param theme_type].
## See [method Control.get_theme_color] for details.
func has_theme_font(name: StringName, theme_type: StringName = "") -> bool:
	pass;

## Returns [code]true[/code] if there is a local override for a theme [Font] with the specified [param name] in this [Control] node.
## See [method add_theme_font_override].
func has_theme_font_override(name: StringName) -> bool:
	pass;

## Returns [code]true[/code] if there is a matching [Theme] in the tree that has a font size item with the specified [param name] and [param theme_type].
## See [method Control.get_theme_color] for details.
func has_theme_font_size(name: StringName, theme_type: StringName = "") -> bool:
	pass;

## Returns [code]true[/code] if there is a local override for a theme font size with the specified [param name] in this [Control] node.
## See [method add_theme_font_size_override].
func has_theme_font_size_override(name: StringName) -> bool:
	pass;

## Returns [code]true[/code] if there is a matching [Theme] in the tree that has an icon item with the specified [param name] and [param theme_type].
## See [method Control.get_theme_color] for details.
func has_theme_icon(name: StringName, theme_type: StringName = "") -> bool:
	pass;

## Returns [code]true[/code] if there is a local override for a theme icon with the specified [param name] in this [Control] node.
## See [method add_theme_icon_override].
func has_theme_icon_override(name: StringName) -> bool:
	pass;

## Returns [code]true[/code] if there is a matching [Theme] in the tree that has a stylebox item with the specified [param name] and [param theme_type].
## See [method Control.get_theme_color] for details.
func has_theme_stylebox(name: StringName, theme_type: StringName = "") -> bool:
	pass;

## Returns [code]true[/code] if there is a local override for a theme [StyleBox] with the specified [param name] in this [Control] node.
## See [method add_theme_stylebox_override].
func has_theme_stylebox_override(name: StringName) -> bool:
	pass;

## Hides the window. This is not the same as minimized state. Hidden window can't be interacted with and needs to be made visible with [method show].
func hide() -> void:
	pass;

## Returns [code]true[/code] if the window is currently embedded in another window.
func is_embedded() -> bool:
	pass;

## Returns [code]true[/code] if layout is right-to-left.
func is_layout_rtl() -> bool:
	pass;

## Returns [code]true[/code] if the window can be maximized (the maximize button is enabled).
func is_maximize_allowed() -> bool:
	pass;

## Returns [code]true[/code] if font oversampling is enabled. See [method set_use_font_oversampling].
func is_using_font_oversampling() -> bool:
	pass;

## Centers a native window on the current screen and an embedded window on its embedder [Viewport].
func move_to_center() -> void:
	pass;

## Moves the [Window] on top of other windows and focuses it.
func move_to_foreground() -> void:
	pass;

## Shows the [Window] and makes it transient (see [member transient]). If [param rect] is provided, it will be set as the [Window]'s size. Fails if called on the main window.
func popup(rect: Rect2i = Rect2i(0, 0, 0, 0)) -> void:
	pass;

## Popups the [Window] at the center of the current screen, with optionally given minimum size. If the [Window] is embedded, it will be centered in the parent [Viewport] instead.
## [b]Note:[/b] Calling it with the default value of [param minsize] is equivalent to calling it with [member size].
func popup_centered(minsize: Vector2i = Vector2i(0, 0)) -> void:
	pass;

## Popups the [Window] centered inside its parent [Window]. [param fallback_ratio] determines the maximum size of the [Window], in relation to its parent.
## [b]Note:[/b] Calling it with the default value of [param minsize] is equivalent to calling it with [member size].
func popup_centered_clamped(minsize: Vector2i = Vector2i(0, 0), fallback_ratio: float = 0.75) -> void:
	pass;

## If [Window] is embedded, popups the [Window] centered inside its embedder and sets its size as a [param ratio] of embedder's size.
## If [Window] is a native window, popups the [Window] centered inside the screen of its parent [Window] and sets its size as a [param ratio] of the screen size.
func popup_centered_ratio(ratio: float = 0.8) -> void:
	pass;

## Attempts to parent this dialog to the last exclusive window relative to [param from_node], and then calls [method Window.popup] on it. The dialog must have no current parent, otherwise the method fails.
## See also [method set_unparent_when_invisible] and [method Node.get_last_exclusive_window].
func popup_exclusive(from_node: Node, rect: Rect2i = Rect2i(0, 0, 0, 0)) -> void:
	pass;

## Attempts to parent this dialog to the last exclusive window relative to [param from_node], and then calls [method Window.popup_centered] on it. The dialog must have no current parent, otherwise the method fails.
## See also [method set_unparent_when_invisible] and [method Node.get_last_exclusive_window].
func popup_exclusive_centered(from_node: Node, minsize: Vector2i = Vector2i(0, 0)) -> void:
	pass;

## Attempts to parent this dialog to the last exclusive window relative to [param from_node], and then calls [method Window.popup_centered_clamped] on it. The dialog must have no current parent, otherwise the method fails.
## See also [method set_unparent_when_invisible] and [method Node.get_last_exclusive_window].
func popup_exclusive_centered_clamped(from_node: Node, minsize: Vector2i = Vector2i(0, 0), fallback_ratio: float = 0.75) -> void:
	pass;

## Attempts to parent this dialog to the last exclusive window relative to [param from_node], and then calls [method Window.popup_centered_ratio] on it. The dialog must have no current parent, otherwise the method fails.
## See also [method set_unparent_when_invisible] and [method Node.get_last_exclusive_window].
func popup_exclusive_centered_ratio(from_node: Node, ratio: float = 0.8) -> void:
	pass;

## Attempts to parent this dialog to the last exclusive window relative to [param from_node], and then calls [method Window.popup_on_parent] on it. The dialog must have no current parent, otherwise the method fails.
## See also [method set_unparent_when_invisible] and [method Node.get_last_exclusive_window].
func popup_exclusive_on_parent(from_node: Node, parent_rect: Rect2i) -> void:
	pass;

## Popups the [Window] with a position shifted by parent [Window]'s position. If the [Window] is embedded, has the same effect as [method popup].
func popup_on_parent(parent_rect: Rect2i) -> void:
	pass;

## Removes a local override for a theme [Color] with the specified [param name] previously added by [method add_theme_color_override] or via the Inspector dock.
func remove_theme_color_override(name: StringName) -> void:
	pass;

## Removes a local override for a theme constant with the specified [param name] previously added by [method add_theme_constant_override] or via the Inspector dock.
func remove_theme_constant_override(name: StringName) -> void:
	pass;

## Removes a local override for a theme [Font] with the specified [param name] previously added by [method add_theme_font_override] or via the Inspector dock.
func remove_theme_font_override(name: StringName) -> void:
	pass;

## Removes a local override for a theme font size with the specified [param name] previously added by [method add_theme_font_size_override] or via the Inspector dock.
func remove_theme_font_size_override(name: StringName) -> void:
	pass;

## Removes a local override for a theme icon with the specified [param name] previously added by [method add_theme_icon_override] or via the Inspector dock.
func remove_theme_icon_override(name: StringName) -> void:
	pass;

## Removes a local override for a theme [StyleBox] with the specified [param name] previously added by [method add_theme_stylebox_override] or via the Inspector dock.
func remove_theme_stylebox_override(name: StringName) -> void:
	pass;

## Tells the OS that the [Window] needs an attention. This makes the window stand out in some way depending on the system, e.g. it might blink on the task bar.
func request_attention() -> void:
	pass;

## Resets the size to the minimum size, which is the max of [member min_size] and (if [member wrap_controls] is enabled) [method get_contents_minimum_size]. This is equivalent to calling [code]set_size(Vector2i())[/code] (or any size below the minimum).
func reset_size() -> void:
	pass;

## Sets a specified window flag.
func set_flag(flag: int, enabled: bool) -> void:
	pass;

## If [param active] is [code]true[/code], enables system's native IME (Input Method Editor).
func set_ime_active(active: bool) -> void:
	pass;

## Moves IME to the given position.
func set_ime_position(position: Vector2i) -> void:
	pass;

## Sets layout direction and text writing direction. Right-to-left layouts are necessary for certain languages (e.g. Arabic and Hebrew).
func set_layout_direction(direction: int) -> void:
	pass;

## If [param unparent] is [code]true[/code], the window is automatically unparented when going invisible.
## [b]Note:[/b] Make sure to keep a reference to the node, otherwise it will be orphaned. You also need to manually call [method Node.queue_free] to free the window if it's not parented.
func set_unparent_when_invisible(unparent: bool) -> void:
	pass;

## Enables font oversampling. This makes fonts look better when they are scaled up.
func set_use_font_oversampling(enable: bool) -> void:
	pass;

## Makes the [Window] appear. This enables interactions with the [Window] and doesn't change any of its property other than visibility (unlike e.g. [method popup]).
func show() -> void:
	pass;


func get_flag() -> bool:
	return always_on_top

func set_flag(value: bool) -> void:
	always_on_top = value

func is_auto_translating() -> bool:
	return auto_translate

func set_auto_translate(value: bool) -> void:
	auto_translate = value

func get_flag() -> bool:
	return borderless

func set_flag(value: bool) -> void:
	borderless = value

func get_content_scale_aspect() -> int:
	return content_scale_aspect

func set_content_scale_aspect(value: int) -> void:
	content_scale_aspect = value

func get_content_scale_factor() -> float:
	return content_scale_factor

func set_content_scale_factor(value: float) -> void:
	content_scale_factor = value

func get_content_scale_mode() -> int:
	return content_scale_mode

func set_content_scale_mode(value: int) -> void:
	content_scale_mode = value

func get_content_scale_size() -> Vector2i:
	return content_scale_size

func set_content_scale_size(value: Vector2i) -> void:
	content_scale_size = value

func get_content_scale_stretch() -> int:
	return content_scale_stretch

func set_content_scale_stretch(value: int) -> void:
	content_scale_stretch = value

func get_current_screen() -> int:
	return current_screen

func set_current_screen(value: int) -> void:
	current_screen = value

func is_exclusive() -> bool:
	return exclusive

func set_exclusive(value: bool) -> void:
	exclusive = value

func get_flag() -> bool:
	return extend_to_title

func set_flag(value: bool) -> void:
	extend_to_title = value

func get_initial_position() -> int:
	return initial_position

func set_initial_position(value: int) -> void:
	initial_position = value

func get_keep_title_visible() -> bool:
	return keep_title_visible

func set_keep_title_visible(value: bool) -> void:
	keep_title_visible = value

func get_max_size() -> Vector2i:
	return max_size

func set_max_size(value: Vector2i) -> void:
	max_size = value

func get_min_size() -> Vector2i:
	return min_size

func set_min_size(value: Vector2i) -> void:
	min_size = value

func get_mode() -> int:
	return mode

func set_mode(value: int) -> void:
	mode = value

func get_flag() -> bool:
	return mouse_passthrough

func set_flag(value: bool) -> void:
	mouse_passthrough = value

func get_mouse_passthrough_polygon() -> PackedVector2Array:
	return mouse_passthrough_polygon

func set_mouse_passthrough_polygon(value: PackedVector2Array) -> void:
	mouse_passthrough_polygon = value

func get_flag() -> bool:
	return popup_window

func set_flag(value: bool) -> void:
	popup_window = value

func get_position() -> Vector2i:
	return position

func set_position(value: Vector2i) -> void:
	position = value

func get_size() -> Vector2i:
	return size

func set_size(value: Vector2i) -> void:
	size = value

func get_theme() -> Theme:
	return theme

func set_theme(value: Theme) -> void:
	theme = value

func get_theme_type_variation() -> StringName:
	return theme_type_variation

func set_theme_type_variation(value: StringName) -> void:
	theme_type_variation = value

func get_title() -> String:
	return title

func set_title(value: String) -> void:
	title = value

func is_transient() -> bool:
	return transient

func set_transient(value: bool) -> void:
	transient = value

func get_flag() -> bool:
	return transparent

func set_flag(value: bool) -> void:
	transparent = value

func get_flag() -> bool:
	return unfocusable

func set_flag(value: bool) -> void:
	unfocusable = value

func get_flag() -> bool:
	return unresizable

func set_flag(value: bool) -> void:
	unresizable = value

func is_visible() -> bool:
	return visible

func set_visible(value: bool) -> void:
	visible = value

func is_wrapping_controls() -> bool:
	return wrap_controls

func set_wrap_controls(value: bool) -> void:
	wrap_controls = value

