extends XRInterface
class_name OpenXRInterface

## Our OpenXR interface.
##
## The OpenXR interface allows Godot to interact with OpenXR runtimes and make it possible to create XR experiences and games.
## Due to the needs of OpenXR this interface works slightly different than other plugin based XR interfaces. It needs to be initialized when Godot starts. You need to enable OpenXR, settings for this can be found in your games project settings under the XR heading. You do need to mark a viewport for use with XR in order for Godot to know which render result should be output to the headset.
##
## @tutorial(Setting up XR): https://docs.godotengine.org/en/stable/tutorials/xr/setting_up_xr.html


## Informs the user queued a recenter of the player position.
signal pose_recentered
## Informs our OpenXR session has been started.
signal session_begun
## Informs our OpenXR session now has focus.
signal session_focussed
## Informs our OpenXR session is stopping.
signal session_stopping
## Informs our OpenXR session is now visible (output is being sent to the HMD).
signal session_visible
## Left hand.
## Right hand.
## Maximum value for the hand enum.
## Palm joint.
## Wrist joint.
## Thumb metacarpal joint.
## Thumb proximal joint.
## Thumb distal joint.
## Thumb tip joint.
## Index metacarpal joint.
## Index proximal joint.
## Index intermediate joint.
## Index distal joint.
## Index tip joint.
## Middle metacarpal joint.
## Middle proximal joint.
## Middle intermediate joint.
## Middle distal joint.
## Middle tip joint.
## Ring metacarpal joint.
## Ring proximal joint.
## Ring intermediate joint.
## Ring distal joint.
## Ring tip joint.
## Little metacarpal joint.
## Little proximal joint.
## Little intermediate joint.
## Little distal joint.
## Little tip joint.
## Maximum value for the hand joint enum.
## No flags are set.
## If set, the orientation data is valid, otherwise, the orientation data is unreliable and should not be used.
## If set, the orientation data comes from tracking data, otherwise, the orientation data contains predicted data.
## If set, the positional data is valid, otherwise, the positional data is unreliable and should not be used.
## If set, the positional data comes from tracking data, otherwise, the positional data contains predicted data.
## If set, our linear velocity data is valid, otherwise, the linear velocity data is unreliable and should not be used.
## If set, our angular velocity data is valid, otherwise, the angular velocity data is unreliable and should not be used.

#enum Hand
enum {
    HAND_LEFT = 0,
    HAND_RIGHT = 1,
    HAND_MAX = 2,
}
#enum HandMotionRange
enum {
    HAND_MOTION_RANGE_UNOBSTRUCTED = 0,
    HAND_MOTION_RANGE_CONFORM_TO_CONTROLLER = 1,
    HAND_MOTION_RANGE_MAX = 2,
}
#enum HandJoints
enum {
    HAND_JOINT_PALM = 0,
    HAND_JOINT_WRIST = 1,
    HAND_JOINT_THUMB_METACARPAL = 2,
    HAND_JOINT_THUMB_PROXIMAL = 3,
    HAND_JOINT_THUMB_DISTAL = 4,
    HAND_JOINT_THUMB_TIP = 5,
    HAND_JOINT_INDEX_METACARPAL = 6,
    HAND_JOINT_INDEX_PROXIMAL = 7,
    HAND_JOINT_INDEX_INTERMEDIATE = 8,
    HAND_JOINT_INDEX_DISTAL = 9,
    HAND_JOINT_INDEX_TIP = 10,
    HAND_JOINT_MIDDLE_METACARPAL = 11,
    HAND_JOINT_MIDDLE_PROXIMAL = 12,
    HAND_JOINT_MIDDLE_INTERMEDIATE = 13,
    HAND_JOINT_MIDDLE_DISTAL = 14,
    HAND_JOINT_MIDDLE_TIP = 15,
    HAND_JOINT_RING_METACARPAL = 16,
    HAND_JOINT_RING_PROXIMAL = 17,
    HAND_JOINT_RING_INTERMEDIATE = 18,
    HAND_JOINT_RING_DISTAL = 19,
    HAND_JOINT_RING_TIP = 20,
    HAND_JOINT_LITTLE_METACARPAL = 21,
    HAND_JOINT_LITTLE_PROXIMAL = 22,
    HAND_JOINT_LITTLE_INTERMEDIATE = 23,
    HAND_JOINT_LITTLE_DISTAL = 24,
    HAND_JOINT_LITTLE_TIP = 25,
    HAND_JOINT_MAX = 26,
}
#enum HandJointFlags
enum {
    HAND_JOINT_NONE = 0,
    HAND_JOINT_ORIENTATION_VALID = 1,
    HAND_JOINT_ORIENTATION_TRACKED = 2,
    HAND_JOINT_POSITION_VALID = 4,
    HAND_JOINT_POSITION_TRACKED = 8,
    HAND_JOINT_LINEAR_VELOCITY_VALID = 16,
    HAND_JOINT_ANGULAR_VELOCITY_VALID = 32,
}
## The display refresh rate for the current HMD. Only functional if this feature is supported by the OpenXR runtime and after the interface has been initialized.
var display_refresh_rate: float:
	get = get_display_refresh_rate, set = set_display_refresh_rate

## Enable dynamic foveation adjustment, the interface must be initialized before this is accessible. If enabled foveation will automatically adjusted between low and [member foveation_level].
var foveation_dynamic: bool:
	get = get_foveation_dynamic, set = set_foveation_dynamic

## Set foveation level from 0 (off) to 3 (high), the interface must be initialized before this is accessible.
var foveation_level: int:
	get = get_foveation_level, set = set_foveation_level

## The render size multiplier for the current HMD. Must be set before the interface has been initialized.
var render_target_size_multiplier: float:
	get = get_render_target_size_multiplier, set = set_render_target_size_multiplier



## Returns a list of action sets registered with Godot (loaded from the action map at runtime).
func get_action_sets() -> Array:
	pass;

## Returns display refresh rates supported by the current HMD. Only returned if this feature is supported by the OpenXR runtime and after the interface has been initialized.
func get_available_display_refresh_rates() -> Array:
	pass;

## If handtracking is enabled, returns the angular velocity of a joint ([param joint]) of a hand ([param hand]) as provided by OpenXR. This is relative to [XROrigin3D]!
func get_hand_joint_angular_velocity(hand: int, joint: int) -> Vector3:
	pass;

## If handtracking is enabled, returns flags that inform us of the validity of the tracking data.
func get_hand_joint_flags(hand: int, joint: int) -> int:
	pass;

## If handtracking is enabled, returns the linear velocity of a joint ([param joint]) of a hand ([param hand]) as provided by OpenXR. This is relative to [XROrigin3D] without worldscale applied!
func get_hand_joint_linear_velocity(hand: int, joint: int) -> Vector3:
	pass;

## If handtracking is enabled, returns the position of a joint ([param joint]) of a hand ([param hand]) as provided by OpenXR. This is relative to [XROrigin3D] without worldscale applied!
func get_hand_joint_position(hand: int, joint: int) -> Vector3:
	pass;

## If handtracking is enabled, returns the radius of a joint ([param joint]) of a hand ([param hand]) as provided by OpenXR. This is without worldscale applied!
func get_hand_joint_radius(hand: int, joint: int) -> float:
	pass;

## If handtracking is enabled, returns the rotation of a joint ([param joint]) of a hand ([param hand]) as provided by OpenXR.
func get_hand_joint_rotation(hand: int, joint: int) -> Quaternion:
	pass;

## If handtracking is enabled and motion range is supported, gets the currently configured motion range for [param hand].
func get_motion_range(hand: int) -> int:
	pass;

## Returns [code]true[/code] if the given action set is active.
func is_action_set_active(name: String) -> bool:
	pass;

## Returns the capabilities of the eye gaze interaction extension.
## [b]Note:[/b] This only returns a valid value after OpenXR has been initialized.
func is_eye_gaze_interaction_supported() -> bool:
	pass;

## Returns [code]true[/code] if OpenXR's foveation extension is supported, the interface must be initialized before this returns a valid value.
## [b]Note:[/b] This feature is only available on the compatibility renderer and currently only available on some stand alone headsets. For Vulkan set [member Viewport.vrs_mode] to [code]VRS_XR[/code] on desktop.
func is_foveation_supported() -> bool:
	pass;

## Returns [code]true[/code] if OpenXR's hand tracking is supported and enabled.
## [b]Note:[/b] This only returns a valid value after OpenXR has been initialized.
func is_hand_tracking_supported() -> bool:
	pass;

## Sets the given action set as active or inactive.
func set_action_set_active(name: String, active: bool) -> void:
	pass;

## If handtracking is enabled and motion range is supported, sets the currently configured motion range for [param hand] to [param motion_range].
func set_motion_range(hand: int, motion_range: int) -> void:
	pass;


func get_display_refresh_rate() -> float:
	return display_refresh_rate

func set_display_refresh_rate(value: float) -> void:
	display_refresh_rate = value

func get_foveation_dynamic() -> bool:
	return foveation_dynamic

func set_foveation_dynamic(value: bool) -> void:
	foveation_dynamic = value

func get_foveation_level() -> int:
	return foveation_level

func set_foveation_level(value: int) -> void:
	foveation_level = value

func get_render_target_size_multiplier() -> float:
	return render_target_size_multiplier

func set_render_target_size_multiplier(value: float) -> void:
	render_target_size_multiplier = value

