extends Control
class_name ItemList

## A vertical list of selectable items with one or multiple columns.
##
## This control provides a vertical list of selectable items that may be in a single or in multiple columns, with each item having options for text and an icon. Tooltips are supported and may be different for every item in the list.
## Selectable items in the list may be selected or deselected and multiple selection may be enabled. Selection with right mouse button may also be enabled to allow use of popup context menus. Items may also be "activated" by double-clicking them or by pressing [kbd]Enter[/kbd].
## Item text only supports single-line strings. Newline characters (e.g. [code]\n[/code]) in the string won't produce a newline. Text wrapping is enabled in [constant ICON_MODE_TOP] mode, but the column's width is adjusted to fully fit its content by default. You need to set [member fixed_column_width] greater than zero to wrap the text.
## All [code]set_*[/code] methods allow negative item indices, i.e. [code]-1[/code] to access the last item, [code]-2[/code] to select the second-to-last item, and so on.
## [b]Incremental search:[/b] Like [PopupMenu] and [Tree], [ItemList] supports searching within the list while the control is focused. Press a key that matches the first letter of an item's name to select the first item starting with the given letter. After that point, there are two ways to perform incremental search: 1) Press the same key again before the timeout duration to select the next item starting with the same letter. 2) Press letter keys that match the rest of the word before the timeout duration to match to select the item in question directly. Both of these actions will be reset to the beginning of the list if the timeout duration has passed since the last keystroke was registered. You can adjust the timeout duration by changing [member ProjectSettings.gui/timers/incremental_search_max_interval_msec].


## Triggered when any mouse click is issued within the rect of the list but on empty space.
signal empty_clicked(at_position: Vector2, mouse_button_index: int)
## Triggered when specified list item is activated via double-clicking or by pressing [kbd]Enter[/kbd].
signal item_activated(index: int)
## Triggered when specified list item has been clicked with any mouse button.
## The click position is also provided to allow appropriate popup of context menus at the correct location.
signal item_clicked(index: int, at_position: Vector2, mouse_button_index: int)
## Triggered when specified item has been selected.
## [member allow_reselect] must be enabled to reselect an item.
signal item_selected(index: int)
## Triggered when a multiple selection is altered on a list allowing multiple selection.
signal multi_selected(index: int, selected: bool)
## Icon is drawn above the text.
## Icon is drawn to the left of the text.
## Only allow selecting a single item.
## Allows selecting multiple items by holding [kbd]Ctrl[/kbd] or [kbd]Shift[/kbd].

#enum IconMode
enum {
    ICON_MODE_TOP = 0,
    ICON_MODE_LEFT = 1,
}
#enum SelectMode
enum {
    SELECT_SINGLE = 0,
    SELECT_MULTI = 1,
}
## If [code]true[/code], the currently selected item can be selected again.
var allow_reselect: bool:
	get = get_allow_reselect, set = set_allow_reselect

## If [code]true[/code], right mouse button click can select items.
var allow_rmb_select: bool:
	get = get_allow_rmb_select, set = set_allow_rmb_select

## If [code]true[/code], allows navigating the [ItemList] with letter keys through incremental search.
var allow_search: bool:
	get = get_allow_search, set = set_allow_search

## If [code]true[/code], the control will automatically resize the height to fit its content.
var auto_height: bool:
	get = has_auto_height, set = set_auto_height

var clip_contents: bool:
	get = is_clipping_contents, set = set_clip_contents

## The width all columns will be adjusted to.
## A value of zero disables the adjustment, each item will have a width equal to the width of its content and the columns will have an uneven width.
var fixed_column_width: int:
	get = get_fixed_column_width, set = set_fixed_column_width

## The size all icons will be adjusted to.
## If either X or Y component is not greater than zero, icon size won't be affected.
var fixed_icon_size: Vector2i:
	get = get_fixed_icon_size, set = set_fixed_icon_size

var focus_mode: int:
	get = get_focus_mode, set = set_focus_mode

## The icon position, whether above or to the left of the text. See the [enum IconMode] constants.
var icon_mode: int:
	get = get_icon_mode, set = set_icon_mode

## The scale of icon applied after [member fixed_icon_size] and transposing takes effect.
var icon_scale: float:
	get = get_icon_scale, set = set_icon_scale

## The number of items currently in the list.
var item_count: int:
	get = get_item_count, set = set_item_count

## Maximum columns the list will have.
## If greater than zero, the content will be split among the specified columns.
## A value of zero means unlimited columns, i.e. all items will be put in the same row.
var max_columns: int:
	get = get_max_columns, set = set_max_columns

## Maximum lines of text allowed in each item. Space will be reserved even when there is not enough lines of text to display.
## [b]Note:[/b] This property takes effect only when [member icon_mode] is [constant ICON_MODE_TOP]. To make the text wrap, [member fixed_column_width] should be greater than zero.
var max_text_lines: int:
	get = get_max_text_lines, set = set_max_text_lines

## Whether all columns will have the same width.
## If [code]true[/code], the width is equal to the largest column width of all columns.
var same_column_width: bool:
	get = is_same_column_width, set = set_same_column_width

## Allows single or multiple item selection. See the [enum SelectMode] constants.
var select_mode: int:
	get = get_select_mode, set = set_select_mode

## Sets the clipping behavior when the text exceeds an item's bounding rectangle. See [enum TextServer.OverrunBehavior] for a description of all modes.
var text_overrun_behavior: int:
	get = get_text_overrun_behavior, set = set_text_overrun_behavior



## Adds an item to the item list with no text, only an icon. Returns the index of an added item.
func add_icon_item(icon: Texture2D, selectable: bool = true) -> int:
	pass;

## Adds an item to the item list with specified text. Returns the index of an added item.
## Specify an [param icon], or use [code]null[/code] as the [param icon] for a list item with no icon.
## If selectable is [code]true[/code], the list item will be selectable.
func add_item(text: String, icon: Texture2D = null, selectable: bool = true) -> int:
	pass;

## Removes all items from the list.
func clear() -> void:
	pass;

## Ensures the item associated with the specified index is not selected.
func deselect(idx: int) -> void:
	pass;

## Ensures there are no items selected.
func deselect_all() -> void:
	pass;

## Ensure current selection is visible, adjusting the scroll position as necessary.
func ensure_current_is_visible() -> void:
	pass;

## Forces an update to the list size based on its items. This happens automatically whenever size of the items, or other relevant settings like [member auto_height], change. The method can be used to trigger the update ahead of next drawing pass.
func force_update_list_size() -> void:
	pass;

## Returns the item index at the given [param position].
## When there is no item at that point, -1 will be returned if [param exact] is [code]true[/code], and the closest item index will be returned otherwise.
## [b]Note:[/b] The returned value is unreliable if called right after modifying the [ItemList], before it redraws in the next frame.
func get_item_at_position(position: Vector2, exact: bool = false) -> int:
	pass;

## Returns the custom background color of the item specified by [param idx] index.
func get_item_custom_bg_color(idx: int) -> Color:
	pass;

## Returns the custom foreground color of the item specified by [param idx] index.
func get_item_custom_fg_color(idx: int) -> Color:
	pass;

## Returns the icon associated with the specified index.
func get_item_icon(idx: int) -> Texture2D:
	pass;

## Returns a [Color] modulating item's icon at the specified index.
func get_item_icon_modulate(idx: int) -> Color:
	pass;

## Returns the region of item's icon used. The whole icon will be used if the region has no area.
func get_item_icon_region(idx: int) -> Rect2:
	pass;

## Returns item's text language code.
func get_item_language(idx: int) -> String:
	pass;

## Returns the metadata value of the specified index.
func get_item_metadata(idx: int) -> Variant:
	pass;

## Returns the position and size of the item with the specified index, in the coordinate system of the [ItemList] node. If [param expand] is [code]true[/code] the last column expands to fill the rest of the row.
## [b]Note:[/b] The returned value is unreliable if called right after modifying the [ItemList], before it redraws in the next frame.
func get_item_rect(idx: int, expand: bool = true) -> Rect2:
	pass;

## Returns the text associated with the specified index.
func get_item_text(idx: int) -> String:
	pass;

## Returns item's text base writing direction.
func get_item_text_direction(idx: int) -> int:
	pass;

## Returns the tooltip hint associated with the specified index.
func get_item_tooltip(idx: int) -> String:
	pass;

## Returns an array with the indexes of the selected items.
func get_selected_items() -> PackedInt32Array:
	pass;

## Returns the vertical scrollbar.
## [b]Warning:[/b] This is a required internal node, removing and freeing it may cause a crash. If you wish to hide it or any of its children, use their [member CanvasItem.visible] property.
func get_v_scroll_bar() -> VScrollBar:
	pass;

## Returns [code]true[/code] if one or more items are selected.
func is_anything_selected() -> bool:
	pass;

## Returns [code]true[/code] if the item at the specified index is disabled.
func is_item_disabled(idx: int) -> bool:
	pass;

## Returns [code]true[/code] if the item icon will be drawn transposed, i.e. the X and Y axes are swapped.
func is_item_icon_transposed(idx: int) -> bool:
	pass;

## Returns [code]true[/code] if the item at the specified index is selectable.
func is_item_selectable(idx: int) -> bool:
	pass;

## Returns [code]true[/code] if the tooltip is enabled for specified item index.
func is_item_tooltip_enabled(idx: int) -> bool:
	pass;

## Returns [code]true[/code] if the item at the specified index is currently selected.
func is_selected(idx: int) -> bool:
	pass;

## Moves item from index [param from_idx] to [param to_idx].
func move_item(from_idx: int, to_idx: int) -> void:
	pass;

## Removes the item specified by [param idx] index from the list.
func remove_item(idx: int) -> void:
	pass;

## Select the item at the specified index.
## [b]Note:[/b] This method does not trigger the item selection signal.
func select(idx: int, single: bool = true) -> void:
	pass;

## Sets the background color of the item specified by [param idx] index to the specified [Color].
func set_item_custom_bg_color(idx: int, custom_bg_color: Color) -> void:
	pass;

## Sets the foreground color of the item specified by [param idx] index to the specified [Color].
func set_item_custom_fg_color(idx: int, custom_fg_color: Color) -> void:
	pass;

## Disables (or enables) the item at the specified index.
## Disabled items cannot be selected and do not trigger activation signals (when double-clicking or pressing [kbd]Enter[/kbd]).
func set_item_disabled(idx: int, disabled: bool) -> void:
	pass;

## Sets (or replaces) the icon's [Texture2D] associated with the specified index.
func set_item_icon(idx: int, icon: Texture2D) -> void:
	pass;

## Sets a modulating [Color] of the item associated with the specified index.
func set_item_icon_modulate(idx: int, modulate: Color) -> void:
	pass;

## Sets the region of item's icon used. The whole icon will be used if the region has no area.
func set_item_icon_region(idx: int, rect: Rect2) -> void:
	pass;

## Sets whether the item icon will be drawn transposed.
func set_item_icon_transposed(idx: int, transposed: bool) -> void:
	pass;

## Sets language code of item's text used for line-breaking and text shaping algorithms, if left empty current locale is used instead.
func set_item_language(idx: int, language: String) -> void:
	pass;

## Sets a value (of any type) to be stored with the item associated with the specified index.
func set_item_metadata(idx: int, metadata: Variant) -> void:
	pass;

## Allows or disallows selection of the item associated with the specified index.
func set_item_selectable(idx: int, selectable: bool) -> void:
	pass;

## Sets text of the item associated with the specified index.
func set_item_text(idx: int, text: String) -> void:
	pass;

## Sets item's text base writing direction.
func set_item_text_direction(idx: int, direction: int) -> void:
	pass;

## Sets the tooltip hint for the item associated with the specified index.
func set_item_tooltip(idx: int, tooltip: String) -> void:
	pass;

## Sets whether the tooltip hint is enabled for specified item index.
func set_item_tooltip_enabled(idx: int, enable: bool) -> void:
	pass;

## Sorts items in the list by their text.
func sort_items_by_text() -> void:
	pass;


func get_allow_reselect() -> bool:
	return allow_reselect

func set_allow_reselect(value: bool) -> void:
	allow_reselect = value

func get_allow_rmb_select() -> bool:
	return allow_rmb_select

func set_allow_rmb_select(value: bool) -> void:
	allow_rmb_select = value

func get_allow_search() -> bool:
	return allow_search

func set_allow_search(value: bool) -> void:
	allow_search = value

func has_auto_height() -> bool:
	return auto_height

func set_auto_height(value: bool) -> void:
	auto_height = value

func is_clipping_contents() -> bool:
	return clip_contents

func set_clip_contents(value: bool) -> void:
	clip_contents = value

func get_fixed_column_width() -> int:
	return fixed_column_width

func set_fixed_column_width(value: int) -> void:
	fixed_column_width = value

func get_fixed_icon_size() -> Vector2i:
	return fixed_icon_size

func set_fixed_icon_size(value: Vector2i) -> void:
	fixed_icon_size = value

func get_focus_mode() -> int:
	return focus_mode

func set_focus_mode(value: int) -> void:
	focus_mode = value

func get_icon_mode() -> int:
	return icon_mode

func set_icon_mode(value: int) -> void:
	icon_mode = value

func get_icon_scale() -> float:
	return icon_scale

func set_icon_scale(value: float) -> void:
	icon_scale = value

func get_item_count() -> int:
	return item_count

func set_item_count(value: int) -> void:
	item_count = value

func get_max_columns() -> int:
	return max_columns

func set_max_columns(value: int) -> void:
	max_columns = value

func get_max_text_lines() -> int:
	return max_text_lines

func set_max_text_lines(value: int) -> void:
	max_text_lines = value

func is_same_column_width() -> bool:
	return same_column_width

func set_same_column_width(value: bool) -> void:
	same_column_width = value

func get_select_mode() -> int:
	return select_mode

func set_select_mode(value: int) -> void:
	select_mode = value

func get_text_overrun_behavior() -> int:
	return text_overrun_behavior

func set_text_overrun_behavior(value: int) -> void:
	text_overrun_behavior = value

