class_name Plane

## A plane in Hessian normal form.
##
## Represents a normalized plane equation. [member normal] is the normal of the plane (a, b, c normalized), and [member d] is the distance from the origin to the plane (in the direction of "normal"). "Over" or "Above" the plane is considered the side of the plane towards where the normal is pointing.
##
## @tutorial(Math documentation index): https://docs.godotengine.org/en/stable/tutorials/math/index.html


## A plane that extends in the Y and Z axes (normal vector points +X).
const PLANE_YZ = Plane(1, 0, 0, 0);

## A plane that extends in the X and Z axes (normal vector points +Y).
const PLANE_XZ = Plane(0, 1, 0, 0);

## A plane that extends in the X and Y axes (normal vector points +Z).
const PLANE_XY = Plane(0, 0, 1, 0);


## The distance from the origin to the plane, expressed in terms of [member normal] (according to its direction and magnitude). Actual absolute distance from the origin to the plane can be calculated as [code]abs(d) / normal.length()[/code] (if [member normal] has zero length then this [Plane] does not represent a valid plane).
## In the scalar equation of the plane [code]ax + by + cz = d[/code], this is [code]d[/code], while the [code](a, b, c)[/code] coordinates are represented by the [member normal] property.
var d: float

## The normal of the plane, typically a unit vector. Shouldn't be a zero vector as [Plane] with such [member normal] does not represent a valid plane.
## In the scalar equation of the plane [code]ax + by + cz = d[/code], this is the vector [code](a, b, c)[/code], where [code]d[/code] is the [member d] property.
var normal: Vector3

## The X component of the plane's [member normal] vector.
var x: float

## The Y component of the plane's [member normal] vector.
var y: float

## The Z component of the plane's [member normal] vector.
var z: float


## Constructs a default-initialized [Plane] with all components set to [code]0[/code].
func Plane() -> Plane:
	pass;

## Constructs a [Plane] as a copy of the given [Plane].
func Plane(from: Plane) -> Plane:
	pass;

## Creates a plane from the four parameters. The three components of the resulting plane's [member normal] are [param a], [param b] and [param c], and the plane has a distance of [param d] from the origin.
func Plane(a: float, b: float, c: float, d: float) -> Plane:
	pass;

## Creates a plane from the normal vector. The plane will intersect the origin.
## The [param normal] of the plane must be a unit vector.
func Plane(normal: Vector3) -> Plane:
	pass;

## Creates a plane from the normal vector and the plane's distance from the origin.
## The [param normal] of the plane must be a unit vector.
func Plane(normal: Vector3, d: float) -> Plane:
	pass;

## Creates a plane from the normal vector and a point on the plane.
## The [param normal] of the plane must be a unit vector.
func Plane(normal: Vector3, point: Vector3) -> Plane:
	pass;

## Creates a plane from the three points, given in clockwise order.
func Plane(point1: Vector3, point2: Vector3, point3: Vector3) -> Plane:
	pass;


## Returns the shortest distance from the plane to the position [param point]. If the point is above the plane, the distance will be positive. If below, the distance will be negative.
func distance_to(point: Vector3) -> float:
	pass;

## Returns the center of the plane.
func get_center() -> Vector3:
	pass;

## Returns [code]true[/code] if [param point] is inside the plane. Comparison uses a custom minimum [param tolerance] threshold.
func has_point(point: Vector3, tolerance: float = 1e-05) -> bool:
	pass;

## Returns the intersection point of the three planes [param b], [param c] and this plane. If no intersection is found, [code]null[/code] is returned.
func intersect_3(b: Plane, c: Plane) -> Variant:
	pass;

## Returns the intersection point of a ray consisting of the position [param from] and the direction normal [param dir] with this plane. If no intersection is found, [code]null[/code] is returned.
func intersects_ray(from: Vector3, dir: Vector3) -> Variant:
	pass;

## Returns the intersection point of a segment from position [param from] to position [param to] with this plane. If no intersection is found, [code]null[/code] is returned.
func intersects_segment(from: Vector3, to: Vector3) -> Variant:
	pass;

## Returns [code]true[/code] if this plane and [param to_plane] are approximately equal, by running [method @GlobalScope.is_equal_approx] on each component.
func is_equal_approx(to_plane: Plane) -> bool:
	pass;

## Returns [code]true[/code] if this plane is finite, by calling [method @GlobalScope.is_finite] on each component.
func is_finite() -> bool:
	pass;

## Returns [code]true[/code] if [param point] is located above the plane.
func is_point_over(point: Vector3) -> bool:
	pass;

## Returns a copy of the plane, with normalized [member normal] (so it's a unit vector). Returns [code]Plane(0, 0, 0, 0)[/code] if [member normal] can't be normalized (it has zero length).
func normalized() -> Plane:
	pass;

## Returns the orthogonal projection of [param point] into a point in the plane.
func project(point: Vector3) -> Vector3:
	pass;


