class_name Basis

## A 3×3 matrix for representing 3D rotation and scale.
##
## A 3×3 matrix used for representing 3D rotation and scale. Usually used as an orthogonal basis for a [Transform3D].
## Contains 3 vector fields X, Y and Z as its columns, which are typically interpreted as the local basis vectors of a transformation. For such use, it is composed of a scaling and a rotation matrix, in that order (M = R.S).
## Basis can also be accessed as an array of 3D vectors. These vectors are usually orthogonal to each other, but are not necessarily normalized (due to scaling).
## For more information, read the "Matrices and transforms" documentation article.
##
## @tutorial(Math documentation index): https://docs.godotengine.org/en/stable/tutorials/math/index.html
## @tutorial(Matrices and transforms): https://docs.godotengine.org/en/stable/tutorials/math/matrices_and_transforms.html
## @tutorial(Using 3D transforms): https://docs.godotengine.org/en/stable/tutorials/3d/using_transforms.html
## @tutorial(Matrix Transform Demo): https://godotengine.org/asset-library/asset/584
## @tutorial(3D Platformer Demo): https://godotengine.org/asset-library/asset/125
## @tutorial(3D Voxel Demo): https://godotengine.org/asset-library/asset/676
## @tutorial(2.5D Demo): https://godotengine.org/asset-library/asset/583


## The identity basis, with no rotation or scaling applied.
## This is identical to calling [code]Basis()[/code] without any parameters. This constant can be used to make your code clearer, and for consistency with C#.
const IDENTITY = Basis(1, 0, 0, 0, 1, 0, 0, 0, 1);

## The basis that will flip something along the X axis when used in a transformation.
const FLIP_X = Basis(-1, 0, 0, 0, 1, 0, 0, 0, 1);

## The basis that will flip something along the Y axis when used in a transformation.
const FLIP_Y = Basis(1, 0, 0, 0, -1, 0, 0, 0, 1);

## The basis that will flip something along the Z axis when used in a transformation.
const FLIP_Z = Basis(1, 0, 0, 0, 1, 0, 0, 0, -1);


## The basis matrix's X vector (column 0). Equivalent to array index [code]0[/code].
var x: Vector3

## The basis matrix's Y vector (column 1). Equivalent to array index [code]1[/code].
var y: Vector3

## The basis matrix's Z vector (column 2). Equivalent to array index [code]2[/code].
var z: Vector3


## Constructs a default-initialized [Basis] set to [constant IDENTITY].
func Basis() -> Basis:
	pass;

## Constructs a [Basis] as a copy of the given [Basis].
func Basis(from: Basis) -> Basis:
	pass;

## Constructs a pure rotation basis matrix, rotated around the given [param axis] by [param angle] (in radians). The axis must be a normalized vector.
func Basis(axis: Vector3, angle: float) -> Basis:
	pass;

## Constructs a pure rotation basis matrix from the given quaternion.
func Basis(from: Quaternion) -> Basis:
	pass;

## Constructs a basis matrix from 3 axis vectors (matrix columns).
func Basis(x_axis: Vector3, y_axis: Vector3, z_axis: Vector3) -> Basis:
	pass;


## Returns the determinant of the basis matrix. If the basis is uniformly scaled, its determinant is the square of the scale.
## A negative determinant means the basis has a negative scale. A zero determinant means the basis isn't invertible, and is usually considered invalid.
func determinant() -> float:
	pass;

## Constructs a pure rotation Basis matrix from Euler angles in the specified Euler rotation order. By default, use YXZ order (most common). See the [enum EulerOrder] enum for possible values.
static func from_euler(euler: Vector3, order: int = 2) -> Basis:
	pass;

## Constructs a pure scale basis matrix with no rotation or shearing. The scale values are set as the diagonal of the matrix, and the other parts of the matrix are zero.
static func from_scale(scale: Vector3) -> Basis:
	pass;

## Returns the basis's rotation in the form of Euler angles. The Euler order depends on the [param order] parameter, by default it uses the YXZ convention: when decomposing, first Z, then X, and Y last. The returned vector contains the rotation angles in the format (X angle, Y angle, Z angle).
## Consider using the [method get_rotation_quaternion] method instead, which returns a [Quaternion] quaternion instead of Euler angles.
func get_euler(order: int = 2) -> Vector3:
	pass;

## Returns the basis's rotation in the form of a quaternion. See [method get_euler] if you need Euler angles, but keep in mind quaternions should generally be preferred to Euler angles.
func get_rotation_quaternion() -> Quaternion:
	pass;

## Assuming that the matrix is the combination of a rotation and scaling, return the absolute value of scaling factors along each axis.
func get_scale() -> Vector3:
	pass;

## Returns the inverse of the matrix.
func inverse() -> Basis:
	pass;

## Returns [code]true[/code] if this basis and [param b] are approximately equal, by calling [method @GlobalScope.is_equal_approx] on all vector components.
func is_equal_approx(b: Basis) -> bool:
	pass;

## Returns [code]true[/code] if this basis is finite, by calling [method @GlobalScope.is_finite] on all vector components.
func is_finite() -> bool:
	pass;

## Creates a Basis with a rotation such that the forward axis (-Z) points towards the [param target] position.
## The up axis (+Y) points as close to the [param up] vector as possible while staying perpendicular to the forward axis. The resulting Basis is orthonormalized. The [param target] and [param up] vectors cannot be zero, and cannot be parallel to each other.
## If [param use_model_front] is [code]true[/code], the +Z axis (asset front) is treated as forward (implies +X is left) and points toward the [param target] position. By default, the -Z axis (camera forward) is treated as forward (implies +X is right).
static func looking_at(target: Vector3, up: Vector3 = Vector3(0, 1, 0), use_model_front: bool = false) -> Basis:
	pass;

## Returns the orthonormalized version of the matrix (useful to call from time to time to avoid rounding error for orthogonal matrices). This performs a Gram-Schmidt orthonormalization on the basis of the matrix.
func orthonormalized() -> Basis:
	pass;

## Introduce an additional rotation around the given axis by [param angle] (in radians). The axis must be a normalized vector.
func rotated(axis: Vector3, angle: float) -> Basis:
	pass;

## Introduce an additional scaling specified by the given 3D scaling factor.
func scaled(scale: Vector3) -> Basis:
	pass;

## Assuming that the matrix is a proper rotation matrix, slerp performs a spherical-linear interpolation with another rotation matrix.
func slerp(to: Basis, weight: float) -> Basis:
	pass;

## Transposed dot product with the X axis of the matrix.
func tdotx(with: Vector3) -> float:
	pass;

## Transposed dot product with the Y axis of the matrix.
func tdoty(with: Vector3) -> float:
	pass;

## Transposed dot product with the Z axis of the matrix.
func tdotz(with: Vector3) -> float:
	pass;

## Returns the transposed version of the matrix.
func transposed() -> Basis:
	pass;


