extends Node
class_name NavigationAgent3D

## A 3D agent used to pathfind to a position while avoiding obstacles.
##
## A 3D agent used to pathfind to a position while avoiding static and dynamic obstacles. The calculation can be used by the parent node to dynamically move it along the path. Requires navigation data to work correctly.
## Dynamic obstacles are avoided using RVO collision avoidance. Avoidance is computed before physics, so the pathfinding information can be used safely in the physics step.
## [b]Note:[/b] After setting the [member target_position] property, the [method get_next_path_position] method must be used once every physics frame to update the internal path logic of the navigation agent. The vector position it returns should be used as the next movement position for the agent's parent node.
##
## @tutorial(Using NavigationAgents): https://docs.godotengine.org/en/stable/tutorials/navigation/navigation_using_navigationagents.html
## @experimental


## Notifies when a navigation link has been reached.
## The details dictionary may contain the following keys depending on the value of [member path_metadata_flags]:
## - [code]position[/code]: The start position of the link that was reached.
## - [code]type[/code]: Always [constant NavigationPathQueryResult3D.PATH_SEGMENT_TYPE_LINK].
## - [code]rid[/code]: The [RID] of the link.
## - [code]owner[/code]: The object which manages the link (usually [NavigationLink3D]).
## - [code]link_entry_position[/code]: If [code]owner[/code] is available and the owner is a [NavigationLink3D], it will contain the global position of the link's point the agent is entering.
## - [code]link_exit_position[/code]: If [code]owner[/code] is available and the owner is a [NavigationLink3D], it will contain the global position of the link's point which the agent is exiting.
signal link_reached(details: Dictionary)
## Notifies when the final position is reached.
signal navigation_finished
## Notifies when the navigation path changes.
signal path_changed
## Notifies when the player-defined [member target_position] is reached.
signal target_reached
## Notifies when the collision avoidance velocity is calculated. Emitted when [member velocity] is set. Only emitted when [member avoidance_enabled] is true.
signal velocity_computed(safe_velocity: Vector3)
## Notifies when a waypoint along the path has been reached.
## The details dictionary may contain the following keys depending on the value of [member path_metadata_flags]:
## - [code]position[/code]: The position of the waypoint that was reached.
## - [code]type[/code]: The type of navigation primitive (region or link) that contains this waypoint.
## - [code]rid[/code]: The [RID] of the containing navigation primitive (region or link).
## - [code]owner[/code]: The object which manages the containing navigation primitive (region or link).
signal waypoint_reached(details: Dictionary)

## If [code]true[/code] the agent is registered for an RVO avoidance callback on the [NavigationServer3D]. When [member velocity] is set and the processing is completed a [code]safe_velocity[/code] Vector3 is received with a signal connection to [signal velocity_computed]. Avoidance processing with many registered agents has a significant performance cost and should only be enabled on agents that currently require it.
var avoidance_enabled: bool:
	get = get_avoidance_enabled, set = set_avoidance_enabled

## A bitfield determining the avoidance layers for this NavigationAgent. Other agent's with a matching bit on the [member avoidance_mask] will avoid this agent.
var avoidance_layers: int:
	get = get_avoidance_layers, set = set_avoidance_layers

## A bitfield determining what other avoidance agents and obstacles this NavigationAgent will avoid when a bit matches at least one of their [member avoidance_layers].
var avoidance_mask: int:
	get = get_avoidance_mask, set = set_avoidance_mask

## The agent does not adjust the velocity for other agents that would match the [member avoidance_mask] but have a lower [member avoidance_priority]. This in turn makes the other agents with lower priority adjust their velocities even more to avoid collision with this agent.
var avoidance_priority: float:
	get = get_avoidance_priority, set = set_avoidance_priority

## If [code]true[/code] shows debug visuals for this agent.
var debug_enabled: bool:
	get = get_debug_enabled, set = set_debug_enabled

## If [member debug_use_custom] is [code]true[/code] uses this color for this agent instead of global color.
var debug_path_custom_color: Color:
	get = get_debug_path_custom_color, set = set_debug_path_custom_color

## If [member debug_use_custom] is [code]true[/code] uses this rasterized point size for rendering path points for this agent instead of global point size.
var debug_path_custom_point_size: float:
	get = get_debug_path_custom_point_size, set = set_debug_path_custom_point_size

## If [code]true[/code] uses the defined [member debug_path_custom_color] for this agent instead of global color.
var debug_use_custom: bool:
	get = get_debug_use_custom, set = set_debug_use_custom

## The height of the avoidance agent. Agents will ignore other agents or obstacles that are above or below their current position + height in 2D avoidance. Does nothing in 3D avoidance which uses radius spheres alone.
var height: float:
	get = get_height, set = set_height

## The maximum number of neighbors for the agent to consider.
var max_neighbors: int:
	get = get_max_neighbors, set = set_max_neighbors

## The maximum speed that an agent can move.
var max_speed: float:
	get = get_max_speed, set = set_max_speed

## A bitfield determining which navigation layers of navigation regions this agent will use to calculate a path. Changing it during runtime will clear the current navigation path and generate a new one, according to the new navigation layers.
var navigation_layers: int:
	get = get_navigation_layers, set = set_navigation_layers

## The distance to search for other agents.
var neighbor_distance: float:
	get = get_neighbor_distance, set = set_neighbor_distance

## The distance threshold before a path point is considered to be reached. This allows agents to not have to hit a path point on the path exactly, but only to reach its general area. If this value is set too high, the NavigationAgent will skip points on the path, which can lead to leaving the navigation mesh. If this value is set too low, the NavigationAgent will be stuck in a repath loop because it will constantly overshoot or undershoot the distance to the next point on each physics frame update.
var path_desired_distance: float:
	get = get_path_desired_distance, set = set_path_desired_distance

## The height offset is subtracted from the y-axis value of any vector path position for this NavigationAgent. The NavigationAgent height offset does not change or influence the navigation mesh or pathfinding query result. Additional navigation maps that use regions with navigation meshes that the developer baked with appropriate agent radius or height values are required to support different-sized agents.
var path_height_offset: float:
	get = get_path_height_offset, set = set_path_height_offset

## The maximum distance the agent is allowed away from the ideal path to the final position. This can happen due to trying to avoid collisions. When the maximum distance is exceeded, it recalculates the ideal path.
var path_max_distance: float:
	get = get_path_max_distance, set = set_path_max_distance

## Additional information to return with the navigation path.
var path_metadata_flags: int:
	get = get_path_metadata_flags, set = set_path_metadata_flags

## The path postprocessing applied to the raw path corridor found by the [member pathfinding_algorithm].
var path_postprocessing: int:
	get = get_path_postprocessing, set = set_path_postprocessing

## The pathfinding algorithm used in the path query.
var pathfinding_algorithm: int:
	get = get_pathfinding_algorithm, set = set_pathfinding_algorithm

## The radius of the avoidance agent. This is the "body" of the avoidance agent and not the avoidance maneuver starting radius (which is controlled by [member neighbor_distance]).
## Does not affect normal pathfinding. To change an actor's pathfinding radius bake [NavigationMesh] resources with a different [member NavigationMesh.agent_radius] property and use different navigation maps for each actor size.
var radius: float:
	get = get_radius, set = set_radius

## The distance threshold before the final target point is considered to be reached. This allows agents to not have to hit the point of the final target exactly, but only to reach its general. If this value is set too low, the NavigationAgent will be stuck in a repath loop because it will constantly overshoot or undershoot the distance to the final target point on each physics frame update.
var target_desired_distance: float:
	get = get_target_desired_distance, set = set_target_desired_distance

## If set a new navigation path from the current agent position to the [member target_position] is requested from the NavigationServer.
var target_position: Vector3:
	get = get_target_position, set = set_target_position

## The minimal amount of time for which this agent's velocities, that are computed with the collision avoidance algorithm, are safe with respect to other agents. The larger the number, the sooner the agent will respond to other agents, but less freedom in choosing its velocities. A too high value will slow down agents movement considerably. Must be positive.
var time_horizon_agents: float:
	get = get_time_horizon_agents, set = set_time_horizon_agents

## The minimal amount of time for which this agent's velocities, that are computed with the collision avoidance algorithm, are safe with respect to static avoidance obstacles. The larger the number, the sooner the agent will respond to static avoidance obstacles, but less freedom in choosing its velocities. A too high value will slow down agents movement considerably. Must be positive.
var time_horizon_obstacles: float:
	get = get_time_horizon_obstacles, set = set_time_horizon_obstacles

## If [code]true[/code], the agent calculates avoidance velocities in 3D omnidirectionally, e.g. for games that take place in air, underwater or space. Agents using 3D avoidance only avoid other agents using 3D avoidance, and react to radius-based avoidance obstacles. They ignore any vertex-based obstacles.
## If [code]false[/code], the agent calculates avoidance velocities in 2D along the x and z-axes, ignoring the y-axis. Agents using 2D avoidance only avoid other agents using 2D avoidance, and react to radius-based avoidance obstacles or vertex-based avoidance obstacles. Other agents using 2D avoidance that are below or above their current position including [member height] are ignored.
var use_3d_avoidance: bool:
	get = get_use_3d_avoidance, set = set_use_3d_avoidance

## Sets the new wanted velocity for the agent. The avoidance simulation will try to fulfill this velocity if possible but will modify it to avoid collision with other agents and obstacles. When an agent is teleported to a new position, use [method set_velocity_forced] as well to reset the internal simulation velocity.
var velocity: Vector3:
	get = get_velocity, set = set_velocity



## Returns the distance to the target position, using the agent's global position. The user must set [member target_position] in order for this to be accurate.
func distance_to_target() -> float:
	pass;

## Returns whether or not the specified layer of the [member avoidance_layers] bitmask is enabled, given a [param layer_number] between 1 and 32.
func get_avoidance_layer_value(layer_number: int) -> bool:
	pass;

## Returns whether or not the specified mask of the [member avoidance_mask] bitmask is enabled, given a [param mask_number] between 1 and 32.
func get_avoidance_mask_value(mask_number: int) -> bool:
	pass;

## Returns this agent's current path from start to finish in global coordinates. The path only updates when the target position is changed or the agent requires a repath. The path array is not intended to be used in direct path movement as the agent has its own internal path logic that would get corrupted by changing the path array manually. Use the intended [method get_next_path_position] once every physics frame to receive the next path point for the agents movement as this function also updates the internal path logic.
func get_current_navigation_path() -> PackedVector3Array:
	pass;

## Returns which index the agent is currently on in the navigation path's [PackedVector3Array].
func get_current_navigation_path_index() -> int:
	pass;

## Returns the path query result for the path the agent is currently following.
func get_current_navigation_result() -> NavigationPathQueryResult3D:
	pass;

## Returns the reachable final position of the current navigation path in global coordinates. This position can change if the navigation path is altered in any way. Because of this, it would be best to check this each frame.
func get_final_position() -> Vector3:
	pass;

## Returns whether or not the specified layer of the [member navigation_layers] bitmask is enabled, given a [param layer_number] between 1 and 32.
func get_navigation_layer_value(layer_number: int) -> bool:
	pass;

## Returns the [RID] of the navigation map for this NavigationAgent node. This function returns always the map set on the NavigationAgent node and not the map of the abstract agent on the NavigationServer. If the agent map is changed directly with the NavigationServer API the NavigationAgent node will not be aware of the map change. Use [method set_navigation_map] to change the navigation map for the NavigationAgent and also update the agent on the NavigationServer.
func get_navigation_map() -> RID:
	pass;

## Returns the next position in global coordinates that can be moved to, making sure that there are no static objects in the way. If the agent does not have a navigation path, it will return the position of the agent's parent. The use of this function once every physics frame is required to update the internal path logic of the NavigationAgent.
func get_next_path_position() -> Vector3:
	pass;

## Returns the [RID] of this agent on the [NavigationServer3D].
func get_rid() -> RID:
	pass;

## Returns true if the navigation path's final position has been reached.
func is_navigation_finished() -> bool:
	pass;

## Returns true if [member target_position] is reachable. The target position is set using [member target_position].
func is_target_reachable() -> bool:
	pass;

## Returns true if [member target_position] is reached. It may not always be possible to reach the target position. It should always be possible to reach the final position though. See [method get_final_position].
func is_target_reached() -> bool:
	pass;

## Based on [param value], enables or disables the specified layer in the [member avoidance_layers] bitmask, given a [param layer_number] between 1 and 32.
func set_avoidance_layer_value(layer_number: int, value: bool) -> void:
	pass;

## Based on [param value], enables or disables the specified mask in the [member avoidance_mask] bitmask, given a [param mask_number] between 1 and 32.
func set_avoidance_mask_value(mask_number: int, value: bool) -> void:
	pass;

## Based on [param value], enables or disables the specified layer in the [member navigation_layers] bitmask, given a [param layer_number] between 1 and 32.
func set_navigation_layer_value(layer_number: int, value: bool) -> void:
	pass;

## Sets the [RID] of the navigation map this NavigationAgent node should use and also updates the [code]agent[/code] on the NavigationServer.
func set_navigation_map(navigation_map: RID) -> void:
	pass;

## Replaces the internal velocity in the collision avoidance simulation with [param velocity]. When an agent is teleported to a new position this function should be used in the same frame. If called frequently this function can get agents stuck.
func set_velocity_forced(velocity: Vector3) -> void:
	pass;


func get_avoidance_enabled() -> bool:
	return avoidance_enabled

func set_avoidance_enabled(value: bool) -> void:
	avoidance_enabled = value

func get_avoidance_layers() -> int:
	return avoidance_layers

func set_avoidance_layers(value: int) -> void:
	avoidance_layers = value

func get_avoidance_mask() -> int:
	return avoidance_mask

func set_avoidance_mask(value: int) -> void:
	avoidance_mask = value

func get_avoidance_priority() -> float:
	return avoidance_priority

func set_avoidance_priority(value: float) -> void:
	avoidance_priority = value

func get_debug_enabled() -> bool:
	return debug_enabled

func set_debug_enabled(value: bool) -> void:
	debug_enabled = value

func get_debug_path_custom_color() -> Color:
	return debug_path_custom_color

func set_debug_path_custom_color(value: Color) -> void:
	debug_path_custom_color = value

func get_debug_path_custom_point_size() -> float:
	return debug_path_custom_point_size

func set_debug_path_custom_point_size(value: float) -> void:
	debug_path_custom_point_size = value

func get_debug_use_custom() -> bool:
	return debug_use_custom

func set_debug_use_custom(value: bool) -> void:
	debug_use_custom = value

func get_height() -> float:
	return height

func set_height(value: float) -> void:
	height = value

func get_max_neighbors() -> int:
	return max_neighbors

func set_max_neighbors(value: int) -> void:
	max_neighbors = value

func get_max_speed() -> float:
	return max_speed

func set_max_speed(value: float) -> void:
	max_speed = value

func get_navigation_layers() -> int:
	return navigation_layers

func set_navigation_layers(value: int) -> void:
	navigation_layers = value

func get_neighbor_distance() -> float:
	return neighbor_distance

func set_neighbor_distance(value: float) -> void:
	neighbor_distance = value

func get_path_desired_distance() -> float:
	return path_desired_distance

func set_path_desired_distance(value: float) -> void:
	path_desired_distance = value

func get_path_height_offset() -> float:
	return path_height_offset

func set_path_height_offset(value: float) -> void:
	path_height_offset = value

func get_path_max_distance() -> float:
	return path_max_distance

func set_path_max_distance(value: float) -> void:
	path_max_distance = value

func get_path_metadata_flags() -> int:
	return path_metadata_flags

func set_path_metadata_flags(value: int) -> void:
	path_metadata_flags = value

func get_path_postprocessing() -> int:
	return path_postprocessing

func set_path_postprocessing(value: int) -> void:
	path_postprocessing = value

func get_pathfinding_algorithm() -> int:
	return pathfinding_algorithm

func set_pathfinding_algorithm(value: int) -> void:
	pathfinding_algorithm = value

func get_radius() -> float:
	return radius

func set_radius(value: float) -> void:
	radius = value

func get_target_desired_distance() -> float:
	return target_desired_distance

func set_target_desired_distance(value: float) -> void:
	target_desired_distance = value

func get_target_position() -> Vector3:
	return target_position

func set_target_position(value: Vector3) -> void:
	target_position = value

func get_time_horizon_agents() -> float:
	return time_horizon_agents

func set_time_horizon_agents(value: float) -> void:
	time_horizon_agents = value

func get_time_horizon_obstacles() -> float:
	return time_horizon_obstacles

func set_time_horizon_obstacles(value: float) -> void:
	time_horizon_obstacles = value

func get_use_3d_avoidance() -> bool:
	return use_3d_avoidance

func set_use_3d_avoidance(value: bool) -> void:
	use_3d_avoidance = value

func get_velocity() -> Vector3:
	return velocity

func set_velocity(value: Vector3) -> void:
	velocity = value

