class_name NodePath

## Pre-parsed scene tree path.
##
## A pre-parsed relative or absolute path in a scene tree, for use with [method Node.get_node] and similar functions. It can reference a node, a resource within a node, or a property of a node or resource. For example, [code]"Path2D/PathFollow2D/Sprite2D:texture:size"[/code] would refer to the [code]size[/code] property of the [code]texture[/code] resource on the node named [code]"Sprite2D"[/code] which is a child of the other named nodes in the path.
## You will usually just pass a string to [method Node.get_node] and it will be automatically converted, but you may occasionally want to parse a path ahead of time with [NodePath] or the literal syntax [code]^"path"[/code]. Exporting a [NodePath] variable will give you a node selection widget in the properties panel of the editor, which can often be useful.
## A [NodePath] is composed of a list of slash-separated node names (like a filesystem path) and an optional colon-separated list of "subnames" which can be resources or properties.
## Some examples of NodePaths include the following:
## [codeblock]
## # No leading slash means it is relative to the current node.
## ^"A" # Immediate child A
## ^"A/B" # A's child B
## ^"." # The current node.
## ^".." # The parent node.
## ^"../C" # A sibling node C.
## # A leading slash means it is absolute from the SceneTree.
## ^"/root" # Equivalent to get_tree().get_root().
## ^"/root/Main" # If your main scene's root node were named "Main".
## ^"/root/MyAutoload" # If you have an autoloaded node or scene.
## [/codeblock]
## See also [StringName], which is a similar concept for general-purpose string interning.
## [b]Note:[/b] In the editor, [NodePath] properties are automatically updated when moving, renaming or deleting a node in the scene tree, but they are never updated at runtime.
##
## @tutorial(2D Role Playing Game Demo): https://godotengine.org/asset-library/asset/520




## Constructs an empty [NodePath].
func NodePath() -> NodePath:
	pass;

## Constructs a [NodePath] as a copy of the given [NodePath]. [code]NodePath("example")[/code] is equivalent to [code]^"example"[/code].
func NodePath(from: NodePath) -> NodePath:
	pass;

## Creates a NodePath from a string, e.g. [code]"Path2D/PathFollow2D/Sprite2D:texture:size"[/code]. A path is absolute if it starts with a slash. Absolute paths are only valid in the global scene tree, not within individual scenes. In a relative path, [code]"."[/code] and [code]".."[/code] indicate the current node and its parent.
## The "subnames" optionally included after the path to the target node can point to resources or properties, and can also be nested.
## Examples of valid NodePaths (assuming that those nodes exist and have the referenced resources or properties):
## [codeblock]
## # Points to the Sprite2D node.
## "Path2D/PathFollow2D/Sprite2D"
## # Points to the Sprite2D node and its "texture" resource.
## # get_node() would retrieve "Sprite2D", while get_node_and_resource()
## # would retrieve both the Sprite2D node and the "texture" resource.
## "Path2D/PathFollow2D/Sprite2D:texture"
## # Points to the Sprite2D node and its "position" property.
## "Path2D/PathFollow2D/Sprite2D:position"
## # Points to the Sprite2D node and the "x" component of its "position" property.
## "Path2D/PathFollow2D/Sprite2D:position:x"
## # Absolute path (from "root")
## "/root/Level/Path2D"
## [/codeblock]
func NodePath(from: String) -> NodePath:
	pass;


## Returns a node path with a colon character ([code]:[/code]) prepended, transforming it to a pure property path with no node name (defaults to resolving from the current node).
## [codeblocks]
## [gdscript]
## # This will be parsed as a node path to the "x" property in the "position" node.
## var node_path = NodePath("position:x")
## # This will be parsed as a node path to the "x" component of the "position" property in the current node.
## var property_path = node_path.get_as_property_path()
## print(property_path) # :position:x
## [/gdscript]
## [csharp]
## // This will be parsed as a node path to the "x" property in the "position" node.
## var nodePath = new NodePath("position:x");
## // This will be parsed as a node path to the "x" component of the "position" property in the current node.
## NodePath propertyPath = nodePath.GetAsPropertyPath();
## GD.Print(propertyPath); // :position:x
## [/csharp]
## [/codeblocks]
func get_as_property_path() -> NodePath:
	pass;

## Returns all paths concatenated with a slash character ([code]/[/code]) as separator without subnames.
func get_concatenated_names() -> StringName:
	pass;

## Returns all subnames concatenated with a colon character ([code]:[/code]) as separator, i.e. the right side of the first colon in a node path.
## [codeblocks]
## [gdscript]
## var nodepath = NodePath("Path2D/PathFollow2D/Sprite2D:texture:load_path")
## print(nodepath.get_concatenated_subnames()) # texture:load_path
## [/gdscript]
## [csharp]
## var nodepath = new NodePath("Path2D/PathFollow2D/Sprite2D:texture:load_path");
## GD.Print(nodepath.GetConcatenatedSubnames()); // texture:load_path
## [/csharp]
## [/codeblocks]
func get_concatenated_subnames() -> StringName:
	pass;

## Gets the node name indicated by [param idx] (0 to [method get_name_count] - 1).
## [codeblocks]
## [gdscript]
## var node_path = NodePath("Path2D/PathFollow2D/Sprite2D")
## print(node_path.get_name(0)) # Path2D
## print(node_path.get_name(1)) # PathFollow2D
## print(node_path.get_name(2)) # Sprite
## [/gdscript]
## [csharp]
## var nodePath = new NodePath("Path2D/PathFollow2D/Sprite2D");
## GD.Print(nodePath.GetName(0)); // Path2D
## GD.Print(nodePath.GetName(1)); // PathFollow2D
## GD.Print(nodePath.GetName(2)); // Sprite
## [/csharp]
## [/codeblocks]
func get_name(idx: int) -> StringName:
	pass;

## Gets the number of node names which make up the path. Subnames (see [method get_subname_count]) are not included.
## For example, [code]"Path2D/PathFollow2D/Sprite2D"[/code] has 3 names.
func get_name_count() -> int:
	pass;

## Gets the resource or property name indicated by [param idx] (0 to [method get_subname_count]).
## [codeblocks]
## [gdscript]
## var node_path = NodePath("Path2D/PathFollow2D/Sprite2D:texture:load_path")
## print(node_path.get_subname(0)) # texture
## print(node_path.get_subname(1)) # load_path
## [/gdscript]
## [csharp]
## var nodePath = new NodePath("Path2D/PathFollow2D/Sprite2D:texture:load_path");
## GD.Print(nodePath.GetSubname(0)); // texture
## GD.Print(nodePath.GetSubname(1)); // load_path
## [/csharp]
## [/codeblocks]
func get_subname(idx: int) -> StringName:
	pass;

## Gets the number of resource or property names ("subnames") in the path. Each subname is listed after a colon character ([code]:[/code]) in the node path.
## For example, [code]"Path2D/PathFollow2D/Sprite2D:texture:load_path"[/code] has 2 subnames.
func get_subname_count() -> int:
	pass;

## Returns the 32-bit hash value representing the [NodePath]'s contents.
func hash() -> int:
	pass;

## Returns [code]true[/code] if the node path is absolute (as opposed to relative), which means that it starts with a slash character ([code]/[/code]). Absolute node paths can be used to access the root node ([code]"/root"[/code]) or autoloads (e.g. [code]"/global"[/code] if a "global" autoload was registered).
func is_absolute() -> bool:
	pass;

## Returns [code]true[/code] if the node path is empty.
func is_empty() -> bool:
	pass;


