extends RefCounted
class_name FileAccess

## Type to handle file reading and writing operations.
##
## File type. This is used to permanently store data into the user device's file system and to read from it. This can be used to store game save data or player configuration files, for example.
## Here's a sample on how to write and read from a file:
## [codeblocks]
## [gdscript]
## func save(content):
## var file = FileAccess.open("user://save_game.dat", FileAccess.WRITE)
## file.store_string(content)
## func load():
## var file = FileAccess.open("user://save_game.dat", FileAccess.READ)
## var content = file.get_as_text()
## return content
## [/gdscript]
## [csharp]
## public void Save(string content)
## {
## using var file = FileAccess.Open("user://save_game.dat", FileAccess.ModeFlags.Write);
## file.StoreString(content);
## }
## public string Load()
## {
## using var file = FileAccess.Open("user://save_game.dat", FileAccess.ModeFlags.Read);
## string content = file.GetAsText();
## return content;
## }
## [/csharp]
## [/codeblocks]
## In the example above, the file will be saved in the user data folder as specified in the [url=$DOCS_URL/tutorials/io/data_paths.html]Data paths[/url] documentation.
## [FileAccess] will close when it's freed, which happens when it goes out of scope or when it gets assigned with [code]null[/code]. In C# the reference must be disposed after we are done using it, this can be done with the [code]using[/code] statement or calling the [code]Dispose[/code] method directly.
## [codeblocks]
## [gdscript]
## var file = FileAccess.open("res://something") # File is opened and locked for use.
## file = null # File is closed.
## [/gdscript]
## [csharp]
## using var file = FileAccess.Open("res://something"); // File is opened and locked for use.
## // The using statement calls Dispose when going out of scope.
## [/csharp]
## [/codeblocks]
## [b]Note:[/b] To access project resources once exported, it is recommended to use [ResourceLoader] instead of the [FileAccess] API, as some files are converted to engine-specific formats and their original source files might not be present in the exported PCK package.
## [b]Note:[/b] Files are automatically closed only if the process exits "normally" (such as by clicking the window manager's close button or pressing [b]Alt + F4[/b]). If you stop the project execution by pressing [b]F8[/b] while the project is running, the file won't be closed as the game process will be killed. You can work around this by calling [method flush] at regular intervals.
##
## @tutorial(File system): https://docs.godotengine.org/en/stable/tutorials/scripting/filesystem.html
## @tutorial(3D Voxel Demo): https://godotengine.org/asset-library/asset/676


## Opens the file for read operations. The cursor is positioned at the beginning of the file.
## Opens the file for write operations. The file is created if it does not exist, and truncated if it does.
## Opens the file for read and write operations. Does not truncate the file. The cursor is positioned at the beginning of the file.
## Opens the file for read and write operations. The file is created if it does not exist, and truncated if it does. The cursor is positioned at the beginning of the file.
## Uses the [url=https://fastlz.org/]FastLZ[/url] compression method.
## Uses the [url=https://en.wikipedia.org/wiki/DEFLATE]DEFLATE[/url] compression method.
## Uses the [url=https://facebook.github.io/zstd/]Zstandard[/url] compression method.
## Uses the [url=https://www.gzip.org/]gzip[/url] compression method.

#enum ModeFlags
enum {
    READ = 1,
    WRITE = 2,
    READ_WRITE = 3,
    WRITE_READ = 7,
}
#enum CompressionMode
enum {
    COMPRESSION_FASTLZ = 0,
    COMPRESSION_DEFLATE = 1,
    COMPRESSION_ZSTD = 2,
    COMPRESSION_GZIP = 3,
}
## If [code]true[/code], the file is read with big-endian [url=https://en.wikipedia.org/wiki/Endianness]endianness[/url]. If [code]false[/code], the file is read with little-endian endianness. If in doubt, leave this to [code]false[/code] as most files are written with little-endian endianness.
## [b]Note:[/b] [member big_endian] is only about the file format, not the CPU type. The CPU endianness doesn't affect the default endianness for files written.
## [b]Note:[/b] This is always reset to [code]false[/code] whenever you open the file. Therefore, you must set [member big_endian] [i]after[/i] opening the file, not before.
var big_endian: bool:
	get = is_big_endian, set = set_big_endian



## Closes the currently opened file and prevents subsequent read/write operations. Use [method flush] to persist the data to disk without closing the file.
## [b]Note:[/b] [FileAccess] will automatically close when it's freed, which happens when it goes out of scope or when it gets assigned with [code]null[/code]. In C# the reference must be disposed after we are done using it, this can be done with the [code]using[/code] statement or calling the [code]Dispose[/code] method directly.
func close() -> void:
	pass;

## Returns [code]true[/code] if the file cursor has already read past the end of the file.
## [b]Note:[/b] [code]eof_reached() == false[/code] cannot be used to check whether there is more data available. To loop while there is more data available, use:
## [codeblocks]
## [gdscript]
## while file.get_position() < file.get_length():
## # Read data
## [/gdscript]
## [csharp]
## while (file.GetPosition() < file.GetLength())
## {
## // Read data
## }
## [/csharp]
## [/codeblocks]
func eof_reached() -> bool:
	pass;

## Returns [code]true[/code] if the file exists in the given path.
## [b]Note:[/b] Many resources types are imported (e.g. textures or sound files), and their source asset will not be included in the exported game, as only the imported version is used. See [method ResourceLoader.exists] for an alternative approach that takes resource remapping into account.
## For a non-static, relative equivalent, use [method DirAccess.file_exists].
static func file_exists(path: String) -> bool:
	pass;

## Writes the file's buffer to disk. Flushing is automatically performed when the file is closed. This means you don't need to call [method flush] manually before closing a file. Still, calling [method flush] can be used to ensure the data is safe even if the project crashes instead of being closed gracefully.
## [b]Note:[/b] Only call [method flush] when you actually need it. Otherwise, it will decrease performance due to constant disk writes.
func flush() -> void:
	pass;

## Returns the next 16 bits from the file as an integer. See [method store_16] for details on what values can be stored and retrieved this way.
func get_16() -> int:
	pass;

## Returns the next 32 bits from the file as an integer. See [method store_32] for details on what values can be stored and retrieved this way.
func get_32() -> int:
	pass;

## Returns the next 64 bits from the file as an integer. See [method store_64] for details on what values can be stored and retrieved this way.
func get_64() -> int:
	pass;

## Returns the next 8 bits from the file as an integer. See [method store_8] for details on what values can be stored and retrieved this way.
func get_8() -> int:
	pass;

## Returns the whole file as a [String]. Text is interpreted as being UTF-8 encoded.
## If [param skip_cr] is [code]true[/code], carriage return characters ([code]\r[/code], CR) will be ignored when parsing the UTF-8, so that only line feed characters ([code]\n[/code], LF) represent a new line (Unix convention).
func get_as_text(skip_cr: bool = false) -> String:
	pass;

## Returns next [param length] bytes of the file as a [PackedByteArray].
func get_buffer(length: int) -> PackedByteArray:
	pass;

## Returns the next value of the file in CSV (Comma-Separated Values) format. You can pass a different delimiter [param delim] to use other than the default [code]","[/code] (comma). This delimiter must be one-character long, and cannot be a double quotation mark.
## Text is interpreted as being UTF-8 encoded. Text values must be enclosed in double quotes if they include the delimiter character. Double quotes within a text value can be escaped by doubling their occurrence.
## For example, the following CSV lines are valid and will be properly parsed as two strings each:
## [codeblock]
## Alice,"Hello, Bob!"
## Bob,Alice! What a surprise!
## Alice,"I thought you'd reply with ""Hello, world""."
## [/codeblock]
## Note how the second line can omit the enclosing quotes as it does not include the delimiter. However it [i]could[/i] very well use quotes, it was only written without for demonstration purposes. The third line must use [code]""[/code] for each quotation mark that needs to be interpreted as such instead of the end of a text value.
func get_csv_line(delim: String = ",") -> PackedStringArray:
	pass;

## Returns the next 64 bits from the file as a floating-point number.
func get_double() -> float:
	pass;

## Returns the last error that happened when trying to perform operations. Compare with the [code]ERR_FILE_*[/code] constants from [enum Error].
func get_error() -> int:
	pass;

## Returns the whole [param path] file contents as a [PackedByteArray] without any decoding.
static func get_file_as_bytes(path: String) -> PackedByteArray:
	pass;

## Returns the whole [param path] file contents as a [String]. Text is interpreted as being UTF-8 encoded.
static func get_file_as_string(path: String) -> String:
	pass;

## Returns the next 32 bits from the file as a floating-point number.
func get_float() -> float:
	pass;

## Returns the size of the file in bytes.
func get_length() -> int:
	pass;

## Returns the next line of the file as a [String].
## Text is interpreted as being UTF-8 encoded.
func get_line() -> String:
	pass;

## Returns an MD5 String representing the file at the given path or an empty [String] on failure.
static func get_md5(path: String) -> String:
	pass;

## Returns the last time the [param file] was modified in Unix timestamp format or returns a [String] "ERROR IN [code]file[/code]". This Unix timestamp can be converted to another format using the [Time] singleton.
static func get_modified_time(file: String) -> int:
	pass;

## Returns the result of the last [method open] call in the current thread.
static func get_open_error() -> int:
	pass;

## Returns a [String] saved in Pascal format from the file.
## Text is interpreted as being UTF-8 encoded.
func get_pascal_string() -> String:
	pass;

## Returns the path as a [String] for the current open file.
func get_path() -> String:
	pass;

## Returns the absolute path as a [String] for the current open file.
func get_path_absolute() -> String:
	pass;

## Returns the file cursor's position.
func get_position() -> int:
	pass;

## Returns the next bits from the file as a floating-point number.
func get_real() -> float:
	pass;

## Returns a SHA-256 [String] representing the file at the given path or an empty [String] on failure.
static func get_sha256(path: String) -> String:
	pass;

## Returns the next [Variant] value from the file. If [param allow_objects] is [code]true[/code], decoding objects is allowed.
## Internally, this uses the same decoding mechanism as the [method @GlobalScope.bytes_to_var] method.
## [b]Warning:[/b] Deserialized objects can contain code which gets executed. Do not use this option if the serialized object comes from untrusted sources to avoid potential security threats such as remote code execution.
func get_var(allow_objects: bool = false) -> Variant:
	pass;

## Returns [code]true[/code] if the file is currently opened.
func is_open() -> bool:
	pass;

## Creates a new [FileAccess] object and opens the file for writing or reading, depending on the flags.
## Returns [code]null[/code] if opening the file failed. You can use [method get_open_error] to check the error that occurred.
static func open(path: String, flags: int) -> FileAccess:
	pass;

## Creates a new [FileAccess] object and opens a compressed file for reading or writing.
## [b]Note:[/b] [method open_compressed] can only read files that were saved by Godot, not third-party compression formats. See [url=https://github.com/godotengine/godot/issues/28999]GitHub issue #28999[/url] for a workaround.
## Returns [code]null[/code] if opening the file failed. You can use [method get_open_error] to check the error that occurred.
static func open_compressed(path: String, mode_flags: int, compression_mode: int = 0) -> FileAccess:
	pass;

## Creates a new [FileAccess] object and opens an encrypted file in write or read mode. You need to pass a binary key to encrypt/decrypt it.
## [b]Note:[/b] The provided key must be 32 bytes long.
## Returns [code]null[/code] if opening the file failed. You can use [method get_open_error] to check the error that occurred.
static func open_encrypted(path: String, mode_flags: int, key: PackedByteArray) -> FileAccess:
	pass;

## Creates a new [FileAccess] object and opens an encrypted file in write or read mode. You need to pass a password to encrypt/decrypt it.
## Returns [code]null[/code] if opening the file failed. You can use [method get_open_error] to check the error that occurred.
static func open_encrypted_with_pass(path: String, mode_flags: int, pass: String) -> FileAccess:
	pass;

## Changes the file reading/writing cursor to the specified position (in bytes from the beginning of the file).
func seek(position: int) -> void:
	pass;

## Changes the file reading/writing cursor to the specified position (in bytes from the end of the file).
## [b]Note:[/b] This is an offset, so you should use negative numbers or the cursor will be at the end of the file.
func seek_end(position: int = 0) -> void:
	pass;

## Stores an integer as 16 bits in the file.
## [b]Note:[/b] The [param value] should lie in the interval [code][0, 2^16 - 1][/code]. Any other value will overflow and wrap around.
## To store a signed integer, use [method store_64] or store a signed integer from the interval [code][-2^15, 2^15 - 1][/code] (i.e. keeping one bit for the signedness) and compute its sign manually when reading. For example:
## [codeblocks]
## [gdscript]
## const MAX_15B = 1 << 15
## const MAX_16B = 1 << 16
## func unsigned16_to_signed(unsigned):
## return (unsigned + MAX_15B) % MAX_16B - MAX_15B
## func _ready():
## var f = FileAccess.open("user://file.dat", FileAccess.WRITE_READ)
## f.store_16(-42) # This wraps around and stores 65494 (2^16 - 42).
## f.store_16(121) # In bounds, will store 121.
## f.seek(0) # Go back to start to read the stored value.
## var read1 = f.get_16() # 65494
## var read2 = f.get_16() # 121
## var converted1 = unsigned16_to_signed(read1) # -42
## var converted2 = unsigned16_to_signed(read2) # 121
## [/gdscript]
## [csharp]
## public override void _Ready()
## {
## using var f = FileAccess.Open("user://file.dat", FileAccess.ModeFlags.WriteRead);
## f.Store16(unchecked((ushort)-42)); // This wraps around and stores 65494 (2^16 - 42).
## f.Store16(121); // In bounds, will store 121.
## f.Seek(0); // Go back to start to read the stored value.
## ushort read1 = f.Get16(); // 65494
## ushort read2 = f.Get16(); // 121
## short converted1 = (short)read1; // -42
## short converted2 = (short)read2; // 121
## }
## [/csharp]
## [/codeblocks]
func store_16(value: int) -> void:
	pass;

## Stores an integer as 32 bits in the file.
## [b]Note:[/b] The [param value] should lie in the interval [code][0, 2^32 - 1][/code]. Any other value will overflow and wrap around.
## To store a signed integer, use [method store_64], or convert it manually (see [method store_16] for an example).
func store_32(value: int) -> void:
	pass;

## Stores an integer as 64 bits in the file.
## [b]Note:[/b] The [param value] must lie in the interval [code][-2^63, 2^63 - 1][/code] (i.e. be a valid [int] value).
func store_64(value: int) -> void:
	pass;

## Stores an integer as 8 bits in the file.
## [b]Note:[/b] The [param value] should lie in the interval [code][0, 255][/code]. Any other value will overflow and wrap around.
## To store a signed integer, use [method store_64], or convert it manually (see [method store_16] for an example).
func store_8(value: int) -> void:
	pass;

## Stores the given array of bytes in the file.
func store_buffer(buffer: PackedByteArray) -> void:
	pass;

## Store the given [PackedStringArray] in the file as a line formatted in the CSV (Comma-Separated Values) format. You can pass a different delimiter [param delim] to use other than the default [code]","[/code] (comma). This delimiter must be one-character long.
## Text will be encoded as UTF-8.
func store_csv_line(values: PackedStringArray, delim: String = ",") -> void:
	pass;

## Stores a floating-point number as 64 bits in the file.
func store_double(value: float) -> void:
	pass;

## Stores a floating-point number as 32 bits in the file.
func store_float(value: float) -> void:
	pass;

## Appends [param line] to the file followed by a line return character ([code]\n[/code]), encoding the text as UTF-8.
func store_line(line: String) -> void:
	pass;

## Stores the given [String] as a line in the file in Pascal format (i.e. also store the length of the string).
## Text will be encoded as UTF-8.
func store_pascal_string(string: String) -> void:
	pass;

## Stores a floating-point number in the file.
func store_real(value: float) -> void:
	pass;

## Appends [param string] to the file without a line return, encoding the text as UTF-8.
## [b]Note:[/b] This method is intended to be used to write text files. The string is stored as a UTF-8 encoded buffer without string length or terminating zero, which means that it can't be loaded back easily. If you want to store a retrievable string in a binary file, consider using [method store_pascal_string] instead. For retrieving strings from a text file, you can use [code]get_buffer(length).get_string_from_utf8()[/code] (if you know the length) or [method get_as_text].
func store_string(string: String) -> void:
	pass;

## Stores any Variant value in the file. If [param full_objects] is [code]true[/code], encoding objects is allowed (and can potentially include code).
## Internally, this uses the same encoding mechanism as the [method @GlobalScope.var_to_bytes] method.
## [b]Note:[/b] Not all properties are included. Only properties that are configured with the [constant PROPERTY_USAGE_STORAGE] flag set will be serialized. You can add a new usage flag to a property by overriding the [method Object._get_property_list] method in your class. You can also check how property usage is configured by calling [method Object._get_property_list]. See [enum PropertyUsageFlags] for the possible usage flags.
func store_var(value: Variant, full_objects: bool = false) -> void:
	pass;


func is_big_endian() -> bool:
	return big_endian

func set_big_endian(value: bool) -> void:
	big_endian = value

