class_name Dictionary

## Dictionary type.
##
## Dictionary type. Associative container, which contains values referenced by unique keys. Dictionaries are composed of pairs of keys (which must be unique) and values. Dictionaries will preserve the insertion order when adding new entries. In other programming languages, this data structure is sometimes referred to as a hash map or associative array.
## You can define a dictionary by placing a comma-separated list of [code]key: value[/code] pairs in curly braces [code]{}[/code].
## [b]Note:[/b] Dictionaries are always passed by reference. To get a copy of a dictionary which can be modified independently of the original dictionary, use [method duplicate].
## Creating a dictionary:
## [codeblocks]
## [gdscript]
## var my_dict = {} # Creates an empty dictionary.
## var dict_variable_key = "Another key name"
## var dict_variable_value = "value2"
## var another_dict = {
## "Some key name": "value1",
## dict_variable_key: dict_variable_value,
## }
## var points_dict = {"White": 50, "Yellow": 75, "Orange": 100}
## # Alternative Lua-style syntax.
## # Doesn't require quotes around keys, but only string constants can be used as key names.
## # Additionally, key names must start with a letter or an underscore.
## # Here, `some_key` is a string literal, not a variable!
## another_dict = {
## some_key = 42,
## }
## [/gdscript]
## [csharp]
## var myDict = new Godot.Collections.Dictionary(); // Creates an empty dictionary.
## var pointsDict = new Godot.Collections.Dictionary
## {
## {"White", 50},
## {"Yellow", 75},
## {"Orange", 100}
## };
## [/csharp]
## [/codeblocks]
## You can access a dictionary's value by referencing its corresponding key. In the above example, [code]points_dict["White"][/code] will return [code]50[/code]. You can also write [code]points_dict.White[/code], which is equivalent. However, you'll have to use the bracket syntax if the key you're accessing the dictionary with isn't a fixed string (such as a number or variable).
## [codeblocks]
## [gdscript]
## @export_enum("White", "Yellow", "Orange") var my_color: String
## var points_dict = {"White": 50, "Yellow": 75, "Orange": 100}
## func _ready():
## # We can't use dot syntax here as `my_color` is a variable.
## var points = points_dict[my_color]
## [/gdscript]
## [csharp]
## [Export(PropertyHint.Enum, "White,Yellow,Orange")]
## public string MyColor { get; set; }
## private Godot.Collections.Dictionary _pointsDict = new Godot.Collections.Dictionary
## {
## {"White", 50},
## {"Yellow", 75},
## {"Orange", 100}
## };
## public override void _Ready()
## {
## int points = (int)_pointsDict[MyColor];
## }
## [/csharp]
## [/codeblocks]
## In the above code, [code]points[/code] will be assigned the value that is paired with the appropriate color selected in [code]my_color[/code].
## Dictionaries can contain more complex data:
## [codeblocks]
## [gdscript]
## var my_dict = {
## "First Array": [1, 2, 3, 4] # Assigns an Array to a String key.
## }
## [/gdscript]
## [csharp]
## var myDict = new Godot.Collections.Dictionary
## {
## {"First Array", new Godot.Collections.Array{1, 2, 3, 4}}
## };
## [/csharp]
## [/codeblocks]
## To add a key to an existing dictionary, access it like an existing key and assign to it:
## [codeblocks]
## [gdscript]
## var points_dict = {"White": 50, "Yellow": 75, "Orange": 100}
## points_dict["Blue"] = 150 # Add "Blue" as a key and assign 150 as its value.
## [/gdscript]
## [csharp]
## var pointsDict = new Godot.Collections.Dictionary
## {
## {"White", 50},
## {"Yellow", 75},
## {"Orange", 100}
## };
## pointsDict["Blue"] = 150; // Add "Blue" as a key and assign 150 as its value.
## [/csharp]
## [/codeblocks]
## Finally, dictionaries can contain different types of keys and values in the same dictionary:
## [codeblocks]
## [gdscript]
## # This is a valid dictionary.
## # To access the string "Nested value" below, use `my_dict.sub_dict.sub_key` or `my_dict["sub_dict"]["sub_key"]`.
## # Indexing styles can be mixed and matched depending on your needs.
## var my_dict = {
## "String Key": 5,
## 4: [1, 2, 3],
## 7: "Hello",
## "sub_dict": {"sub_key": "Nested value"},
## }
## [/gdscript]
## [csharp]
## // This is a valid dictionary.
## // To access the string "Nested value" below, use `((Godot.Collections.Dictionary)myDict["sub_dict"])["sub_key"]`.
## var myDict = new Godot.Collections.Dictionary {
## {"String Key", 5},
## {4, new Godot.Collections.Array{1,2,3}},
## {7, "Hello"},
## {"sub_dict", new Godot.Collections.Dictionary{{"sub_key", "Nested value"}}}
## };
## [/csharp]
## [/codeblocks]
## The keys of a dictionary can be iterated with the [code]for[/code] keyword:
## [codeblocks]
## [gdscript]
## var groceries = {"Orange": 20, "Apple": 2, "Banana": 4}
## for fruit in groceries:
## var amount = groceries[fruit]
## [/gdscript]
## [csharp]
## var groceries = new Godot.Collections.Dictionary{{"Orange", 20}, {"Apple", 2}, {"Banana", 4}};
## foreach (var (fruit, amount) in groceries)
## {
## // `fruit` is the key, `amount` is the value.
## }
## [/csharp]
## [/codeblocks]
## [b]Note:[/b] Erasing elements while iterating over dictionaries is [b]not[/b] supported and will result in unpredictable behavior.
## [b]Note:[/b] When declaring a dictionary with [code]const[/code], the dictionary becomes read-only. A read-only Dictionary's entries cannot be overridden at run-time. This does [i]not[/i] affect nested [Array] and [Dictionary] values.
##
## @tutorial(GDScript basics: Dictionary): https://docs.godotengine.org/en/stable/tutorials/scripting/gdscript/gdscript_basics.html#dictionary
## @tutorial(3D Voxel Demo): https://godotengine.org/asset-library/asset/676
## @tutorial(OS Test Demo): https://godotengine.org/asset-library/asset/677




## Constructs an empty [Dictionary].
func Dictionary() -> Dictionary:
	pass;

## Returns the same dictionary as [param from]. If you need a copy of the dictionary, use [method duplicate].
func Dictionary(from: Dictionary) -> Dictionary:
	pass;


## Clears the dictionary, removing all entries from it.
func clear() -> void:
	pass;

## Creates and returns a new copy of the dictionary. If [param deep] is [code]true[/code], inner [Dictionary] and [Array] keys and values are also copied, recursively.
func duplicate(deep: bool = false) -> Dictionary:
	pass;

## Removes the dictionary entry by key, if it exists. Returns [code]true[/code] if the given [param key] existed in the dictionary, otherwise [code]false[/code].
## [b]Note:[/b] Do not erase entries while iterating over the dictionary. You can iterate over the [method keys] array instead.
func erase(key: Variant) -> bool:
	pass;

## Finds and returns the first key whose associated value is equal to [param value], or [code]null[/code] if it is not found.
## [b]Note:[/b] [code]null[/code] is also a valid key. If inside the dictionary, [method find_key] may give misleading results.
func find_key(value: Variant) -> Variant:
	pass;

## Returns the corresponding value for the given [param key] in the dictionary. If the [param key] does not exist, returns [param default], or [code]null[/code] if the parameter is omitted.
func get(key: Variant, default: Variant = null) -> Variant:
	pass;

## Returns [code]true[/code] if the dictionary contains an entry with the given [param key].
## [codeblocks]
## [gdscript]
## var my_dict = {
## "Godot" : 4,
## 210 : null,
## }
## print(my_dict.has("Godot")) # Prints true
## print(my_dict.has(210))     # Prints true
## print(my_dict.has(4))       # Prints false
## [/gdscript]
## [csharp]
## var myDict = new Godot.Collections.Dictionary
## {
## { "Godot", 4 },
## { 210, default },
## };
## GD.Print(myDict.Contains("Godot")); // Prints true
## GD.Print(myDict.Contains(210));     // Prints true
## GD.Print(myDict.Contains(4));       // Prints false
## [/csharp]
## [/codeblocks]
## In GDScript, this is equivalent to the [code]in[/code] operator:
## [codeblock]
## if "Godot" in {"Godot": 4}:
## print("The key is here!") # Will be printed.
## [/codeblock]
## [b]Note:[/b] This method returns [code]true[/code] as long as the [param key] exists, even if its corresponding value is [code]null[/code].
func has(key: Variant) -> bool:
	pass;

## Returns [code]true[/code] if the dictionary contains all keys in the given [param keys] array.
## [codeblock]
## var data = {"width" : 10, "height" : 20}
## data.has_all(["height", "width"]) # Returns true
## [/codeblock]
func has_all(keys: Array) -> bool:
	pass;

## Returns a hashed 32-bit integer value representing the dictionary contents.
## [codeblocks]
## [gdscript]
## var dict1 = {"A": 10, "B": 2}
## var dict2 = {"A": 10, "B": 2}
## print(dict1.hash() == dict2.hash()) # Prints true
## [/gdscript]
## [csharp]
## var dict1 = new Godot.Collections.Dictionary{{"A", 10}, {"B", 2}};
## var dict2 = new Godot.Collections.Dictionary{{"A", 10}, {"B", 2}};
## // Godot.Collections.Dictionary has no Hash() method. Use GD.Hash() instead.
## GD.Print(GD.Hash(dict1) == GD.Hash(dict2)); // Prints true
## [/csharp]
## [/codeblocks]
## [b]Note:[/b] Dictionaries with the same entries but in a different order will not have the same hash.
## [b]Note:[/b] Dictionaries with equal hash values are [i]not[/i] guaranteed to be the same, because of hash collisions. On the countrary, dictionaries with different hash values are guaranteed to be different.
func hash() -> int:
	pass;

## Returns [code]true[/code] if the dictionary is empty (its size is [code]0[/code]). See also [method size].
func is_empty() -> bool:
	pass;

## Returns [code]true[/code] if the dictionary is read-only. See [method make_read_only]. Dictionaries are automatically read-only if declared with [code]const[/code] keyword.
func is_read_only() -> bool:
	pass;

## Returns the list of keys in the dictionary.
func keys() -> Array:
	pass;

## Makes the dictionary read-only, i.e. disables modification of the dictionary's contents. Does not apply to nested content, e.g. content of nested dictionaries.
func make_read_only() -> void:
	pass;

## Adds entries from [param dictionary] to this dictionary. By default, duplicate keys are not copied over, unless [param overwrite] is [code]true[/code].
func merge(dictionary: Dictionary, overwrite: bool = false) -> void:
	pass;

## Returns the number of entries in the dictionary. Empty dictionaries ([code]{ }[/code]) always return [code]0[/code]. See also [method is_empty].
func size() -> int:
	pass;

## Returns the list of values in this dictionary.
func values() -> Array:
	pass;


