extends Node
class_name AnimationTree

## A node to be used for advanced animation transitions in an [AnimationPlayer].
##
## A node to be used for advanced animation transitions in an [AnimationPlayer].
## [b]Note:[/b] When linked with an [AnimationPlayer], several properties and methods of the corresponding [AnimationPlayer] will not function as expected. Playback and transitions should be handled using only the [AnimationTree] and its constituent [AnimationNode](s). The [AnimationPlayer] node should be used solely for adding, deleting, and editing animations.
##
## @tutorial(Using AnimationTree): https://docs.godotengine.org/en/stable/tutorials/animation/animation_tree.html
## @tutorial(Third Person Shooter Demo): https://godotengine.org/asset-library/asset/678


## Notifies when an animation finished playing.
## [b]Note:[/b] This signal is not emitted if an animation is looping or aborted. Also be aware of the possibility of unseen playback by sync and xfade.
signal animation_finished(anim_name: StringName)
## Emitted when the [member anim_player] is changed.
signal animation_player_changed
## Notifies when an animation starts playing.
## [b]Note:[/b] This signal is not emitted if an animation is looping or playbacked from the middle. Also be aware of the possibility of unseen playback by sync and xfade.
signal animation_started(anim_name: StringName)
## The animations will progress during the physics frame (i.e. [method Node._physics_process]).
## The animations will progress during the idle frame (i.e. [method Node._process]).
## The animations will only progress manually (see [method advance]).

#enum AnimationProcessCallback
enum {
    ANIMATION_PROCESS_PHYSICS = 0,
    ANIMATION_PROCESS_IDLE = 1,
    ANIMATION_PROCESS_MANUAL = 2,
}
## If [code]true[/code], the [AnimationTree] will be processing.
var active: bool:
	get = is_active, set = set_active

## The path to the [Node] used to evaluate the AnimationNode [Expression] if one is not explicitly specified internally.
var advance_expression_base_node: NodePath:
	get = get_advance_expression_base_node, set = set_advance_expression_base_node

## The path to the [AnimationPlayer] used for animating.
var anim_player: NodePath:
	get = get_animation_player, set = set_animation_player

## The number of possible simultaneous sounds for each of the assigned AudioStreamPlayers.
## For example, if this value is [code]32[/code] and the animation has two audio tracks, the two [AudioStreamPlayer]s assigned can play simultaneously up to [code]32[/code] voices each.
var audio_max_polyphony: int:
	get = get_audio_max_polyphony, set = set_audio_max_polyphony

## The process mode of this [AnimationTree]. See [enum AnimationProcessCallback] for available modes.
var process_callback: int:
	get = get_process_callback, set = set_process_callback

## The path to the Animation track used for root motion. Paths must be valid scene-tree paths to a node, and must be specified starting from the parent node of the node that will reproduce the animation. To specify a track that controls properties or bones, append its name after the path, separated by [code]":"[/code]. For example, [code]"character/skeleton:ankle"[/code] or [code]"character/mesh:transform/local"[/code].
## If the track has type [constant Animation.TYPE_POSITION_3D], [constant Animation.TYPE_ROTATION_3D] or [constant Animation.TYPE_SCALE_3D] the transformation will be canceled visually, and the animation will appear to stay in place. See also [method get_root_motion_position], [method get_root_motion_rotation], [method get_root_motion_scale] and [RootMotionView].
var root_motion_track: NodePath:
	get = get_root_motion_track, set = set_root_motion_track

## The root animation node of this [AnimationTree]. See [AnimationNode].
var tree_root: AnimationNode:
	get = get_tree_root, set = set_tree_root



## A virtual function for processing after key getting during playback.
func _post_process_key_value(animation: Animation, track: int, value: Variant, object: Object, object_idx: int) -> Variant:
	pass;

## Manually advance the animations by the specified time (in seconds).
func advance(delta: float) -> void:
	pass;

## Retrieve the motion delta of position with the [member root_motion_track] as a [Vector3] that can be used elsewhere.
## If [member root_motion_track] is not a path to a track of type [constant Animation.TYPE_POSITION_3D], returns [code]Vector3(0, 0, 0)[/code].
## See also [member root_motion_track] and [RootMotionView].
## The most basic example is applying position to [CharacterBody3D]:
## [codeblocks]
## [gdscript]
## var current_rotation: Quaternion
## func _process(delta):
## if Input.is_action_just_pressed("animate"):
## current_rotation = get_quaternion()
## state_machine.travel("Animate")
## var velocity: Vector3 = current_rotation * animation_tree.get_root_motion_position() / delta
## set_velocity(velocity)
## move_and_slide()
## [/gdscript]
## [/codeblocks]
## By using this in combination with [method get_root_motion_position_accumulator], you can apply the root motion position more correctly to account for the rotation of the node.
## [codeblocks]
## [gdscript]
## func _process(delta):
## if Input.is_action_just_pressed("animate"):
## state_machine.travel("Animate")
## set_quaternion(get_quaternion() * animation_tree.get_root_motion_rotation())
## var velocity: Vector3 = (animation_tree.get_root_motion_rotation_accumulator().inverse() * get_quaternion()) * animation_tree.get_root_motion_position() / delta
## set_velocity(velocity)
## move_and_slide()
## [/gdscript]
## [/codeblocks]
func get_root_motion_position() -> Vector3:
	pass;

## Retrieve the blended value of the position tracks with the [member root_motion_track] as a [Vector3] that can be used elsewhere.
## This is useful in cases where you want to respect the initial key values of the animation.
## For example, if an animation with only one key [code]Vector3(0, 0, 0)[/code] is played in the previous frame and then an animation with only one key [code]Vector3(1, 0, 1)[/code] is played in the next frame, the difference can be calculated as follows:
## [codeblocks]
## [gdscript]
## var prev_root_motion_position_accumulator: Vector3
## func _process(delta):
## if Input.is_action_just_pressed("animate"):
## state_machine.travel("Animate")
## var current_root_motion_position_accumulator: Vector3 = animation_tree.get_root_motion_position_accumulator()
## var difference: Vector3 = current_root_motion_position_accumulator - prev_root_motion_position_accumulator
## prev_root_motion_position_accumulator = current_root_motion_position_accumulator
## transform.origin += difference
## [/gdscript]
## [/codeblocks]
## However, if the animation loops, an unintended discrete change may occur, so this is only useful for some simple use cases.
func get_root_motion_position_accumulator() -> Vector3:
	pass;

## Retrieve the motion delta of rotation with the [member root_motion_track] as a [Quaternion] that can be used elsewhere.
## If [member root_motion_track] is not a path to a track of type [constant Animation.TYPE_ROTATION_3D], returns [code]Quaternion(0, 0, 0, 1)[/code].
## See also [member root_motion_track] and [RootMotionView].
## The most basic example is applying rotation to [CharacterBody3D]:
## [codeblocks]
## [gdscript]
## func _process(delta):
## if Input.is_action_just_pressed("animate"):
## state_machine.travel("Animate")
## set_quaternion(get_quaternion() * animation_tree.get_root_motion_rotation())
## [/gdscript]
## [/codeblocks]
func get_root_motion_rotation() -> Quaternion:
	pass;

## Retrieve the blended value of the rotation tracks with the [member root_motion_track] as a [Quaternion] that can be used elsewhere.
## This is necessary to apply the root motion position correctly, taking rotation into account. See also [method get_root_motion_position].
## Also, this is useful in cases where you want to respect the initial key values of the animation.
## For example, if an animation with only one key [code]Quaternion(0, 0, 0, 1)[/code] is played in the previous frame and then an animation with only one key [code]Quaternion(0, 0.707, 0, 0.707)[/code] is played in the next frame, the difference can be calculated as follows:
## [codeblocks]
## [gdscript]
## var prev_root_motion_rotation_accumulator: Quaternion
## func _process(delta):
## if Input.is_action_just_pressed("animate"):
## state_machine.travel("Animate")
## var current_root_motion_rotation_accumulator: Quaternion = animation_tree.get_root_motion_Quaternion_accumulator()
## var difference: Quaternion = prev_root_motion_rotation_accumulator.inverse() * current_root_motion_rotation_accumulator
## prev_root_motion_rotation_accumulator = current_root_motion_rotation_accumulator
## transform.basis *= difference
## [/gdscript]
## [/codeblocks]
## However, if the animation loops, an unintended discrete change may occur, so this is only useful for some simple use cases.
func get_root_motion_rotation_accumulator() -> Quaternion:
	pass;

## Retrieve the motion delta of scale with the [member root_motion_track] as a [Vector3] that can be used elsewhere.
## If [member root_motion_track] is not a path to a track of type [constant Animation.TYPE_SCALE_3D], returns [code]Vector3(0, 0, 0)[/code].
## See also [member root_motion_track] and [RootMotionView].
## The most basic example is applying scale to [CharacterBody3D]:
## [codeblocks]
## [gdscript]
## var current_scale: Vector3 = Vector3(1, 1, 1)
## var scale_accum: Vector3 = Vector3(1, 1, 1)
## func _process(delta):
## if Input.is_action_just_pressed("animate"):
## current_scale = get_scale()
## scale_accum = Vector3(1, 1, 1)
## state_machine.travel("Animate")
## scale_accum += animation_tree.get_root_motion_scale()
## set_scale(current_scale * scale_accum)
## [/gdscript]
## [/codeblocks]
func get_root_motion_scale() -> Vector3:
	pass;

## Retrieve the blended value of the scale tracks with the [member root_motion_track] as a [Vector3] that can be used elsewhere.
## For example, if an animation with only one key [code]Vector3(1, 1, 1)[/code] is played in the previous frame and then an animation with only one key [code]Vector3(2, 2, 2)[/code] is played in the next frame, the difference can be calculated as follows:
## [codeblocks]
## [gdscript]
## var prev_root_motion_scale_accumulator: Vector3
## func _process(delta):
## if Input.is_action_just_pressed("animate"):
## state_machine.travel("Animate")
## var current_root_motion_scale_accumulator: Vector3 = animation_tree.get_root_motion_scale_accumulator()
## var difference: Vector3 = current_root_motion_scale_accumulator - prev_root_motion_scale_accumulator
## prev_root_motion_scale_accumulator = current_root_motion_scale_accumulator
## transform.basis = transform.basis.scaled(difference)
## [/gdscript]
## [/codeblocks]
## However, if the animation loops, an unintended discrete change may occur, so this is only useful for some simple use cases.
func get_root_motion_scale_accumulator() -> Vector3:
	pass;


func is_active() -> bool:
	return active

func set_active(value: bool) -> void:
	active = value

func get_advance_expression_base_node() -> NodePath:
	return advance_expression_base_node

func set_advance_expression_base_node(value: NodePath) -> void:
	advance_expression_base_node = value

func get_animation_player() -> NodePath:
	return anim_player

func set_animation_player(value: NodePath) -> void:
	anim_player = value

func get_audio_max_polyphony() -> int:
	return audio_max_polyphony

func set_audio_max_polyphony(value: int) -> void:
	audio_max_polyphony = value

func get_process_callback() -> int:
	return process_callback

func set_process_callback(value: int) -> void:
	process_callback = value

func get_root_motion_track() -> NodePath:
	return root_motion_track

func set_root_motion_track(value: NodePath) -> void:
	root_motion_track = value

func get_tree_root() -> AnimationNode:
	return tree_root

func set_tree_root(value: AnimationNode) -> void:
	tree_root = value

