extends RefCounted
class_name AStar3D

## An implementation of A* to find the shortest paths among connected points in space.
##
## A* (A star) is a computer algorithm that is widely used in pathfinding and graph traversal, the process of plotting short paths among vertices (points), passing through a given set of edges (segments). It enjoys widespread use due to its performance and accuracy. Godot's A* implementation uses points in three-dimensional space and Euclidean distances by default.
## You must add points manually with [method add_point] and create segments manually with [method connect_points]. Then you can test if there is a path between two points with the [method are_points_connected] function, get a path containing indices by [method get_id_path], or one containing actual coordinates with [method get_point_path].
## It is also possible to use non-Euclidean distances. To do so, create a class that extends [code]AStar3D[/code] and override methods [method _compute_cost] and [method _estimate_cost]. Both take two indices and return a length, as is shown in the following example.
## [codeblocks]
## [gdscript]
## class MyAStar:
## extends AStar3D
## func _compute_cost(u, v):
## return abs(u - v)
## func _estimate_cost(u, v):
## return min(0, abs(u - v) - 1)
## [/gdscript]
## [csharp]
## public partial class MyAStar : AStar3D
## {
## public override float _ComputeCost(long fromId, long toId)
## {
## return Mathf.Abs((int)(fromId - toId));
## }
## public override float _EstimateCost(long fromId, long toId)
## {
## return Mathf.Min(0, Mathf.Abs((int)(fromId - toId)) - 1);
## }
## }
## [/csharp]
## [/codeblocks]
## [method _estimate_cost] should return a lower bound of the distance, i.e. [code]_estimate_cost(u, v) <= _compute_cost(u, v)[/code]. This serves as a hint to the algorithm because the custom [code]_compute_cost[/code] might be computation-heavy. If this is not the case, make [method _estimate_cost] return the same value as [method _compute_cost] to provide the algorithm with the most accurate information.
## If the default [method _estimate_cost] and [method _compute_cost] methods are used, or if the supplied [method _estimate_cost] method returns a lower bound of the cost, then the paths returned by A* will be the lowest-cost paths. Here, the cost of a path equals the sum of the [method _compute_cost] results of all segments in the path multiplied by the [code]weight_scale[/code]s of the endpoints of the respective segments. If the default methods are used and the [code]weight_scale[/code]s of all points are set to [code]1.0[/code], then this equals the sum of Euclidean distances of all segments in the path.





## Called when computing the cost between two connected points.
## Note that this function is hidden in the default [code]AStar3D[/code] class.
func _compute_cost(from_id: int, to_id: int) -> float:
	pass;

## Called when estimating the cost between a point and the path's ending point.
## Note that this function is hidden in the default [code]AStar3D[/code] class.
func _estimate_cost(from_id: int, to_id: int) -> float:
	pass;

## Adds a new point at the given position with the given identifier. The [param id] must be 0 or larger, and the [param weight_scale] must be 0.0 or greater.
## The [param weight_scale] is multiplied by the result of [method _compute_cost] when determining the overall cost of traveling across a segment from a neighboring point to this point. Thus, all else being equal, the algorithm prefers points with lower [param weight_scale]s to form a path.
## [codeblocks]
## [gdscript]
## var astar = AStar3D.new()
## astar.add_point(1, Vector3(1, 0, 0), 4) # Adds the point (1, 0, 0) with weight_scale 4 and id 1
## [/gdscript]
## [csharp]
## var astar = new AStar3D();
## astar.AddPoint(1, new Vector3(1, 0, 0), 4); // Adds the point (1, 0, 0) with weight_scale 4 and id 1
## [/csharp]
## [/codeblocks]
## If there already exists a point for the given [param id], its position and weight scale are updated to the given values.
func add_point(id: int, position: Vector3, weight_scale: float = 1.0) -> void:
	pass;

## Returns whether the two given points are directly connected by a segment. If [param bidirectional] is [code]false[/code], returns whether movement from [param id] to [param to_id] is possible through this segment.
func are_points_connected(id: int, to_id: int, bidirectional: bool = true) -> bool:
	pass;

## Clears all the points and segments.
func clear() -> void:
	pass;

## Creates a segment between the given points. If [param bidirectional] is [code]false[/code], only movement from [param id] to [param to_id] is allowed, not the reverse direction.
## [codeblocks]
## [gdscript]
## var astar = AStar3D.new()
## astar.add_point(1, Vector3(1, 1, 0))
## astar.add_point(2, Vector3(0, 5, 0))
## astar.connect_points(1, 2, false)
## [/gdscript]
## [csharp]
## var astar = new AStar3D();
## astar.AddPoint(1, new Vector3(1, 1, 0));
## astar.AddPoint(2, new Vector3(0, 5, 0));
## astar.ConnectPoints(1, 2, false);
## [/csharp]
## [/codeblocks]
func connect_points(id: int, to_id: int, bidirectional: bool = true) -> void:
	pass;

## Deletes the segment between the given points. If [param bidirectional] is [code]false[/code], only movement from [param id] to [param to_id] is prevented, and a unidirectional segment possibly remains.
func disconnect_points(id: int, to_id: int, bidirectional: bool = true) -> void:
	pass;

## Returns the next available point ID with no point associated to it.
func get_available_point_id() -> int:
	pass;

## Returns the ID of the closest point to [param to_position], optionally taking disabled points into account. Returns [code]-1[/code] if there are no points in the points pool.
## [b]Note:[/b] If several points are the closest to [param to_position], the one with the smallest ID will be returned, ensuring a deterministic result.
func get_closest_point(to_position: Vector3, include_disabled: bool = false) -> int:
	pass;

## Returns the closest position to [param to_position] that resides inside a segment between two connected points.
## [codeblocks]
## [gdscript]
## var astar = AStar3D.new()
## astar.add_point(1, Vector3(0, 0, 0))
## astar.add_point(2, Vector3(0, 5, 0))
## astar.connect_points(1, 2)
## var res = astar.get_closest_position_in_segment(Vector3(3, 3, 0)) # Returns (0, 3, 0)
## [/gdscript]
## [csharp]
## var astar = new AStar3D();
## astar.AddPoint(1, new Vector3(0, 0, 0));
## astar.AddPoint(2, new Vector3(0, 5, 0));
## astar.ConnectPoints(1, 2);
## Vector3 res = astar.GetClosestPositionInSegment(new Vector3(3, 3, 0)); // Returns (0, 3, 0)
## [/csharp]
## [/codeblocks]
## The result is in the segment that goes from [code]y = 0[/code] to [code]y = 5[/code]. It's the closest position in the segment to the given point.
func get_closest_position_in_segment(to_position: Vector3) -> Vector3:
	pass;

## Returns an array with the IDs of the points that form the path found by AStar3D between the given points. The array is ordered from the starting point to the ending point of the path.
## [codeblocks]
## [gdscript]
## var astar = AStar3D.new()
## astar.add_point(1, Vector3(0, 0, 0))
## astar.add_point(2, Vector3(0, 1, 0), 1) # Default weight is 1
## astar.add_point(3, Vector3(1, 1, 0))
## astar.add_point(4, Vector3(2, 0, 0))
## astar.connect_points(1, 2, false)
## astar.connect_points(2, 3, false)
## astar.connect_points(4, 3, false)
## astar.connect_points(1, 4, false)
## var res = astar.get_id_path(1, 3) # Returns [1, 2, 3]
## [/gdscript]
## [csharp]
## var astar = new AStar3D();
## astar.AddPoint(1, new Vector3(0, 0, 0));
## astar.AddPoint(2, new Vector3(0, 1, 0), 1); // Default weight is 1
## astar.AddPoint(3, new Vector3(1, 1, 0));
## astar.AddPoint(4, new Vector3(2, 0, 0));
## astar.ConnectPoints(1, 2, false);
## astar.ConnectPoints(2, 3, false);
## astar.ConnectPoints(4, 3, false);
## astar.ConnectPoints(1, 4, false);
## int[] res = astar.GetIdPath(1, 3); // Returns [1, 2, 3]
## [/csharp]
## [/codeblocks]
## If you change the 2nd point's weight to 3, then the result will be [code][1, 4, 3][/code] instead, because now even though the distance is longer, it's "easier" to get through point 4 than through point 2.
func get_id_path(from_id: int, to_id: int) -> PackedInt64Array:
	pass;

## Returns the capacity of the structure backing the points, useful in conjunction with [code]reserve_space[/code].
func get_point_capacity() -> int:
	pass;

## Returns an array with the IDs of the points that form the connection with the given point.
## [codeblocks]
## [gdscript]
## var astar = AStar3D.new()
## astar.add_point(1, Vector3(0, 0, 0))
## astar.add_point(2, Vector3(0, 1, 0))
## astar.add_point(3, Vector3(1, 1, 0))
## astar.add_point(4, Vector3(2, 0, 0))
## astar.connect_points(1, 2, true)
## astar.connect_points(1, 3, true)
## var neighbors = astar.get_point_connections(1) # Returns [2, 3]
## [/gdscript]
## [csharp]
## var astar = new AStar3D();
## astar.AddPoint(1, new Vector3(0, 0, 0));
## astar.AddPoint(2, new Vector3(0, 1, 0));
## astar.AddPoint(3, new Vector3(1, 1, 0));
## astar.AddPoint(4, new Vector3(2, 0, 0));
## astar.ConnectPoints(1, 2, true);
## astar.ConnectPoints(1, 3, true);
## int[] neighbors = astar.GetPointConnections(1); // Returns [2, 3]
## [/csharp]
## [/codeblocks]
func get_point_connections(id: int) -> PackedInt64Array:
	pass;

## Returns the number of points currently in the points pool.
func get_point_count() -> int:
	pass;

## Returns an array of all point IDs.
func get_point_ids() -> PackedInt64Array:
	pass;

## Returns an array with the points that are in the path found by AStar3D between the given points. The array is ordered from the starting point to the ending point of the path.
## [b]Note:[/b] This method is not thread-safe. If called from a [Thread], it will return an empty [PackedVector3Array] and will print an error message.
func get_point_path(from_id: int, to_id: int) -> PackedVector3Array:
	pass;

## Returns the position of the point associated with the given [param id].
func get_point_position(id: int) -> Vector3:
	pass;

## Returns the weight scale of the point associated with the given [param id].
func get_point_weight_scale(id: int) -> float:
	pass;

## Returns whether a point associated with the given [param id] exists.
func has_point(id: int) -> bool:
	pass;

## Returns whether a point is disabled or not for pathfinding. By default, all points are enabled.
func is_point_disabled(id: int) -> bool:
	pass;

## Removes the point associated with the given [param id] from the points pool.
func remove_point(id: int) -> void:
	pass;

## Reserves space internally for [param num_nodes] points. Useful if you're adding a known large number of points at once, such as points on a grid. New capacity must be greater or equals to old capacity.
func reserve_space(num_nodes: int) -> void:
	pass;

## Disables or enables the specified point for pathfinding. Useful for making a temporary obstacle.
func set_point_disabled(id: int, disabled: bool = true) -> void:
	pass;

## Sets the [param position] for the point with the given [param id].
func set_point_position(id: int, position: Vector3) -> void:
	pass;

## Sets the [param weight_scale] for the point with the given [param id]. The [param weight_scale] is multiplied by the result of [method _compute_cost] when determining the overall cost of traveling across a segment from a neighboring point to this point.
func set_point_weight_scale(id: int, weight_scale: float) -> void:
	pass;


