extends Object
class_name UndoRedo

## Provides a high-level interface for implementing undo and redo operations.
##
## UndoRedo works by registering methods and property changes inside "actions". You can create an action, then provide ways to do and undo this action using function calls and property changes, then commit the action.
## When an action is committed, all of the [code]do_*[/code] methods will run. If the [method undo] method is used, the [code]undo_*[/code] methods will run. If the [method redo] method is used, once again, all of the [code]do_*[/code] methods will run.
## Here's an example on how to add an action:
## [codeblocks]
## [gdscript]
## var undo_redo = UndoRedo.new()
## func do_something():
## pass # Put your code here.
## func undo_something():
## pass # Put here the code that reverts what's done by "do_something()".
## func _on_my_button_pressed():
## var node = get_node("MyNode2D")
## undo_redo.create_action("Move the node")
## undo_redo.add_do_method(do_something)
## undo_redo.add_undo_method(undo_something)
## undo_redo.add_do_property(node, "position", Vector2(100,100))
## undo_redo.add_undo_property(node, "position", node.position)
## undo_redo.commit_action()
## [/gdscript]
## [csharp]
## private UndoRedo _undoRedo;
## public override void _Ready()
## {
## _undoRedo = new UndoRedo();
## }
## public void DoSomething()
## {
## // Put your code here.
## }
## public void UndoSomething()
## {
## // Put here the code that reverts what's done by "DoSomething()".
## }
## private void OnMyButtonPressed()
## {
## var node = GetNode<Node2D>("MyNode2D");
## _undoRedo.CreateAction("Move the node");
## _undoRedo.AddDoMethod(new Callable(this, MethodName.DoSomething));
## _undoRedo.AddUndoMethod(new Callable(this, MethodName.UndoSomething));
## _undoRedo.AddDoProperty(node, "position", new Vector2(100, 100));
## _undoRedo.AddUndoProperty(node, "position", node.Position);
## _undoRedo.CommitAction();
## }
## [/csharp]
## [/codeblocks]
## [method create_action], [method add_do_method], [method add_undo_method], [method add_do_property], [method add_undo_property], and [method commit_action] should be called one after the other, like in the example. Not doing so could lead to crashes.
## If you don't need to register a method, you can leave [method add_do_method] and [method add_undo_method] out; the same goes for properties. You can also register more than one method/property in the order they should run.
## If you are making an [EditorPlugin] and want to integrate into the editor's undo history, use [EditorUndoRedoManager] instead.


## Called when [method undo] or [method redo] was called.
signal version_changed
## Makes "do"/"undo" operations stay in separate actions.
## Makes so that the action's "undo" operations are from the first action created and the "do" operations are from the last subsequent action with the same name.
## Makes subsequent actions with the same name be merged into one.

#enum MergeMode
enum {
    MERGE_DISABLE = 0,
    MERGE_ENDS = 1,
    MERGE_ALL = 2,
}


## Register a [Callable] that will be called when the action is committed.
func add_do_method(callable: Callable) -> void:
	pass;

## Register a [param property] that would change its value to [param value] when the action is committed.
func add_do_property(object: Object, property: StringName, value: Variant) -> void:
	pass;

## Register a reference for "do" that will be erased if the "do" history is lost. This is useful mostly for new nodes created for the "do" call. Do not use for resources.
## [codeblock]
## var node = Node2D.new()
## undo_redo.create_action("Add node")
## undo_redo.add_do_method(add_child.bind(node))
## undo_redo.add_do_reference(node)
## undo_redo.add_undo_method(remove_child.bind(node))
## undo_redo.commit_action()
## [/codeblock]
func add_do_reference(object: Object) -> void:
	pass;

## Register a [Callable] that will be called when the action is undone.
func add_undo_method(callable: Callable) -> void:
	pass;

## Register a [param property] that would change its value to [param value] when the action is undone.
func add_undo_property(object: Object, property: StringName, value: Variant) -> void:
	pass;

## Register a reference for "undo" that will be erased if the "undo" history is lost. This is useful mostly for nodes removed with the "do" call (not the "undo" call!).
## [codeblock]
## var node = $Node2D
## undo_redo.create_action("Remove node")
## undo_redo.add_do_method(remove_child.bind(node))
## undo_redo.add_undo_method(add_child.bind(node))
## undo_redo.add_undo_reference(node)
## undo_redo.commit_action()
## [/codeblock]
func add_undo_reference(object: Object) -> void:
	pass;

## Clear the undo/redo history and associated references.
## Passing [code]false[/code] to [param increase_version] will prevent the version number from increasing when the history is cleared.
func clear_history(increase_version: bool = true) -> void:
	pass;

## Commit the action. If [param execute] is [code]true[/code] (which it is by default), all "do" methods/properties are called/set when this function is called.
func commit_action(execute: bool = true) -> void:
	pass;

## Create a new action. After this is called, do all your calls to [method add_do_method], [method add_undo_method], [method add_do_property], and [method add_undo_property], then commit the action with [method commit_action].
## The way actions are merged is dictated by [param merge_mode]. See [enum MergeMode] for details.
func create_action(name: String, merge_mode: int = 0) -> void:
	pass;

## Stops marking operations as to be processed even if the action gets merged with another in the [constant MERGE_ENDS] mode. See [method start_force_keep_in_merge_ends].
func end_force_keep_in_merge_ends() -> void:
	pass;

## Gets the action name from its index.
func get_action_name(id: int) -> String:
	pass;

## Gets the index of the current action.
func get_current_action() -> int:
	pass;

## Gets the name of the current action, equivalent to [code]get_action_name(get_current_action())[/code].
func get_current_action_name() -> String:
	pass;

## Returns how many elements are in the history.
func get_history_count() -> int:
	pass;

## Gets the version. Every time a new action is committed, the [UndoRedo]'s version number is increased automatically.
## This is useful mostly to check if something changed from a saved version.
func get_version() -> int:
	pass;

## Returns [code]true[/code] if a "redo" action is available.
func has_redo() -> bool:
	pass;

## Returns [code]true[/code] if an "undo" action is available.
func has_undo() -> bool:
	pass;

## Returns [code]true[/code] if the [UndoRedo] is currently committing the action, i.e. running its "do" method or property change (see [method commit_action]).
func is_committing_action() -> bool:
	pass;

## Redo the last action.
func redo() -> bool:
	pass;

## Marks the next "do" and "undo" operations to be processed even if the action gets merged with another in the [constant MERGE_ENDS] mode. Return to normal operation using [method end_force_keep_in_merge_ends].
func start_force_keep_in_merge_ends() -> void:
	pass;

## Undo the last action.
func undo() -> bool:
	pass;


