extends Resource
class_name PackedScene

## An abstraction of a serialized scene.
##
## A simplified interface to a scene file. Provides access to operations and checks that can be performed on the scene resource itself.
## Can be used to save a node to a file. When saving, the node as well as all the nodes it owns get saved (see [member Node.owner] property).
## [b]Note:[/b] The node doesn't need to own itself.
## [b]Example of loading a saved scene:[/b]
## [codeblocks]
## [gdscript]
## # Use load() instead of preload() if the path isn't known at compile-time.
## var scene = preload("res://scene.tscn").instantiate()
## # Add the node as a child of the node the script is attached to.
## add_child(scene)
## [/gdscript]
## [csharp]
## // C# has no preload, so you have to always use ResourceLoader.Load<PackedScene>().
## var scene = ResourceLoader.Load<PackedScene>("res://scene.tscn").Instantiate();
## // Add the node as a child of the node the script is attached to.
## AddChild(scene);
## [/csharp]
## [/codeblocks]
## [b]Example of saving a node with different owners:[/b] The following example creates 3 objects: [Node2D] ([code]node[/code]), [RigidBody2D] ([code]body[/code]) and [CollisionObject2D] ([code]collision[/code]). [code]collision[/code] is a child of [code]body[/code] which is a child of [code]node[/code]. Only [code]body[/code] is owned by [code]node[/code] and [code]pack[/code] will therefore only save those two nodes, but not [code]collision[/code].
## [codeblocks]
## [gdscript]
## # Create the objects.
## var node = Node2D.new()
## var body = RigidBody2D.new()
## var collision = CollisionShape2D.new()
## # Create the object hierarchy.
## body.add_child(collision)
## node.add_child(body)
## # Change owner of `body`, but not of `collision`.
## body.owner = node
## var scene = PackedScene.new()
## # Only `node` and `body` are now packed.
## var result = scene.pack(node)
## if result == OK:
## var error = ResourceSaver.save(scene, "res://path/name.tscn")  # Or "user://..."
## if error != OK:
## push_error("An error occurred while saving the scene to disk.")
## [/gdscript]
## [csharp]
## // Create the objects.
## var node = new Node2D();
## var body = new RigidBody2D();
## var collision = new CollisionShape2D();
## // Create the object hierarchy.
## body.AddChild(collision);
## node.AddChild(body);
## // Change owner of `body`, but not of `collision`.
## body.Owner = node;
## var scene = new PackedScene();
## // Only `node` and `body` are now packed.
## Error result = scene.Pack(node);
## if (result == Error.Ok)
## {
## Error error = ResourceSaver.Save(scene, "res://path/name.tscn"); // Or "user://..."
## if (error != Error.Ok)
## {
## GD.PushError("An error occurred while saving the scene to disk.");
## }
## }
## [/csharp]
## [/codeblocks]
##
## @tutorial(2D Role Playing Game Demo): https://godotengine.org/asset-library/asset/520


## If passed to [method instantiate], blocks edits to the scene state.
## If passed to [method instantiate], provides local scene resources to the local scene.
## [b]Note:[/b] Only available in editor builds.
## If passed to [method instantiate], provides local scene resources to the local scene. Only the main scene should receive the main edit state.
## [b]Note:[/b] Only available in editor builds.
## It's similar to [constant GEN_EDIT_STATE_MAIN], but for the case where the scene is being instantiated to be the base of another one.
## [b]Note:[/b] Only available in editor builds.

#enum GenEditState
enum {
    GEN_EDIT_STATE_DISABLED = 0,
    GEN_EDIT_STATE_INSTANCE = 1,
    GEN_EDIT_STATE_MAIN = 2,
    GEN_EDIT_STATE_MAIN_INHERITED = 3,
}
## A dictionary representation of the scene contents.
## Available keys include "rnames" and "variants" for resources, "node_count", "nodes", "node_paths" for nodes, "editable_instances" for base scene children overrides, "conn_count" and "conns" for signal connections, and "version" for the format style of the PackedScene.
var _bundled: Dictionary:
	get = _get_bundled_scene, set = _set_bundled_scene



## Returns [code]true[/code] if the scene file has nodes.
func can_instantiate() -> bool:
	pass;

## Returns the [code]SceneState[/code] representing the scene file contents.
func get_state() -> SceneState:
	pass;

## Instantiates the scene's node hierarchy. Triggers child scene instantiation(s). Triggers a [constant Node.NOTIFICATION_SCENE_INSTANTIATED] notification on the root node.
func instantiate(edit_state: int = 0) -> Node:
	pass;

## Pack will ignore any sub-nodes not owned by given node. See [member Node.owner].
func pack(path: Node) -> int:
	pass;


func _get_bundled_scene() -> Dictionary:
	return _bundled

func _set_bundled_scene(value: Dictionary) -> void:
	_bundled = value

