$ErrorActionPreference = "Stop"
$ProgressPreference = "SilentlyContinue"
Set-StrictMode -Version 3.0

# from https://ss64.com/ps/syntax-consolesize.html
# or powershell will break line at console width in interactive mode
$host.UI.RawUI.BufferSize = New-Object System.Management.Automation.Host.size(5000,500)
$host.UI.RawUI.WindowSize = New-Object System.Management.Automation.Host.size(5000,500)

enum ExitCode {
  EXIT_SUCCESS = 0
  EXIT_UNHANDLED_CLI_COMMAND = 54
  EXIT_NO_DISK_SPACE = 82
  EXIT_DOWNLOAD_ERROR = 83
  EXIT_LAUNCH_ERROR = 84
  EXIT_SELF_CHECK_ERROR = 85
  EXIT_JAVA_MISSING_ERROR = 86
  EXIT_UNSUPPORTED_OS = 87
  EXIT_UNTAR_ERROR = 88
  EXIT_TAR_MISSING = 89
}

function Die {
  param(
    [Parameter(Position=0, Mandatory=$true)]
    [ExitCode]$exitCode,

    [Parameter(Position=1, Mandatory=$true)]
    [string]$errorMessage
  )

  Write-Host "~ERROR: $exitCode $errorMessage"

  $intValue = [int]$exitCode
  Write-Host "~EXIT-CODE: $intValue"

  [Environment]::Exit($intValue)
}

function DownloadFile() {
  param (
    [String] $FileUrl,
    [String] $DestinationPath,
    [String] $FileSha256
  )

  if ($FileUrl.StartsWith("http://") -or $FileUrl.StartsWith("https://")) {
    $randomSuffix = [System.IO.Path]::GetRandomFileName()
    $tmpFile = "$DestinationPath-$randomSuffix"

    $Web_client = New-Object System.Net.WebClient
    $Web_client.DownloadFile($FileUrl, $tmpFile)

    if (-not (CheckSha256 $tmpFile $FileSha256 $FileUrl)) {
      Die EXIT_DOWNLOAD_ERROR "Checksum check failed for $tmpFile ($FileUrl), could not continue"
    }

    Move-Item -Path $tmpFile -Destination $DestinationPath -Force
  } elseif ($url.StartsWith("inline://")) {
    # ?
    Die EXIT_DOWNLOAD_ERROR "Unsupported url scheme: inline"
  } else {
    Die EXIT_DOWNLOAD_ERROR "Unsupported url scheme: $FileUrl"
  }
}

function Untar($inputArchive, $outputDir) {
  if (Get-Command -Name "tar" -ErrorAction SilentlyContinue) {
    New-Item -ItemType Directory -Path $outputDir -Force
    tar -C $outputDir --strip-components 1 -xzf $inputArchive
    if ($LastExitCode -ne 0) {
      Die EXIT_UNTAR_ERROR "Failed to extract file $inputArchive, could not continue"
    }
    New-Item "$outputDir\.extracted" -type file
  } else {
    Die EXIT_TAR_MISSING "'tar' command is required, could not continue"
  }
}

function CheckSha256($path, $expectedSha256, $fileMoniker) {
  $actualSha256 = (Get-FileHash -Algorithm SHA256 -Path $path).Hash.ToString()
  if ($actualSha256 -ne $expectedSha256) {
    Write-Host "~WARN: Checksum mismatch for $path (downloaded from $fileMoniker): expected checksum $expectedSha256 but got $actualSha256"
  }
  return $actualSha256 -eq $expectedSha256
}

function DownloadFileInline {
  param(
    [string]$Url,
    [string]$DestinationFile,
    [string]$SHA256
  )

  $randomSuffix = [System.IO.Path]::GetRandomFileName()
  $tempFile = "$DestinationFile-$randomSuffix"

  Write-Host "~STATE: download file $Url:::$tempFile"

  while ($true) {
    $line = [Console]::ReadLine()
    if ($line -eq "~END DOWNLOAD FILE $Url") {
      break
    }
  }

  if (-not ([System.IO.File]::Exists($tempFile))) {
    throw "File was not found at $tempFile after uploading"
  }

  if (-not(CheckSha256 $tempFile $SHA256 $Url)) {
    Die EXIT_DOWNLOAD_ERROR "Checksum check failed for $tempFile ($Url), could not continue"
  }

  Move-Item -Path $tempFile -Destination $DestinationFile -Force
}

function DownloadFileWithFallback() {
  param(
    [string]$Url,
    [string]$DestinationFile,
    [string]$SHA256
  )

  if ($Url.StartsWith("inline://")) {
    DownloadFileInline $Url $DestinationFile $SHA256
  } else {
    try {
      DownloadFile $Url $DestinationFile $SHA256
    } catch {
      Write-Warning $Error[0]

      Write-Host "Downloading of $Url failed, falling back to inline download"
      DownloadFileInline $Url $DestinationFile $SHA256
    }
  }
}

function CheckDiskSpace() {
  param(
    [string]$Path,
    [int]$MinFreeMB = 200 # Default minimum free space required
  )

  Write-Host "~DEBUG: Checking free disk space at $Path"

  # Errors in checking for disk space should not prevent deploy script from working

  $errorMessage = ""

  try {
    $driveLetter = [System.IO.Path]::GetPathRoot($path).TrimEnd('\\')
    Write-Host "~DEBUG: Drive letter of '$Path' is $driveLetter"

    $driveInfo = New-Object System.IO.DriveInfo($driveLetter)
    if ($driveInfo.IsReady) {
      Write-Host "Drive $($driveInfo.Name)"
      Write-Host "  Drive type: $($driveInfo.DriveType)"

      $freeMB = [math]::Round($driveInfo.AvailableFreeSpace / 1MB)
      Write-Host "  Available space to current user: $($driveInfo.AvailableFreeSpace) bytes or $freeMB MB"

      if ($freeMB -lt $MinFreeMB) {
        $errorMessage = "Free disk space on $($driveInfo.Name) is less than $MinFreeMB MB: only $freeMB MB is free"
        Write-Host "~ERROR: $errorMessage"
      }
    } else {
      Write-Host "~WARN: The drive is not ready or the path is invalid at $driveLetter"
    }
  } catch {
    Write-Warning "Error while checking for free disk space, see the exception below"
    Write-Warning $Error[0]
  }

  if ($errorMessage) {
    throw $errorMessage
  }
}

# This variables could be overridden in env.ps1
$cache_dir = ""
$cli_dir_path = ""
$java_home = ""
$async_profiler_path = ""
$async_profiler_snapshots_dir = ""

$arch = ($Env:PROCESSOR_ARCHITECTURE).ToUpper()
$cli_url = Get-Variable -Name "cli_url_Windows_$arch" -ValueOnly
$cli_sha256 = Get-Variable -Name "cli_sha256_Windows_$arch" -ValueOnly
$cli_version = Get-Variable -Name "cli_version_Windows_$arch" -ValueOnly

$commonApplicationDataSpecialFolder = [Environment+SpecialFolder]::CommonApplicationData
$programDataPath = [Environment]::GetFolderPath($commonApplicationDataSpecialFolder)
$envOverrideFile = [System.IO.Path]::Combine($programDataPath, "JetBrains", "ToolboxSshDeploy", "env.ps1")

if ([System.IO.File]::Exists($envOverrideFile)) {
  Write-Host "~DEBUG: sourcing env override file at $envOverrideFile"
  $scriptContent = Get-Content $envOverrideFile -Raw
  Invoke-Expression $scriptContent
} else {
  Write-Host "~DEBUG: env override file is missing at $envOverrideFile, not sourcing"
}

if (-not [string]::IsNullOrWhiteSpace($cache_dir)) {
  Write-Host "~DEBUG: cache_dir was overridden: $cache_dir"
} else {
  $localAppDataSpecialFolder = [Environment+SpecialFolder]::LocalApplicationData
  $localAppData = [Environment]::GetFolderPath($localAppDataSpecialFolder)
  $cache_dir = [System.IO.Path]::Combine($localAppData, "JetBrains", "Toolbox-CLI-dist")

  Write-Host "~DEBUG: using default cache_dir: $cache_dir"
}

New-Item -ItemType Directory -Path $cache_dir -Force

if (-not [string]::IsNullOrWhiteSpace($cli_dir_path)) {
  Write-Host "~DEBUG: cli dir path is already set to $cli_dir_path"
} else {
  $cli_dir_path = [System.IO.Path]::Combine($cache_dir, "tbcli-${cli_version}")
  $cli_archive_path="$cli_dir_path.tar.gz"
  Write-Host "~DEBUG: Checking for already downloaded CLI at $cli_dir_path"
  if ((Test-Path -Path $cli_dir_path -PathType Container) -and (Test-Path -Path "$cli_dir_path\.extracted" -PathType Leaf)) {
    Write-Host "~DEBUG: Using already downloaded Toolbox Agent: $cli_path"
  } else {
    if (Test-Path -Path $cli_dir_path -PathType Container) {
      Write-Host "~DEBUG: cli directory $cli_dir_path exists but is corrupted, removing"
      Remove-Item $cli_dir_path -Recurse -Force
    }
    CheckDiskSpace $cache_dir
    DownloadFileWithFallback $cli_url $cli_archive_path $cli_sha256
    Untar $cli_archive_path $cli_dir_path
    Remove-Item $cli_archive_path -Force
  }
}

$cli_path="$cli_dir_path\bin\tbcli.bat"

if (-not [string]::IsNullOrWhiteSpace($java_path)) {
  Write-Host "~DEBUG: java path is already set to $java_path"
} else {
  Write-Host "~DEBUG: Current arch: $arch"

  $jbr_name = Get-Variable -Name "jbr_name_Windows_$arch" -ValueOnly
  $jbr_url = Get-Variable -Name "jbr_url_Windows_$arch" -ValueOnly
  $jbr_sha256 = Get-Variable -Name "jbr_sha256_Windows_$arch" -ValueOnly

  $extract_code_version=1 # increment it if you want to invalidate all existing jbr downloads (e.g. download or extract code was changed in incompatible way)
  $java_path="$cache_dir\$jbr_name-$extract_code_version"
  $java_bin_path = "$java_path\bin\java.exe"
  $flag_path = "$java_path\.extracted"

  if (([System.IO.File]::Exists($java_bin_path)) -and ([System.IO.File]::Exists($flag_path))) {
    echo "~DEBUG: Using already downloaded JVM at $java_path"
  } else {
    echo "~DEBUG: Downloading JVM from $jbr_url"

    CheckDiskSpace $cache_dir

    $randomSuffix = [System.IO.Path]::GetRandomFileName()

    # TODO remove $archive_path on exit (any exit)
    $archive_path="$cache_dir\$jbr_name-$extract_code_version.tmp.$randomSuffix"

    DownloadFileWithFallback $jbr_url $archive_path $jbr_sha256

    if (Test-Path $java_path) {
      Remove-Item -Recurse -Force $java_path
    }
    New-Item -ItemType Directory $java_path
    & tar -x -C "$java_path" --strip-components 1 -z -f "$archive_path"
    if ($LastExitCode -ne 0) {
      throw "Unable to extract archive at ${archive_path}: $LastExitCode"
    }
    Remove-Item -Recurse -Force $archive_path
    if ([System.IO.File]::Exists($java_bin_path)) {
      Write-Host "~DEBUG: found java executable at $java_path"
    } else {
      Get-ChildItem -Recurse $java_path
      Die EXIT_JAVA_MISSING_ERROR "Unable to find java executable under $java_path, see available files listing in debug output"
    }

    New-Item $flag_path -type file
  }
}

$run_script_env_vars = "set TB_JAVA_HOME=$java_path`nset TB_CLI_PATH=$cli_path`nset TB_CLI_HASH=$cli_sha256`n"

if (-not ([string]::IsNullOrEmpty("$async_profiler_path"))) {
  $run_script_env_vars = ($run_script_env_vars + "set TB_ASYNC_PROFILER_PATH=$async_profiler_path`n")
}

if (-not ([string]::IsNullOrEmpty("$async_profiler_snapshots_dir"))) {
  $run_script_env_vars = ($run_script_env_vars + "set TB_ASYNC_PROFILER_SNAPSHOTS_DIR=$async_profiler_snapshots_dir`n")
}

# We need to distinguish debug port for proxy and debug port for agent

if ((Get-Variable -Name "cli_debug_port" -ErrorAction SilentlyContinue) -and (-not ([string]::IsNullOrEmpty("$cli_debug_port")))) {
  $run_script_env_vars = ($run_script_env_vars + "set TB_DEBUG_PORT=$cli_debug_port`n")
}

if ((Get-Variable -Name "cli_debug_suspend" -ErrorAction SilentlyContinue) -and (-not ([string]::IsNullOrEmpty("$cli_debug_suspend")))) {
  $run_script_env_vars = ($run_script_env_vars + "set TB_DEBUG_SUSPEND=$cli_debug_suspend`n")
}

if ((Get-Variable -Name "toolbox_log_level_option" -ErrorAction SilentlyContinue) -and (-not ([string]::IsNullOrEmpty("$toolbox_log_level_option")))) {
  $run_script_env_vars = ($run_script_env_vars + "set TB_AGENT_LOG_LEVEL=$toolbox_log_level_option`n")
}

# We need to export information about async profiler location to toolbox_cli runs

Write-Host "Running self-check with command: $cli_path --structured-logging self-check"

$env:TB_JAVA_HOME = $java_path
& $cli_path --structured-logging self-check

if ($LastExitCode -ne 0) {
  Die EXIT_SELF_CHECK_ERROR "Self check failed"
}

$run_script_file = "$Env:TEMP\$run_script_name"


if ((Get-Variable -Name "toolbox_log_level_option" -ErrorAction SilentlyContinue) -and (-not ([string]::IsNullOrEmpty("$toolbox_log_level_option")))) {
  $run_script_content = "$run_script_env_vars `"$cli_path`" --$toolbox_log_level_option --structured-logging --detach agent"
} else {
  $run_script_content = "$run_script_env_vars `"$cli_path`" --debug --structured-logging --detach agent"
}

Write-Host "~DEBUG: Launching $run_script_content via run script at $run_script_file"

Set-Content -Path $run_script_file -Value $run_script_content

Write-Host "~END DEPLOY SCRIPT"

# The & and | Out-Host is the result of long experimenting around Windows deployment to achieve undelayed and reliable output
# from the agent command. There's a chance it's not really required but after 3 days I gave up.
& cmd.exe /c $run_script_file | Out-Host

if ($LastExitCode -ne 0) {
  Write-Host "~EXIT-CODE: ${LastExitCode}"

  [Environment]::Exit($LastExitCode)
}
